;;
;;
;;      Title : MakeSpRules
;;
;;      Function : Compiles a set of Koskenniemi Rules into their equivalent
;;                 automata
;;
;;      Author : Alan W Black September 1985
;;               Dept of AI, University of Edinburgh
;;     
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description :
;;         This takes a set of rules as defined by Kimmo at the workshop
;;         on morphology CSLI July 1985.  This is basically a reimplementation
;;         of his rule compiler as he wrote for the Xerox Dandelion.
;;         This is implemented as part of the Alvey project at Cambridge
;;         and Edinburgh on a Dictionary and Morphological Analyser
;;         for English Language Processing Systems
;;
;;
;;         The compilation produces a specialised form of automata
;;         
;;      1.18 modified to do the parsing of the spelling rules in lisp
;;           rather than in yacc and lex.  Output to the *.sp.ma file
;;           is now a number of s-expressions rather than setq's This allow
;;           easier porting to system that have no case distinction.
;;      2.1  30th July 1986
;;           Removed closures and replaced with simpler system of actually
;;           saving and restoring globals.  The compilation method has
;;           changed from the form Kimmo suggested to a more simpler form
;;           This does make the interpretation more complex but it runs
;;           faster.  The interpretation is similar to John Bear's system
;;           but rules are still Koskenniemi form.
;;      2.2  9th October 1986
;;           Changed references to keyword to be macros declared in the
;;           file keywords.  This should make it easier for people who
;;           porting the code to systems that have problems with case
;;           distinction.
;;      2.3  25th October 1986
;;           Made compilation save an alist of what each surface character
;;           can correspond to on the lexical tape.
;;      2.4  1st February 1987
;;           Added markers for staes to say which rules they came from
;;      2.4.F March 1987
;;           Joined the two spelling rule forms into one compiler.
;;           It detects which form the spelling rule file is and
;;           then simply calles the approriate function
;;      2.6 April 1987
;;           Removed the alternate spelling rule notation
;;           Added D-AUTOSYMBOLS to save me doing a concat during
;;           analysis.
;;
;;      Parameters :
;;         name   : symbols, where <name>.sp is the name of the
;;                  file containing the spelling rules
;;      Returns :
;;      Side Effect :
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;        the automata built can't distinguish epsilon transitions
;;        and complex symbols in the alphabet
;;

(declare
   (special
      D-SPELLINGRULES
      D-SURFACEALPHABET
      D-LEXICALALPHABET
      D-SURFACESETS
      D-LEXICALSETS
      D-AUFILEID
      D-DEFAULTPAIRS
      D-CURRENTRULENAME
      D-AUTOSYMBOLS
   )
   (localf
      D-InitSpGlobals
      D-FindUnusualFeasibles 
      D-FindSLSets
      D-FindAutoSymbols
      D-PrintTrans                    
      D-WriteLambdaDef
      D-WriteTrans
      D-ExpandWClauses                
      D-ExpandWhereClause             
      D-SubstituteSymbol              
      D-FindFeasibles                 
      D-FindConcretePairs             
      D-FindCPIndividualRule          
      D-FindCPContext                 
      D-IsConcretePair                
      D-MakeAutomata                  
      D-FindRestrictedPairs
      D-MakeRulePairs
      D-MakeAutoFromRules             
      D-MakeAutomaton                 
      D-MakeAutomatonCR               
      D-MakeAutomatonSC               
      D-MakeAutomatonSCCR             
      D-MakeAutoContext               
      D-MkAutoSEQ                     
      D-MkAutoOR                      
      D-MkAutoOPT                     
      D-MkAutoKS                      
      D-MkAutoKP                      
      D-MkAutoContextPair             
      D-MakeTransitions               
      D-AddNewTransition              
      D-CheckAndAddOrigins
      D-ExpandPair                    
      D-ComPair                       
      D-ExpandSet                     
      D-NewStateName                  
   )
)

(include "macros")
(include "keywords")    ;; macros returning constants of key words
(include "subrout")
(include "kkruletype")  ;; macros accessing the parts of a kk rule
(include "catrouts")
(include "parserouts") 
(include "parsesp")    ;; parsing of the basic spelling rules

(defun D-MakeSpRules (name)
;;
;;  This is loads and compiles a set of koskenniemi spelling rules
;;  It does effect any currently loaded set
;;
   (let  (expandedrules cps transitions unusualfeasibles
		surfacetolexicalsets autosymbols)
      (D-SaveGlobalEnvironment)
      (D-InitSpGlobals)
      (setq D-fileid (infile (concat name (DK-SP))))
      (setq D-INCLUDEFILES (ncons (concat name (DK-SP))))
      (D-GetNextSym)

      (D-ParseRules)     ;; reads in the spelling rules

      (close D-fileid)
      (setq expandedrules (D-ExpandWClauses D-SPELLINGRULES))
      (setq cps (D-FindFeasibles expandedrules))
      (setq transitions
	 (D-MakeAutomata
	    expandedrules
	    cps               ;; feasibles (CPS)
	 )
      )
      (setq unusualfeasibles
	 (D-FindUnusualFeasibles cps))
      (setq surfacetolexicalsets 
	  (D-FindSLSets D-SURFACEALPHABET cps)
      )
      (setq autosymbols (D-FindAutoSymbols surfacetolexicalsets))

      (setq D-AUFILEID (outfile (concat name (DK-SP-MA))))
      (print (DK-VERSION-STAMP) D-AUFILEID) (terpri D-AUFILEID)
      (print D-LEXICALALPHABET D-AUFILEID) (terpri D-AUFILEID)
      (print D-SURFACEALPHABET D-AUFILEID) (terpri D-AUFILEID)
      (D-PrintTrans transitions)                ;; print transition list
      (print unusualfeasibles D-AUFILEID) (terpri D-AUFILEID)
      (print (cdr surfacetolexicalsets) D-AUFILEID) 
      (terpri D-AUFILEID)
      (print (cdar surfacetolexicalsets) D-AUFILEID)  ;; null possibles
      (terpri D-AUFILEID)
      (print autosymbols D-AUFILEID) (terpri D-AUFILEID)
      (terpri D-AUFILEID)
      (close D-AUFILEID)
      
      (D-RestoreGlobalEnvironment)
      t
   )
)

(defun D-InitSpGlobals ()
;;
;;   init the global variables
;;
   (D-InitReader)    ;; initialise the reader
   (setq D-LASTSYMBOL nil)
   (setq D-INCLUDESTACK nil)
   (setq D-CURRENTRULENAME nil)
   (mapcar 
      #'(lambda (glob) (set glob nil))
      '( D-SPELLINGRULES D-SURFACEALPHABET D-LEXICALALPHABET
         D-SURFACESETS D-LEXICALSETS D-AUFILEID D-DEFAULTPAIRS )
   )
)

(defun D-FindUnusualFeasibles (feaslist)
;;
;;   returns a list of all feasibles that do not have eq surface
;;   and lexical characters.
;;
   (mapcan
      #'(lambda (pair)
	 (cond
	    ((eq (car pair) (cadr pair)) nil)
	    (t
	       (ncons (concatl (cons 'D pair)))
            )
         ))
      feaslist
   )
)

(defun D-FindSLSets (surfacealphabet cps)
;;
;;  constructs an alist with pairs of surface character to possible
;;  lexical characters.  This can be used in analysis
;;
   ;(nconc
   (mapcar
      #'(lambda (surfch)
	 (cons surfch
	    (sort
	       (mapcan    ;; all lexical characters of feasible pairs that
		  #'(lambda (pair)   ;; have surfch on the surface
		     (cond
			((eq (cadr pair) surfch) (ncons (car pair)))
			(t nil)))
		  cps) #'D-OrderChars)))
      (cons (DK-NULL) surfacealphabet))
   ;(ncons (list (DK-ENDMARKER) 
		;(DK-ENDMARKER))))   ;; this is needed to continue to the end
)

(defun D-FindAutoSymbols (surfacetolexsets)
;;
;; returns an assq structure that allows faster creation of symbols
;; from a surface and lexical character.  This saves a concat in analysis
;;
   (mapcar
      #'(lambda (slist)
	 (cons (car slist)
	    (mapcar
	       #'(lambda (lex)
		     (cons lex (concat 'D lex (car slist))))
               (cdr slist))))
       surfacetolexsets
   )
)

;(defun D-PrintTrans (transitions)
;;;
;;;  This prints out transitions is a reasonably pretty way
;;;  without using the list pretty printer which is very slow
;;;  and space consuming;
;;;  This also reverses the order of the transitions to put the
;;;  default state to the front of the list, this should speed up
;;;  the analysis.
;;;
   ;;(pp (P D-AUFILEID) transitions)
   ;(terpri D-AUFILEID)
   ;(princ "(" D-AUFILEID)
   ;(terpri D-AUFILEID)
   ;(mapcar
      ;#'(lambda (x)
	    ;(princ x D-AUFILEID) (terpri D-AUFILEID)
        ;)
      ;transitions)
   ;(princ ")" D-AUFILEID)
   ;(terpri D-AUFILEID)
;)

(defun D-PrintTrans (transitions)
;;
;;  prints it out the transitions so that of reloading a structure
;;  of assq list is built - experimental
;;
   (D-WriteLambdaDef transitions 'D-INITSTATE)
)

(defun D-WriteLambdaDef (trans startstate)
;;
;;   Writes out the transitions in a lambda form that when called will 
;;   created a linked structure.  This by-passes the need for names of states
;;   now arcs go to new states as opposed to names of them
;;
   (let ( )
      (terpri D-AUFILEID)
      (princ "(defun " D-AUFILEID)
      (princ 'D-BUILDSPRULES D-AUFILEID)
      (princ " ()" D-AUFILEID) (terpri D-AUFILEID)
      (princ "(let (" D-AUFILEID) (terpri D-AUFILEID)
	 (mapcar     ;; cons cells for each states so references can be made
	    #'(lambda (statetrans)
                (cond
		   ((null (car statetrans)) nil)
		   (t
		      (princ "(" D-AUFILEID) (princ (car statetrans) D-AUFILEID)
		      (princ " (ncons '" D-AUFILEID)
		      (print (cadr statetrans) D-AUFILEID) ;; state info
		      (princ "))" D-AUFILEID)
		      (terpri D-AUFILEID)
                   )
                )
              )
            trans)
         (princ "(DEND (ncons 'DEND))" D-AUFILEID) (terpri D-AUFILEID)
	 (princ "(" D-AUFILEID)
	 (print (DK-D-ERROR) D-AUFILEID)
	 (princ " (ncons '" D-AUFILEID)
	 (print (DK-D-ERROR) D-AUFILEID)
	 (princ " ))" D-AUFILEID) (terpri D-AUFILEID)
      (princ ")" D-AUFILEID) (terpri D-AUFILEID)  ;; end for list in let
      ;;
      ;;  reset all these states to transitions         
      ;;
      (mapcar
	 #'(lambda (statetrans)
	      (D-WriteTrans (car statetrans)
		            (cddr statetrans)) ;; the actual transitions
           )
         trans)
      (terpri D-AUFILEID)
      (print startstate D-AUFILEID) (terpri D-AUFILEID);; return the structure
      (princ "))" D-AUFILEID) (terpri D-AUFILEID)
   )
)

(defun D-WriteTrans (statename symbolstates)
;;
;;   writes out the code that will build an a-list of ('symbol <state>)
;;
   (princ "(setq " D-AUFILEID)
   (print statename D-AUFILEID)
   (princ " (rplacd " D-AUFILEID)
   (print statename D-AUFILEID)
   (princ " (list " D-AUFILEID)
   (terpri D-AUFILEID)
   (mapcar
      #'(lambda (trans)   ;; the assoc list
         (princ "(list '" D-AUFILEID)
	 (print (car trans) D-AUFILEID)    ;; key
         (princ " " D-AUFILEID)
	 (mapcar         ;; print out the names of the other a-lists
	    #'(lambda (name)
	       (princ name D-AUFILEID) (princ " " D-AUFILEID)) (cdr trans))
         (princ " " D-AUFILEID)
         (princ ")" D-AUFILEID)
         (terpri D-AUFILEID)
      )
      symbolstates
   )
   (princ ")))" D-AUFILEID)
   (terpri D-AUFILEID)
)

(defun D-ExpandWClauses (rules)
;;
;;  expands the where clause in each rule
;;
   (mapcar 
      #'D-ExpandWhereClause
      rules
   )
)

(defun D-ExpandWhereClause (rule)
;;
;;  takes a KK rules and expands out the WHERE cluse.  This duplicates
;;  the rule.  The result is a list of rules.  These are not appended
;;  with the other expended rules but kept together, as an automaton is
;;  created for each user specified rule.
;;
   (cond
      ((null (D-GetWhereClause rule))
	 (list rule)                   ;; no where clause to expand
      )
      (t
	 (mapcar             ;; for each symbol in where list substitute
	    #'(lambda (symbol)   ;; it in the rule
		  (D-SubstituteSymbol
		     (D-GetWhereSymbol rule)
		     symbol 
		     rule
		  )
              )
            (D-GetWhereList rule)
         )
      )
   )
)

(defun D-SubstituteSymbol (whereletter symbol rule)
;;
;;  replaces all occurrences of whereletter in rule with symbol
;;
;;  Im not sure if this is correct but I can always correct it
;;  later (as they say)
;;
   (subst symbol whereletter rule)
)

(defun D-FindFeasibles (rules)
;;
;;  Finds all the feasibles pairs in the rules (CPS - Kimmo).  A concrete pair
;;  consists of no set names (or where letters which should have been
;;  removed at this point).  The feasibles are made up from all concrete
;;  pairs mentioned in the rules, the default pairs defined in the
;;  basic declarations and identity pairs made from the intersection
;;  of the lexical and surface alphabets.  
;;
   (D-union
      (D-RemoveDuplicates
	 (D-FindConcretePairs rules)
      )
      (D-union
	 (D-RemoveDuplicates
	    D-DEFAULTPAIRS         ;; as defined in the declarations
         )
	 (mapcar     ;; identity pairs from the intersection of the
	    #'(lambda (symbol)   ;; lexical and surface alphabets
		  (list symbol symbol))
            (D-intersection D-SURFACEALPHABET D-LEXICALALPHABET)
         )
      )
   )
)

(defun D-FindConcretePairs (rules)
;;
;;  find all concrete pairs mentioned in the rules.  The rules should
;;  have their where clauses expanded.
;;
   (mapcan
      #'(lambda (rule)
	 (mapcan
	    #'D-FindCPIndividualRule
	    rule                ;; split up expansion down by where clause
	 ))
      rules
   )
)

(defun D-FindCPIndividualRule (rule)
;;
;; search individual rule for concrete pair
;;
  ;(print rule) (terpri)
  (cond
     ((D-IsConcretePair (D-GetRulePair rule))
	(cons
	   (D-GetRulePair rule)   ;; rule pair is concrete
	   (mapcan
	      #'(lambda (context)
		 (nconc
		    (D-FindCPContext (car context))
		    (D-FindCPContext (cadr context))))
	      (D-GetRuleContexts rule))
        )
     )
     (t         ;; rule pair is not concrete
	(mapcan
	   #'(lambda (context)
	      (nconc
		 (D-FindCPContext (car context))
		 (D-FindCPContext (cadr context))))
	   (D-GetRuleContexts rule))
     )
   )
)

(defun D-FindCPContext (context)
;;
;;   finds all concrete pairs in the given context
;;
   (cond
      ((null context)         ;; end of search
	 nil
      )
      ((memq (car context) '(SEQ OR OPT))   ;; complex item
	 (mapcan
            #'D-FindCPContext
	    (cdr context)
         )
      )
      ((D-IsConcretePair context)  ;; assumed to be simple pair
	 (ncons context)
      )
      (t
	  nil             ;; not a concrete pair
      )
   )
)

(defun D-IsConcretePair (pair)
;;
;;  returns true if pair is concrete that is car in member of
;;  lexical alphabet and cadr is member of surface alphabet
;;
   (and
      (or (eq (car pair) (DK-NULL)) (memq (car pair) D-LEXICALALPHABET))
      (or (eq (cadr pair) (DK-NULL)) (memq (cadr pair) D-SURFACEALPHABET))
   )
)

(defun D-MakeAutomata (rules feasibles)
;;
;;  takes a list of kk rules (with where list expanded) and
;;  returns a list of transitions that describe an non-deterministic automaton
;;  which represents the rules
;;
   (let ( statename  
	  (transitions (ncons (list 'D-INITSTATE '(nil nil t (INIT))))) )
        (mapcar
	 #'(lambda (rule)
	    (princ "Compiling: ")
	    (princ (D-GetRuleName (car rule)))
	    (terpri)
	    (setq D-CURRENTRULENAME (D-GetRuleName (car rule)))
	    (D-MakeAutoFromRules
	       (D-GetRuleName (car rule))
	       rule
	       (D-MakeRulePairs rule feasibles)
	       feasibles
	       transitions
	    ))
	 rules)
      ;; all non-restricted pairs are acceptable in any context
      (setq D-CURRENTRULENAME 'DEFAULT)
      (setq statename (D-NewStateName '(TERMINAL LICENCE) transitions))
      (mapcar 
	 #'(lambda (pair)
	    (D-MakeTransitions
	       'D-INITSTATE
	       pair
	       statename
	       feasibles transitions))
         (D-ldifference feasibles 
	    (D-FindRestrictedPairs rules feasibles))
      )
      transitions     ;; return the transitions list
   )
)

(defun D-FindRestrictedPairs (rules feasibles)
;;
;;   Returns all pairs which are restricted by => or <=> rules
;;   note this could be a bag rather than a set.  If it is a set
;;   it means no rules are about the same pairs which is a GOOD THING
;;
;;
   (mapcan
      #'(lambda (rule)
	     (cond
		((memq (D-GetOperatorType (car rule)) '(=> <=>))
		   (D-MakeRulePairs rule feasibles)
		)
		(t nil)
	     )
        )
      rules
   )
)

(defun D-MakeRulePairs (rule feasibles) 
;; 
;;  Takes the rule and finds a actual (concrete) pairs that the 
;;  original rule pair stands for.
;;  This requires two stages
;;    1. Each pair from each where clause
;;    2. the expansion of which to remove sets
;;  then these are made into a set (duplicates removed)
;;
   (D-RemoveDuplicates
      (mapcan
         #'(lambda (rulepart)
	    (D-ExpandPair (D-GetRulePair rulepart) feasibles)
         )
         rule
      )
   )
)

(defun D-MakeAutoFromRules (name rule rulepairs feasibles transitions)
;;
;;   builds a pattern for each OR clause in a rule
;;
   (mapcar
      #'(lambda (partrule)   ;; split where parts
	    (D-MakeAutomaton
	       partrule rulepairs (caar transitions)  ;; initstate
	       transitions feasibles))
      rule
   )
)

(defun D-MakeAutomaton (rule rulepairs initstate transitions feasibles)
;;
;;  Build a pattern for the given context and rule pair.
;;
;;  built up by destruction modification of transitions
;;  transitions is returned
;;
   (cond
      ((eq (D-GetOperatorType rule) '=>)
	 (mapcar
	    #'(lambda (contexts)
		  (D-MakeAutomatonCR 
		     (D-GetRulePair rule)
		     rulepairs         ;; all the relevant pairs
		     feasibles
		     (car contexts)    ;; left context
		     (cadr contexts)   ;; right context
		     initstate
		     transitions))     ;; will be destructively modified
	    (D-GetRuleContexts rule)
         )
      )
      ((eq (D-GetOperatorType rule) '<=)
	 (mapcar
	    #'(lambda (contexts)
		  (D-MakeAutomatonSC 
		     (D-GetRulePair rule)
		     rulepairs         ;; all the relevant pairs
		     feasibles
		     (car contexts)    ;; left context
		     (cadr contexts)   ;; right context
		     initstate
		     transitions))     ;; will be destructively modified
	    (D-GetRuleContexts rule)
         )
      )
      (t      ;; must be '<=> then
	 (mapcar
	    #'(lambda (contexts)
		  (D-MakeAutomatonSCCR
		     (D-GetRulePair rule)
		     rulepairs         ;; all the relevant pairs
		     feasibles
		     (car contexts)    ;; left context
		     (cadr contexts)   ;; right context
		     initstate
		     transitions))     ;; will be destructively modified
	    (D-GetRuleContexts rule)
         )
      )
   )
   transitions     ;; return the transitions list
)

(defun D-MakeAutomatonCR (pair rulepairs feasibles leftcontext
			     rightcontext start transitions)
;;
;;  Makes an automaton for the Context restriction rule
;;         pair => leftcontext -- rightcontext
;;  That is pair can be accepted only within the left and right 
;;  context.  Any thing else (feasible) can be accepted during the
;;  left context but if the pair is found after the leftcontext
;;  the rightcontext MUST exist.
;;
;;  builds the transitions by modifying the variable transitions
;;
   (let ( (leftend (D-NewStateName '(FINAL) transitions))      ;; final
	  (rightstart (D-NewStateName '(LICENCE) transitions))
	  (rightend (D-NewStateName '(TERMINAL) transitions)) )
      (D-MakeAutoContext               ;; get pair goto error
         leftcontext 
         start
         leftend
         feasibles '(FINAL) transitions)
      (D-MakeTransitions leftend pair rightstart feasibles transitions)
      (D-MakeAutoContext               ;; get anything else is error
         rightcontext 
         rightstart        ;; once this state is hit must finish
         rightend
         feasibles '() transitions)
   )
)

(defun D-MakeAutomatonSC (pair rulepairs feasibles leftcontext
			     rightcontext start transitions)
;;
;;  Makes an automaton for the Surface Coercion rule
;;         pair <= leftcontext -- rightcontext
;;  That is if context left and right exist pair MUST exist between
;;  them.
;;
;;  builds the transitions by modifying the structure pointed to in 
;;  the variable transitions
;;
   (let ( (leftend (D-NewStateName '(FINAL) transitions))       ;; final
	  (rightstart (D-NewStateName '(FINAL) transitions)) )  ;; final
      (D-MakeAutoContext             ;; Left context
         leftcontext 
         start
         leftend
         feasibles '(FINAL) transitions)
      (D-MakeTransitions             ;; Not the Rule Pair
	 leftend 
	 (list
	    (car pair)            ;; lexical and
	    (caar D-SURFACESETS)) ;; all other surface characters
	 rightstart
	 (D-ldifference feasibles rulepairs)
	 transitions)
      (D-MakeAutoContext             ;; Right Context goes to error state
         rightcontext 
         rightstart
         (DK-D-ERROR)
         feasibles '(FINAL) transitions)
   )
)

(defun D-MakeAutomatonSCCR (pair rulepairs feasibles leftcontext
			       rightcontext start transitions)
;;
;;  Makes an automaton for the Composite Rules made from <= and =>
;;         pair <=> leftcontext -- rightcontext
;;  That is if context left and right exist pair MUST exist between
;;  them and pair cannot appear at any other time.
;;
;;  This could formally be the intersection of the <= and => automata
;;  but that is quite computationally expensive so its done as a
;;  separate automaton instead
;;
;;  the sub parts of this modify the variable transitions
;;
   (let ( (leftend (D-NewStateName '(FINAL) transitions)) ;; final state
	  (rightstart (D-NewStateName '(LICENCE) transitions))
	  (rightend (D-NewStateName '(TERMINAL) transitions))
	  (rightnotpstart (D-NewStateName '(FINAL) transitions)) ) ;; not final
      (D-MakeAutoContext             ;; Left Conext
         leftcontext                 ;; same as CR
         start
         leftend
         feasibles '(FINAL) transitions)    ;; same as =>
      (D-MakeTransitions
	 leftend 
	 (list 
	    (car pair) 
	    (caar D-SURFACESETS))  ;; lexical and any other surface characters
	 rightnotpstart
	 (D-ldifference feasibles rulepairs)
	 transitions)
      (D-MakeAutoContext     ;; same as => right context
         rightcontext 
         rightnotpstart
         (DK-D-ERROR)
	 feasibles
	 '(FINAL) transitions)
      (D-MakeTransitions
         leftend pair rightstart feasibles transitions)
      (D-MakeAutoContext     ;; same as => right context
         rightcontext 
         rightstart
         rightend 
	 feasibles
	 '() transitions)
   )
)

(defun D-MakeAutoContext (context start end feasibles type transitions)
;;
;;  converts a context into a list of automaton transitions
;;
   (cond
      ((null context)   ;; ??????
	 nil
      )
      ((eq (car context) 'SEQ)   ;; sequential operator
	 (D-MkAutoSEQ
	    (cdr context) start end feasibles type transitions)
      )
      ((eq (car context) 'OR)    ;; choice operator
	 (D-MkAutoOR
	    (cdr context) start end feasibles type transitions)
      )
      ;; the next three are not implemented as they are two
      ;; difficult to get right without introducing nulls into
      ;; the automata
      ((eq (car context) 'OPT)   ;; optional operator
	 (D-MkAutoOPT           
	    (cdr context) start end feasibles type transitions)
      )
      ((eq (car context) '0+)    ;; kleene star
	 (D-MkAutoKS
	    (cdr context) start end feasibles type transitions)
      )
      ((eq (car context) '1+)    ;; kleene plus
	 (D-MkAutoKP
	    (cdr context) start end feasibles type transitions)
      )
      (t          ;; must be simple pair then
	 (D-MkAutoContextPair
	    context start end feasibles transitions)
      )
   )
)

(defun D-MkAutoSEQ (things start end feasibles type transitions)
;;
;;  deal with the sequential operator
;;
   (cond
      ((eq (length things) 1)  ;; if its the last one
	 (D-MakeAutoContext
	    (car things) start end feasibles type transitions)
      )
      (t             ;; not last item
	 (let ( (intermediate (D-NewStateName type transitions)) )
	    (D-MakeAutoContext
	       (car things) start intermediate feasibles type transitions)
            (D-MkAutoSEQ
	       (cdr things) intermediate end feasibles type transitions)
         )
      )
   )
)

(defun D-MkAutoOR (choices start end feasibles type transitions)
;;
;;  deal with the choice operator
;;
   (mapcar
      #'(lambda (choice)
	 (D-MakeAutoContext
	    choice start end feasibles type transitions))
      choices  
   )
)

(defun D-MkAutoOPT (option start end feasibles type transitions)
;;
;; deal with optional operator
;;
   (D-AddNewTransition start nil end transitions)
   (D-MakeAutoContext
      (car option) start end feasibles type transitions)
)

(defun D-MkAutoKS (thing start end feasibles type transitions)
;;
;;  kleene start  (zero or more occurrences)
;;
   (D-AddNewTransition start nil end transitions)
			 ;; epsilon transition for zero occurrences
   (D-MakeAutoContext
      (car thing) end end feasibles type transitions)
)

(defun D-MkAutoKP (thing start end feasibles type transitions)
;;
;;  kleene plus  (one or more occurrences)
;;
   (D-MakeAutoContext
      (car thing) start end feasibles type transitions)
   (D-MakeAutoContext
      (car thing) end end feasibles type transitions)
)

(defun D-MkAutoContextPair (pair start end feasibles transitions)
;;
;;  I am seem to have reduced this to something very simple.  That is 
;;  just building a simple arc (or arcs if pair is complex) between
;;  the start and the end.
;;  
   (D-MakeTransitions start pair end feasibles transitions)
)

(defun D-MakeTransitions (start pair end feasibles transitions)
;;
;;   expands a pair to produce all possible transitions, with concrete
;;   pairs as the symbols
;;
;;  note transitions is modified
;;
   (cond
      ((D-IsConcretePair pair)  ;; if its concrete dont bother expanding
	 (D-AddNewTransition    ;; this is not necessary but saves time
	    start
	    (D-MakePairToken pair) end transitions)
      )
      (t
	 (mapcar
	    #'(lambda (concretepair)
		  (D-AddNewTransition
		     start
		     (D-MakePairToken concretepair) end transitions))
            (D-ExpandPair pair feasibles)  ;; find all the feasibles that match
         )
      )
   )
)

(defun D-AddNewTransition (state symbol newstate transitions)
;;
;;  This adds the new transiton to the trnasitions structure.
;;  This is done modifying the transition structure rather than 
;;  rebuilding it.  The compilation process is computationally
;;  expensive so I have to try and save as much time as possible
;;
;;  The state will always exist by this time
;;
;;  note transitions is assumed to be not empty
;;
;;  new states are added on the front cause its easy and states
;;  are usually dealt with together.
;;
;;  this cant cope with nulls and complex symbols in the surface
;;  and lexical alphabets 
;;
;;  This also ensures duplicate transitions are not added to the structure
;;
   (let ( (statepos (assq state transitions)) symbpos)
      (cond
	 ((null (setq symbpos (assq symbol (cddr statepos))))
	    (rplacd            ;; state is not new but symbol is 
	       (cdr statepos)
	       (cons
		  (cons symbol (ncons newstate))
		  (cddr statepos)))
         )
	 ((memq newstate (cdr symbpos))     ;; check that this transitions
            nil                              ;; is not already there
         )
	 (t             ;; symbol and state are already there
	    (rplacd
	       symbpos
	       (cons newstate (cdr symbpos)))
         )
      )
      transitions    ;; so I can see it during trace
   )
)

(defun D-CheckAndAddOrigins (state)
;;
;;   Checks to see if this state is already marked as comming from the 
;;   current rule.  If not the name of the current rule is added to the
;;   list of origins of the state
;;
   (cond
      ((memq D-CURRENTRULENAME (D-GetOrigins state))
         state   ;; no need to add it as it exists already
      )
      (t         ;; not there so add it
         (D-AddOrigin D-CURRENTRULENAME state)
      )
   )
)

(defun D-ExpandPair (pair feasibles)
;;
;; This function searches the feasibles for all pairs that match
;; pair
;; 
   (let  (possiblepairs)
      (mapc
	 #'(lambda (feasiblepair)
	    (cond
               ((D-ComPair pair feasiblepair)
		  (setq possiblepairs
	             (cons
	                feasiblepair possiblepairs)))
               (t nil)))
         feasibles)
      ;(cond                       ;; this doesn't work as sometimes 
	 ;((null possiblepairs)    ;; pairs do expand to nil - deliberately
	    ;(princ "Pair \"")
	    ;(princ pair)
	    ;(princ "\" represents no feasible pairs")
	    ;(terpri)
	    ;(error " "))
         ;(t
	    ;possiblepairs)
      ;)
      possiblepairs
   )
)

(defun D-ComPair (pair1 pair2)
;;
;;    Compares two pairs if they match (thats not the same 
;;    as equal) then return t else nil
;;
;;    pair1 is from a rule
;;    pair2 is a concrete pair from the list of feasibles (CPS)
;;
   (and
      (memq   
	 (car pair2) 
	 (D-ExpandSet (car pair1) D-LEXICALSETS))
      (memq
	 (cadr pair2)
	 (D-ExpandSet (cadr pair1) D-SURFACESETS))
   )
)

(defun D-ExpandSet (setname setlist)
;;
;;  returns the members of the set if it is a set otherwise
;;  it returns the setname in a list
;;
   (let  ( (setpair (assq setname setlist)) )
      (cond
	 ((null setpair)             ;; not a set
	    (ncons setname) ;; this is a symbol is a set containing itself
         )
	 (t                          ;; is a set
	    (cadr setpair)  ;; the set range
         )
      )
   )
)

(defun D-NewStateName (type transitions)
;;
;;  adds a new state to transitions and sets its type.  Returns 
;;  the new state
;;
   (let ( (newname (newsym 'D))
	  (origins (ncons D-CURRENTRULENAME))
	  (terminaltype (memq 'TERMINAL type))
	  (licencetype (memq 'LICENCE type))
	  (finaltype (memq 'FINAL type)) )
      (rplacd             ;; splices newstate into transitions
	 transitions
	 (cons
	    (list
	       newname
	       (list
		  terminaltype licencetype finaltype
		  newname origins))
	    (cdr transitions)))
      newname     ;; return new name    
   )
)

