;;
;;      Title :  Recog
;;
;;      Function :  Matches surface words to morphemes using the 
;;                  automata method (Koskenniemi & Karttunen)
;;
;;      Author :  Alan W Black     Nov 1984
;;                Dept of A.I. University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;       1.5  Added D-LookUpDict a routine that just look ups
;;            a lexicon irrespective of the morphology.
;;       1.11  Changed the format of the automata, now created from
;;             Kimmo Spelling rules.  In this version the automata
;;             are nondeterministic (September 1985)
;;       1.16  Changed the format of the lexicon and tried to tidy
;;             the search up a bit
;;       1.17  Changed the format of the automata to be hash-tables
;;             and also made the automata be represented as a structure
;;             rather than transitions being held separately.
;;             - then changed it back again
;;       1.18  This has changes to allow the actual spelling formalism
;;             interpreter to be contained in another file.  Thus changes
;;             of formalism can be made more easier.  Also the searching
;;             algorithm has been modified to avoid duplicate moving of
;;             the rules
;;       2.3 25th October 1986
;;          Changed search strategy to try to stop unnecessary searching
;;          on the lexicon tree.  The tree is now ordered and the global
;;          D-SURFTOLEXSETS holds possible lexical alternatives for
;;          each surface pair.
;;
;;      Description :
;;       These functions implement a spelling rule
;;       analyser allowing words to be split up into individual
;;       morphemes (taking into account dropped e's etc)
;;
;;       This file deals with the search strategy, that is comparing the 
;;       the surface string with the lexicon up to the next morpheme
;;       boundary.
;;
;;       The actual formalism and interpretation of rules is dealt with
;;       in the file spmove (via the function D-CheckPairMatch)
;;
;;      Parameters :
;;
;;      Returns :
;;
;;      >>>>   see D-CreateReturn   <<<<
;;
;;      Side Effect :
;;         none                   
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;


(declare
   (special
      D-LEXICON
      D-ENTRYFILEID
      D-LookUpDict     ;; simple dictionary look up
      D-MORPHEMES
      D-NULLPOSSIBLES
   )
   (localf 
      D-SearchRelevantSubtrees
      D-SearchRelevantSubtreesNull
      D-CreateReturn                  
      D-CompDict                      
   )
)

(include "keywords")
(include "subrout")
;;(include "sprulemove")   ;; new spelling rule form
(include "spmoveau") 
(include "catrouts")

(defun D-Recog (word)
;;
;;   This is the main calling function.
;;   This searches up until the next morpheme. i.e. how many matches
;;   are possible with this lexicon
;;   Returns a list of possible first morphemes in the given word
;;   D-LEXICON is list of lexicons whose car is an open port to the
;;   entries file, and cdr is a list of lexicon trees
;;
   (setq D-MORPHEMES nil)
   (mapc
      #'(lambda (lex)     ;; do this for each lexicon
	    (setq D-ENTRYFILEID (car lex))  ;; look up right entry file
	       (D-SearchRelevantSubtreesNull
		  (car word)           ;; configuration
		  (cdr word)           ;; remainder 
		  (cdr lex))            ;; subtrees
	       (D-SearchRelevantSubtrees
		  (car word)        ;; initial configuration of the automata
		  (cdr word)        ;; the remainder of the word
		  (cdr lex)         ;; subtrees
		  (D-PossibleLexChars (cadr word))))
      D-LEXICON
   )
   D-MORPHEMES
)

(defun D-SearchRelevantSubtrees (config remainder sublexs lexpossibles)
;;
;;   Searches down the sublexs to the next sub lex that is in both
;;   lexpossibles.  Then submits it for test to the spelling rules
;;   Returns a list of possible next morphemes
;;
   (let ( newconfig )
      (cond
	 ((eq (caar sublexs) (DK-ENDTREE)) ;; an end of tree
	    (D-CreateReturn
	       (D-LexSubs (car sublexs)) config remainder)))

      (cond                  ;; lexical null
	 ((eq (car lexpossibles) (DK-NULL))
	    (cond
	       ((eq (setq newconfig 
			(D-CheckPairMatch (DK-NULL) (car remainder) config))
                    'ERROR)
                   nil)    ;; is a subtree but spelling rules don't accept it
               (t     ;; does accept so drop a level
		  (D-SearchRelevantSubtreesNull
		     newconfig (cdr remainder)
		     sublexs) 
                  (D-SearchRelevantSubtrees
		     newconfig (cdr remainder)
		     sublexs (D-PossibleLexChars (cadr remainder)))
                  t)
            )
	    (setq lexpossibles (cdr lexpossibles))
         )
      )

      (mapc
	 #'(lambda (lexpossible)
	    (let ( (subtree (assq lexpossible sublexs)) )
            (cond
	       ((null subtree) nil)   ;; no matching sublex
	       ((eq (setq newconfig 
			(D-CheckPairMatch lexpossible (car remainder) config))
                    'ERROR)
                   nil)    ;; is a subtree but spelling rules don't accept it
               (t     ;; does accept so drop a level
		  (D-SearchRelevantSubtreesNull
		     newconfig (cdr remainder)
		     (cdr subtree))
                  (D-SearchRelevantSubtrees
		     newconfig (cdr remainder)
		     (cdr subtree) (D-PossibleLexChars (cadr remainder))))
            )))
         lexpossibles)
   )
)
 
(defun D-SearchRelevantSubtreesNull (config remainder sublexs)
;;
;;   deals with the case with null on the surface this is
;;   slightly different from the above so to save speed I have
;;   just duplicated most of the above routine
;;
   (let ( newconfig )
      ;; I assume there can never be 0:0 as a pair in any context
      (mapc
	 #'(lambda (lexpossible)
	    (let ( (subtree (assq lexpossible sublexs)) )
            (cond
	       ((null subtree) nil)   ;; no matching sublex
	       ((eq (setq newconfig 
			(D-CheckPairMatch lexpossible (DK-NULL) config))
                    'ERROR)
                   nil)    ;; is a subtree but spelling rules don't accept it
               (t     ;; does accept so drop a level
		  (D-SearchRelevantSubtreesNull
		     newconfig remainder
		     (cdr subtree))
                  (D-SearchRelevantSubtrees
		     newconfig remainder
		     (cdr subtree) (D-PossibleLexChars (car remainder))))
            )))
         D-NULLPOSSIBLES)
   )
)

(defun D-CreateReturn (entries config remainder)
;;
;;  This function creates the structure that is returned to the 
;;  main caller.  
;;  returned format is
;;  ( <list of entries> ( <config> <remainder> ) )  or
;;  ( <list of entries> 'END )    if valid end of word
;;
;;  If not at end of remainder and a non-inflectable word is found
;;  do not return it
;;
   (cond
      ((eq (car remainder) (DK-ENDMARKER))     ;; at end of word
	 (cond
            ((D-Final config) ;; acceptable config
	       (setq D-MORPHEMES (cons
	          (list (mapcar #'D-ReadLexicalEntry entries) 'END)
		  D-MORPHEMES))
               )))
      (t     ;; not at end of word so only load inflectable entries
	 (setq D-MORPHEMES (cons
	    (list
	       (mapcan
	          #'(lambda (entrypos)
		     (cond
		        ((car entrypos)   ;; non-inflectable
			   nil)
                        (t
			   (ncons (D-ReadLexicalEntry entrypos)))))
                  entries)
               (cons
	          config
	          remainder)) D-MORPHEMES)))
   )
)

(defun D-LookUpDict (word)
;;
;;  This will look up the given word in the dictionary and return
;;  a list of possible entries for that word
;;
;;  Returns nil if there are no entries
;;
      (mapcan
         #'(lambda (biglex)     ;; do this for each lexicon
	       (setq D-ENTRYFILEID (car biglex))  ;; look up right entry file
               (mapcan 
	          #'(lambda (lex)    ;; each tree in a lexicon
                    (D-CompDict 
                       (append (D-SplitIntoChars word) (ncons (DK-ENDTREE)))
                       lex))                                 ;; lexicon tree
                  (cdr biglex)
               ))
         D-LEXICON)
)

(defun D-CompDict (lst tree)
;;
;;  Look up tree for all paths matching word
;;
   (cond 
      ((null lst)      ;; found list of entries
	 (let ( (le (D-ReadLexicalEntry tree)) )
	 (ncons 
	    (list
	       (D-CitationForm le)
	       (D-PhonologicalForm le)
	       (D-MakePCategory (D-Syntax-Field le))
	       (D-Semantic-Field le)
	       (D-User-Field le))
	 ))
      )
      ((eq (car lst) (D-LexHead tree))     ;; drop level in tree
	 (mapcan
	    #'(lambda (sublex)
		(D-CompDict (cdr lst) sublex))
            (D-LexSubs tree))
      )
      (t                   ;; does not match
	 nil
      )
   )
)

