# include <stdio.h>				/* xref.c (rev3.7) */
# include <ctype.h>
# include <signal.h>

usage()			/* print usage and synopsis of options */
{
	puts("Cross Reference Generator\t\t\t(rev3.7)");
	puts("Usage: xref [-r -ln -pn -ic -dPfile -] filename(s)");
	puts("-r : reset linenumber to 1 at beginning of every file");
	puts("-ln: line numbering begins with line n (instead of 1)");
	puts("-pn: page numbering begins with page n (instead of 1)");
	puts("-ic: page incrementer is character c (defaults to =)");
	puts("-wn: width of output page is n (defaults to 80)");
	puts("-d : define punctuation set according to Pfile");
	puts("-  : read text from standard input (terminal or pipe)");
	exit(1);
}

long lineno = 1;	/* count line numbers */
int resetno = 0;	/* reset lineno for new file */
char pgincr = '=';	/* page incrementing character */
int pageno = 0;		/* toggle and count page numbers */
int width = 70;		/* page width for entries */

char punctuation[512] = ",.;:-?!\"()[]{}" ;
char *tempfile = "/tmp/RefXXXXX";

main(argc, argv)	/* cross reference generator for a text */
int argc;
char *argv[];
{
	FILE *fp, *tfp, *fopen();
	char command[BUFSIZ], *mktemp(), *sprintf();
	int i, catch();

	if (argc == 1)
		usage();
	mktemp(tempfile);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, catch);
	for (i = 1; i < argc; i++)
	{
		if (*argv[i] == '-')
			getflag(argv[i]);
		else if ((fp = fopen(argv[i], "r")) != NULL)
		{
			xref(fp);
			fclose(fp);
			if (resetno)
				lineno = 1;
		}
		else  /* cannot open file */
		{
			fprintf(stderr,
			"Xref cannot access the file: %s\n", argv[i]);
			exit(1);
		}
	}
	sprintf(command, "sort %s -o %s", tempfile, tempfile);
	system(command);
	if ((tfp = fopen(tempfile, "r")) != NULL)
		frmt(tfp);
	unlink(tempfile);
	exit(0);
}

getflag(f)		/* parses command line to set options */
char *f;
{
	char *pfile;
	long atol();

	f++;
	switch(*f++)
	{
		case 'r':
			resetno = 1;
			break;
		case 'l':
			lineno = atol(f);
			break;
		case 'p':
			pageno = atoi(f);
			break;
		case 'i':
			pgincr = *f;
			break;
		case 'd':
			pfile = f;
			getpunct(pfile);
			break;
		case 'w':
			width = atoi(f) - 10;
			break;
		case NULL:
			xref(stdin);
			break;
		default:
			fprintf(stderr,
			"Invalid xref flag: -%s\n", --f);
			exit(1);
			break;
	}
}

getpunct(pfile)		/* read user's punctuation from pfile */
char *pfile;
{
	FILE *pfp, *fopen();
	char s[512], *strcpy();

	if ((pfp = fopen(pfile, "r")) == NULL)
	{
		fprintf(stderr,
		"Xref cannot access Punctfile: %s\n", pfile);
		exit(1);
	}
	else
		while (fgets(s, 512, pfp))
			strcpy(punctuation, s);
}

xref(fp)		/* prints cross reference for each word */
FILE *fp;
{
	FILE *tfp, *fopen();
	char word[BUFSIZ];

	if ((tfp = fopen(tempfile, "a")) == NULL)
	{
		perror(tempfile);
		exit(1);
	}
	while (getword(word, fp))
	{
		fprintf(tfp, "%s\002", word);  /* ctrl-b */

		if (pageno)
			fprintf(tfp, "%3d,%2ld; \n", pageno, lineno);
		else
			fprintf(tfp, "%5ld; \n", lineno);
	}
	fclose(tfp);
}

getword(word, fp)	/* drives program through text word by word */
char word[];
FILE *fp;
{
	static char nl = 0;

	if (nl)		/* increments lineno at beginning of line */
	{
		nl = 0;
		lineno++;
	}
	while ((*word = getc(fp)) && isskip(*word) && *word != EOF)
		if (*word == '\n')	/* skip to text */
			lineno++;

	if (*word == pgincr)
	{
		pageno++;	/* pgincr must begin word */
		lineno = 0;
	}
	if (*word == EOF)
		return(0);
	if (isupper(*word))
		*word = tolower(*word);

	while ((*++word = getc(fp)) && !isskip(*word) && *word !=EOF)
		if (isupper(*word))
			*word = tolower(*word);

	if (*word == '\n')	/* set nl at end of line */
		nl = 1;
	*word = NULL;

	return(1);
}

isskip(c)		/* function to evaluate punctuation */
char c;
{
	char *ptr;

	if (isspace(c))
		return(1);
	for (ptr = punctuation; *ptr != c && *ptr != NULL; ptr++)
		;
	if (*ptr == NULL)
		return(0);
	else
		return(1);
}

frmt(fp)		/* format keywords and line numbers */
FILE *fp;
{
	char s[BUFSIZ], entry[BUFSIZ], *initial = "~~~~~";
	char old[BUFSIZ], new[BUFSIZ], ref[BUFSIZ], *strcat();
	int i, j;

	strcpy(old, initial);  /* make sure 1st kwd printed */
	*entry = NULL;

	while (fgets(s, BUFSIZ, fp))
	{
		for (i = 0; s[i] != '\002' && i < 510; i++)
			new[i] = s[i];
		new[i] = NULL;

		for (++i, j = 0; s[i] != '\n' && i < 510; i++, j++)
			ref[j] = s[i];
		ref[j] = NULL;

		if (strcmp(new, old) != 0)
		{
			if (*entry != NULL)
				printf("%s\n", entry);
			entry[0] = ' ';
			entry[1] = NULL;
			printf("%s\n", new);
		}
		if (strlen(entry) < width)
			strcat(entry, ref);
		else  /* build entry */
		{
			printf("%s\n", entry);
			entry[0] = ' ';
			entry[1] = NULL;
			strcat(entry, ref);
		}
		strcpy(old, new);
	}
	printf("%s\n", entry);
}

catch()			/* remove tempfile in case of interrupt */
{
	unlink(tempfile);
	fprintf(stderr, "\nInterrupt\n");
	exit(1);
}
