/****************************************************************/
/* Copyright 1993 : Johns Hopkins University			*/
/*                  Department of Computer Science		*/
/****************************************************************/
/* Contact : murthy@cs.jhu.edu					*/
/****************************************************************/
/* File Name :	util.c						*/
/* Author : Sreerama K. Murthy					*/
/* Last modified : October 1993					*/
/* Contains modules : 	error					*/
/*			myrandom				*/
/*			vector					*/
/*			free_vector				*/
/*			ivector					*/
/*			free_ivector				*/
/*			dvector					*/
/*			free_dvector				*/
/* Uses modules in : None.					*/
/* Is used by modules in :	classify.c			*/
/*				classify_util.c			*/
/*				compute_impurity.c		*/
/*				display.c			*/
/*				gendata.c			*/
/*				impurity_measure.c		*/
/*				load_data.c			*/
/*				mktree.c			*/
/*				perturb.c			*/
/*				train_util.c			*/
/*				util.c				*/
/* Remarks       :	These are utility routines to display	*/
/*			error messages, to allocate one 	*/
/*			dimensional vectors; to	generate random	*/
/*			numbers etc. These routines are used by	*/
/*			almost all other files in the package.	*/
/****************************************************************/		
#include "oc1.h"


/************************************************************************/
/* Module name : error							*/ 
/* Functionality :	Displays an error message, and exits execution	*/
/*			normally.					*/
/* Parameters :		error_text : Error message to be displayed.	*/
/* Returns :	Nothing.						*/
/* Calls modules :	None.						*/
/* Is called by modules :	almost all modules in the package.	*/
/************************************************************************/
void error(error_text)
char error_text[];
{

 printf("Run_Time Error: %s\n",error_text);
 printf("Now exiting to system..\n\n");
 exit(1);
}

/************************************************************************/
/* Module name :	myrandom					*/ 
/* Functionality :	Generates a random number between 0 and 1, and	*/
/*			scales it to the required range.		*/
/* Parameters :	above, below : lower and upper limits, respectively on	*/
/*		the random number to be generated.			*/
/* Returns :	a floating point number.				*/
/* Calls modules :	drand48 (C library call)			*/
/* Is called by modules :	main (gendata.c)			*/
/*				shuffle_points (load_data.c)		*/
/*				oblique_split (mktree.c)		*/
/*				suggest_perturbation (perturb.c)	*/
/*				linear_split (perturb.c)		*/
/*				generate_random_hyperplane 		*/
/*						(train_util.c)		*/
/************************************************************************/
float myrandom(above,below)
int above,below;
{
 double drand48(), x;

 x = drand48(); /* x is in the range 0.0 to 1.0 */

 x = above + x * (below - above);
 return((float)x);

}


/************************************************************************/
/* Module name : ivector						*/
/* Functionality :	Allocates a 1D integer array, whose indices	*/
/*			range from "nl" through "nh", and returns a	*/
/*			pointer to this array.				*/
/* Parameters :	nl,nh : lowest and highest indices.			*/
/* Returns :	pointer to the integer array 				*/
/* Calls modules :	error						*/
/* Is called by modules :	classify_and_estimate_accuracy 		*/
/*						(classify.c)		*/
/*				main (gendata.c)			*/
/*				initialize (mktree.c)			*/
/*				cross_validate (mktree.c)		*/
/************************************************************************/
int *ivector(nl,nh)
int nl,nh;
{
  int *v;

  v=(int *)malloc((unsigned)(nh-nl+1)*sizeof(int));
  if (!v) error("IVECTOR : allocation failure.");
  return v-nl;
}

/************************************************************************/
/* Module name : vector							*/
/* Functionality :	Allocates a 1D float array, whose indices	*/
/*			range from "nl" through "nh", and returns a	*/
/*			pointer to this array.				*/
/* Parameters :	nl,nh : lowest and highest indices.			*/
/* Returns :	pointer to the float array 				*/
/* Calls modules :	error						*/
/* Is called by modules :	read_hp (classify_util.c) 		*/
/*				main (gendata.c)			*/
/*				allocate_point_array (load_data.c)	*/
/*				initialize (mktree.c)			*/
/*				build_dt (mktree.c)			*/
/*				perturb_randomly (perturb.c)		*/
/************************************************************************/
float *vector(nl,nh)
int nl,nh;
{
  float *v;

  v=(float *)malloc((unsigned)(nh-nl+1)*sizeof(float));
  if (!v) error("VECTOR : allocation failure.");
  return v-nl;
}

/************************************************************************/
/* Module name : dvector						*/
/* Functionality :	Allocates a 1D double array, whose indices	*/
/*			range from "nl" through "nh", and returns a	*/
/*			pointer to this array.				*/
/* Parameters :	nl,nh : lowest and highest indices.			*/
/* Returns :	pointer to the double array 				*/
/* Calls modules :	error						*/
/* Is called by modules :	initialize (mktree.c)	 		*/
/************************************************************************/
double *dvector(nl,nh)
int nl,nh;
{
  double *v;

  v=(double *)malloc((unsigned)(nh-nl+1)*sizeof(double));
  if (!v) error("DVECTOR : allocation failure.");
  return v-nl;
}

/************************************************************************/
/* Module name : free_ivector						*/
/* Functionality :	Frees a 1D integer array. 			*/
/* Parameters :	v : pointer to the array				*/
/*		nl,nh : lowest and highest indices.			*/
/* Returns :	nothing.		 				*/
/* Calls modules :	None.						*/
/* Is called by modules :	classify_and_estimate_accuracy		*/
/*						(classify.c)		*/
/*				main(gendata.c)				*/
/*				deallocate (mktree.c)			*/
/************************************************************************/
void free_ivector(v,nl,nh)
int *v,nl,nh;
{
  free((char*)(v+nl));
}

/************************************************************************/
/* Module name : free_vector						*/
/* Functionality :	Frees a 1D float array. 			*/
/* Parameters :	v : pointer to the array				*/
/*		nl,nh : lowest and highest indices.			*/
/* Returns :	nothing.		 				*/
/* Calls modules :	None.						*/
/* Is called by modules :	main (gendata.c)			*/
/*				deallocate (mktree.c)			*/
/*				build_dt (mktree.c)			*/
/*				perturb_randomly (mktree.c)		*/
/************************************************************************/
void free_vector(v,nl,nh)
int nl,nh;
float *v;
{
  free((char*)(v+nl));
}

/************************************************************************/
/* Module name : free_dvector						*/
/* Functionality :	Frees a 1D double array. 			*/
/* Parameters :	v : pointer to the array				*/
/*		nl,nh : lowest and highest indices.			*/
/* Returns :	nothing.		 				*/
/* Calls modules :	None.						*/
/* Is called by modules :	deallocate (mktree.c)			*/
/************************************************************************/
void free_dvector(v,nl,nh)
int nl,nh;
double *v;
{
  free((char*)(v+nl));
}

/************************************************************************/
/************************************************************************/
