;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

(defvar *show-special-r-structs* t)

;;;_____________________________________
;;;  compute-message-height

(defun compute-message-height (message width font)
        (let ((total-length (length message))
              (end 0)
              (height 0))
          (do ((start 0 (+ end 1)))
              ((> start total-length))
            (setq end (or (position #\newline message :start start) total-length)
                  height (+ (ceiling (string-width (subseq message start end) font) width) height)))
          (multiple-value-bind (ascent descent maxwidth leading) (font-info font)
            (declare (ignore maxwidth))
            (* height (+ ascent descent leading)))))

;;;_____________________________________
;;;  add-message

(defun add-message (window string)
  (let* ((message-width (- (point-h (view-size window)) 10))
         (message-height (compute-message-height string message-width '("monaco" 9 :bold))))
    (add-subviews window (make-dialog-item 'static-text-dialog-item #@(5 5) (make-point message-width message-height)
                                           string nil :view-font '("monaco" 9 :bold) :view-nick-name :message))
    (+ message-height 10)))

;;;_____________________________________
;;;  selector-dialog

(defclass selector-dialog (window)
  ((default-when-selected :initarg :default-when-selected :initform nil :accessor default-when-selected)
   (enable-only-when-selected :initarg :enable-only-when-selected :initform nil :accessor enable-only-when-selected)
   (default-when-unselected :initarg :default-when-unselected :initform nil :accessor default-when-unselected)
   (buttons :initarg :buttons :initform nil :accessor buttons)
   (table-element-name-function :initarg :table-element-name-function :initform nil :accessor table-element-name-function)))

;;;_____________________________________
;;;  add-buttons-and-selector

(defmethod add-buttons-and-selector ((window selector-dialog) list
                                      &key
                                      (message "Select and item from list")
                                      (buttons nil)
                                      (enable-only-when-selected nil)
                                      (default-when-unselected nil)
                                      (default-when-selected nil)
                                      (print-function #'princ)
                                      (name-function nil)
                                      (selection-type :single)
                                      (window-select t))
  (setf (buttons window) (reverse buttons)
        (enable-only-when-selected window) enable-only-when-selected
        (default-when-unselected window) default-when-unselected
        (default-when-selected window) default-when-selected)
  (let* ((window-size (view-size window))
         (bh (- (point-h window-size) 15))
         (bv (- (point-v window-size) 26))
         (mv (add-message window message)))
    (add-subviews window
                  (make-dialog-item 'sequence-dialog-item (make-point 4 mv) nil "" nil
                                    :cell-size #@(10 5)
                                    :table-hscrollp nil
                                    :table-vscrollp t
                                    :table-sequence list
                                    :table-print-function print-function
                                    :selection-type selection-type
                                    :view-font '("Monaco" 9 :SRCOR :PLAIN)
                                    :view-nick-name :item-selector))
    (setf (table-element-name-function window) name-function)
    (dolist (button (buttons window))
      (setf bh (- bh (point-h (view-size button)) 10))
      (set-view-position button bh bv)
      (add-subviews window button))
    (move-selector-and-buttons window #@(0 0))
    (when window-select (window-select window))))

;;;_____________________________________
;;;  move-selector-and-buttons

(defmethod move-selector-and-buttons ((window selector-dialog) offset)
  (without-interrupts
   (with-focused-view window
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
       (#_eraserect view-rect)
       (#_beginupdate (wptr window))
       (let* ((item-selector (view-named :item-selector window))
              (new-size (view-size window))
              (window-height (point-v new-size))
              (width (- (point-h new-size) 8))
              (item-selector-height (* (floor (- window-height (point-v (view-position item-selector)) 36) 14) 14)))
         (set-view-size item-selector width item-selector-height)
         (set-cell-size item-selector (- width 15) 14)
         (dolist (button (buttons window))
           (set-view-position button (add-points (view-position button) offset)))
         (#_endupdate (wptr window))
         (#_invalrect view-rect))))))

;;;_____________________________________
;;;  set-view-size

(defmethod set-view-size ((window selector-dialog) h &optional v)
  (without-interrupts
   (let ((old-size (view-size window)))
     (call-next-method window h v)
     (move-selector-and-buttons window (subtract-points (view-size window) old-size)))))

;;;_____________________________________
;;;  window-zoom-event-handler

(defmethod window-zoom-event-handler ((window selector-dialog) message)
  (without-interrupts
   (let ((old-size (view-size window)))
     (call-next-method window message)
     (move-selector-and-buttons window (subtract-points (view-size window) old-size)))))

;;;_____________________________________
;;;  view-key-event-handler

(defmethod view-key-event-handler ((window selector-dialog) char)
  (let* ((selector (view-named :item-selector window))
         (top-visible-v (point-v (scroll-position selector)))
         (bottom-visible-v (+ top-visible-v (- (point-v (visible-dimensions selector)) 1)))
         (max-v (- (point-v (table-dimensions selector)) 1))
         (table-sequence (table-sequence selector))
         (upper-case (char-upcase char))
         (selected-cells (selected-cells selector))
         (selected-cell (first selected-cells))
         (table-element-name-function (table-element-name-function window))   
         (v (when selected-cell (point-v selected-cell))))

    (cond ((string= char #\UpArrow)
           (when v
             (let ((new-v (max 0 (- v 1))))
               (unless (= v new-v)
                 (cell-deselect selector 0 v)
                 (cell-select selector 0 new-v)
                 (when (< new-v top-visible-v)
                   (scroll-to-cell selector 0 (- top-visible-v 1)))))))

          ((string= char #\DownArrow)
           (when v
             (let ((new-v (min max-v (+ v 1))))
               (unless (= v new-v)
                 (cell-deselect selector 0 v)
                 (cell-select selector 0 new-v)
                 (when (> new-v bottom-visible-v)
                   (scroll-to-cell selector 0 (+ top-visible-v 1)))))))

          ((and table-element-name-function (string<= #\A upper-case) (string<= upper-case #\Z))
           (when v (cell-deselect selector 0 v))
           (setf v (or (position-if #'(lambda (table-element) 
                                        (string<= upper-case (elt (funcall table-element-name-function table-element) 0)))
                                    table-sequence)
                       max-v))
           (cell-select selector 0 v)
           (scroll-to-cell selector 0 v)
           (update-selector-dialog window))
          (t
           (call-next-method window char)))))

;;;_____________________________________
;;; select-corresponding-cell

(defmethod select-corresponding-cell ((dialog selector-dialog) contents)
  (let ((item-selector (view-named :item-selector dialog)))
    (cell-select item-selector
                 (index-to-cell item-selector (position contents (table-sequence item-selector))))))

;;;_____________________________________
;;;  view-click-event-handler

(defmethod view-click-event-handler ((window selector-dialog) where)
  (call-next-method window where)
  (update-selector-dialog window)
  (when (and (double-click-p) (default-button window))
    (dialog-item-action (default-button window))))

;;;_____________________________________
;;;  update-selector-dialog

(defmethod update-selector-dialog ((window selector-dialog) &optional (item-selector (view-named :item-selector window)))
  (let ((enable-disable-list (enable-only-when-selected window)))
    (cond ((selected-cells item-selector)
           (set-default-button window (default-when-selected window))
           (dolist (button enable-disable-list)
             (dialog-item-enable button)))
          (t 
           (set-default-button window (default-when-unselected window))
           (dolist (button enable-disable-list)
             (dialog-item-disable button))))))

;;;_____________________________________
;;;  pretty-print functions

(defun pretty-print-r-struct-name (r-struct stream)
  (format stream "~A ~(~A~) ~A"
          (cond ((rule-p r-struct) "")
                ((pred-p r-struct) " ")
                ((member r-struct *special-r-structs*) "")
                ((not (eq (r-kind r-struct) :undefined)) "")
                (t " "))
          (r-name  r-struct)
          (cond ((eq (r-kind r-struct) :undefined) "  <--  undefined")
                ((null (r-type r-struct)) "")
                (t (format nil "~S" (r-type  r-struct))))))

(defun pretty-print-type-name (type stream)
  (if (or (member type *all-types*)
          (member type *special-types*))
    (format stream "~(~s~)" type)
    (format stream "~(~s~)  <--  UNDEFINED" type)))

;;;_____________________________________
;;; create-relation-button

(defun create-relation-button (&optional (size #@(60 20)) (position #@(0 0)) (name " New "))
  (make-instance
    'pop-up-menu :view-size size :view-position position :auto-update-default nil :item-display name
    :menu-items
    (list
     (make-instance 'menu-item
       :menu-item-title "Fact"
       :menu-item-action #'(lambda ()
                             (user-create-relation :extensional)))
     (make-instance 'menu-item 
       :menu-item-title "Builtin"
       :menu-item-action #'(lambda ()
                             (user-create-relation :builtin)))
     (make-instance 'menu-item
       :menu-item-title "Rule"
       :menu-item-action #'(lambda ()
                             (user-create-relation :intensional))))))

;;;_____________________________________
;;; create-relation-manipulation-window

(defun create-relation-manipulation-window (&key (r-structs *r-structs*)
                                                 (title "Relations")
                                                 (message "Select a Relation")
                                                 (view-position :centered)
                                                 (view-size #@(450 200)))
  (let* ((window (make-instance 
                   'selector-dialog
                   :window-title title
                   :window-show nil
                   :view-position view-position
                   :view-size view-size))
         (new-button
          (create-relation-button #@(60 20) #@(0 0) " New "))
         
         (edit-button 
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) " Edit "
           #'(lambda (item)
               (let* ((item-selector (find-named-sibling item :item-selector))
                      (selected-cell (first (selected-cells item-selector))))
                 (when selected-cell
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-edit-relation *user-monitor*)))
                   (user-edit-relation (cell-contents item-selector selected-cell)))))
           :dialog-item-enabled-p nil))
         
         (rename-button
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) " Rename "
           #'(lambda (item)
               (let* ((item-selector (find-named-sibling item :item-selector))
                      (selected-cell (first (selected-cells item-selector))))
                 (when selected-cell
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-rename-relation *user-monitor*)))
                   (rename-relation (cell-contents item-selector selected-cell)))))
           :dialog-item-enabled-p nil))
         
         (delete-button
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) " Delete "
           #'(lambda (item)
               (let* ((item-selector (find-named-sibling item :item-selector))
                      (selected-cell (first (selected-cells item-selector))))
                 (when selected-cell
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-delete-relation *user-monitor*)))
                   (user-delete-relation (cell-contents item-selector selected-cell)))))
           :dialog-item-enabled-p nil))
         
         (who-calls-button 
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) " Callers "
           #'(lambda (item)
               (let* ((item-selector (find-named-sibling item :item-selector))
                      (selected-cell (first (selected-cells item-selector))))
                 (when selected-cell
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-who-calls-relation *user-monitor*)))
                   (who-calls (r-name (cell-contents item-selector selected-cell))))))
           :dialog-item-enabled-p nil))
         
         (display-button 
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) " Display "
           #'(lambda (item)
               (let* ((item-selector (find-named-sibling item :item-selector))
                      (selected-cell (first (selected-cells item-selector))))
                 (when selected-cell
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-display-relation *user-monitor*)))
                   (user-display-relation (cell-contents item-selector selected-cell)))))
           :dialog-item-enabled-p nil))
         
         (all-buttons (list new-button edit-button rename-button delete-button who-calls-button display-button))
         (enable-only-when-selected-buttons (list edit-button rename-button delete-button who-calls-button display-button)))
    
    (add-buttons-and-selector 
     window
     r-structs
     :message message
     :buttons all-buttons
     :enable-only-when-selected enable-only-when-selected-buttons
     :print-function #'pretty-print-r-struct-name
     :name-function #'(lambda (r-struct) (symbol-name (r-name r-struct))))
    
    (window-select window)
    window))

;;;_____________________________________
;;;  manipulate-relation

(defun manipulate-relation (&key (r-structs (if *show-special-r-structs* *r-structs* (user-defined-r-structs)))
                                 (title "Relations")
                                 (message "Select a Relation")
                                 (view-position :centered)
                                 (view-size #@(450 200)))
  (let ((window (find-window "Relations")))
    (unless (window-open? window)
      (setf window (create-relation-manipulation-window :r-structs r-structs
                                                        :title title
                                                        :message message
                                                        :view-position view-position
                                                        :view-size view-size)))
    (window-select window)
    window))

;;;_____________________________________
;;;  update-relations

(defun update-relations ()
  (let ((window (find-window "Relations")))
    (when (window-open? window)
      (let* ((selector (view-named :item-selector window))
             (sp (scroll-position  selector)))
        (dolist (cell (selected-cells selector))
          (cell-deselect selector cell))
        (set-table-sequence selector (if *show-special-r-structs* *r-structs* (user-defined-r-structs)))
        (scroll-to-cell selector sp)
        (update-selector-dialog window selector)
        ))))

;;;_____________________________________
;;;  manipulate-type

(defun manipulate-type (&key (types (user-defined-types))
                             (title "Types")
                             (message "Select a Type")
                             (view-position :centered)
                             (view-size #@(380 200)))
  (let ((window (find-window "Types")))
    (unless window
      (setf window (make-instance 
                     'selector-dialog
                     :window-title title
                     :window-show nil
                     :view-position view-position
                     :view-size view-size))
      (let* ((new-button
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " New "
               #'(lambda (item) item
                  (when (user-monitor-p *user-monitor*)
                    (incf (user-monitor-new-type *user-monitor*)))
                  (text-edit-type nil))))
             
             (edit-button 
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Edit "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-edit-type *user-monitor*)))
                       (text-edit-type (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             (rename-button 
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Rename "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-edit-type *user-monitor*)))
                       (rename-type (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             (copy-button 
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Copy "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-edit-type *user-monitor*)))
                       (copy-type (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             (delete-button
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Delete "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-delete-type *user-monitor*)))
                       (delete-type (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             
             
             (all-buttons (list new-button edit-button rename-button copy-button delete-button))
             (enable-only-when-selected-buttons (list edit-button rename-button copy-button delete-button)))
        
        (add-buttons-and-selector 
         window
         types
         :message message
         :buttons all-buttons
         :enable-only-when-selected enable-only-when-selected-buttons
         :print-function #'pretty-print-type-name
         :name-function #'(lambda (x) (symbol-name x)))))
    (window-select window)
    window))

;;;_____________________________________
;;;  update-types

(defun update-types ()
  (let ((window (find-window "Types")))
    (when (window-open? window)
      (let* ((selector (view-named :item-selector window))
             (sp (scroll-position  selector)))
        (dolist (cell (selected-cells selector))
          (cell-deselect selector cell))
        (set-table-sequence selector (user-defined-types))
        (scroll-to-cell selector sp)
        (update-selector-dialog window selector)
        ))))


;;;_____________________________________
;;;  manipulate-cliche

(defun manipulate-cliche (&key (cliches *all-cliches*)
                               (title "Clichs")
                               (message "Select a Clich")
                               (view-position :centered)
                               (view-size #@(400 200)))
  (let ((window (find-window "Clichs")))
    (unless window
      (setf window (make-instance 
                     'selector-dialog
                     :window-title title
                     :window-show nil
                     :view-position view-position
                     :view-size view-size)) 
      (let ((new-button
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(60 20) " New "
              #'(lambda (item) item
                 (when (user-monitor-p *user-monitor*)
                   (incf (user-monitor-new-cliche *user-monitor*)))
                 (text-edit-cliche nil))))
            
            (edit-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(60 20) " Edit "
              #'(lambda (item)
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (when (user-monitor-p *user-monitor*)
                        (incf (user-monitor-edit-cliche *user-monitor*)))
                      (text-edit-cliche (rest (cell-contents item-selector selected-cell))))))
              :dialog-item-enabled-p nil))
            
            (delete-button
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(60 20) " Delete "
              #'(lambda (item)
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (when (user-monitor-p *user-monitor*)
                        (incf (user-monitor-delete-cliche *user-monitor*)))
                      (delete-cliche (rest (cell-contents item-selector selected-cell))))))
              :dialog-item-enabled-p nil))
            
            (use-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(60 20) " Use "
              #'(lambda (item) 
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (when (user-monitor-p *user-monitor*)
                        (incf (user-monitor-use-cliche *user-monitor*)))
                      (let ((name (first (cell-contents item-selector selected-cell))))
                        (setf *available-relational-cliches* (pushnew name *available-relational-cliches*))
                        (when *focl-problem*
                          (setf (getf (rest *focl-problem*) :available-cliches) (pushnew name (getf (rest *focl-problem*) :available-cliches)))))
                      (invalidate-view item-selector))))
              :dialog-item-enabled-p nil))
            
            (dont-use-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(80 20) " Don't Use "
              #'(lambda (item) 
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (when (user-monitor-p *user-monitor*)
                        (incf (user-monitor-dont-use-cliche *user-monitor*)))
                      (let ((name (first (cell-contents item-selector selected-cell))))
                        (setf *available-relational-cliches* (delete name *available-relational-cliches*))
                        (when *focl-problem*
                          (setf (getf (rest *focl-problem*) :available-cliches) (delete name (getf (rest *focl-problem*) :available-cliches)))))
                      (invalidate-view item-selector))))
              :dialog-item-enabled-p nil))
            )
        
        (add-buttons-and-selector 
         window
         cliches
         :message message
         :buttons (list new-button edit-button delete-button use-button dont-use-button)
         :enable-only-when-selected (list edit-button delete-button use-button dont-use-button)
         :print-function #'(lambda (pair stream)
                             (let ((name (cliche-name (rest pair))))
                               (format stream "~A ~(~S~)" 
                                       (if (member name (getf (rest *focl-problem*) :available-cliches *available-relational-cliches*))
                                         "" " ")
                                       name)))
         :name-function #'(lambda (pair) (symbol-name (cliche-name (rest pair))))
         :selection-type :single)))
    (window-select window)
    window))

#|
            (inspect-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(60 20) " Inspect "
              #'(lambda (item)
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector)))
                         (selected-pair (cell-contents item-selector selected-cell)))
                    (inspect (get-cliche-struct (cliche-name (rest selected-pair))))))
              :dialog-item-enabled-p nil))

            
|#


;;;_____________________________________
;;;  update-cliches

(defun update-cliches ()
  (let ((window (find-window "Clichs")))
    (when (window-open? window)
      (let* ((selector (view-named :item-selector window))
             (sp (scroll-position  selector)))
        (dolist (cell (selected-cells selector))
          (cell-deselect selector cell))
        (set-table-sequence selector *all-cliches*)
        (scroll-to-cell selector sp)
        (update-selector-dialog window selector)))))


#|
;;;_____________________________________
;;;  specify-frontier-operators

(defun specify-frontier-operators (&key (cliches *all-frontier-operators*)
                                        (title "Frontier Operators")
                                        (message "Select a Frontier Operator")
                                        (view-position :centered)
                                        (view-size #@(400 200)))
  (let ((window (find-window "Frontier Operators")))
    (unless window
      (setf window (make-instance 
                     'selector-dialog
                     :window-title title
                     :window-show nil
                     :view-position view-position
                     :view-size view-size))
      (let ((use-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(80 20) " Use "
              #'(lambda (item) 
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (setf *active-frontier-operators* (pushnew (cell-contents item-selector selected-cell) *active-frontier-operators*))
                      (invalidate-view item-selector))))
              :dialog-item-enabled-p nil))

            (dont-use-button 
             (make-dialog-item
              'button-dialog-item #@(0 0) #@(80 20) " Don't Use "
              #'(lambda (item) 
                  (let* ((item-selector (find-named-sibling item :item-selector))
                         (selected-cell (first (selected-cells item-selector))))
                    (when selected-cell
                      (setf *active-frontier-operators* (delete (cell-contents item-selector selected-cell) *active-frontier-operators*))
                      (invalidate-view item-selector))))
              :dialog-item-enabled-p nil)))

        (add-buttons-and-selector 
         window
         cliches
         :message message
         :buttons (list use-button dont-use-button)
         :enable-only-when-selected (list use-button dont-use-button)
         :print-function #'(lambda (name stream) (format stream "~A ~(~S~)" (if (member name *active-frontier-operators*) "" " ") name))
         :name-function #'(lambda (name) (symbol-name name))
         :selection-type :single)))
    (window-select window)
    window))

|#


;;;_____________________________________
;;;  manipulate-templates

(defun manipulate-templates (&key (templates *example-templates*)
                             (title "Example Templates")
                             (message "Select a Template")
                             (view-position :centered)
                             (view-size #@(400 200)))
  (let ((window (find-window "Example Templates")))
    (unless window
      (setf window (make-instance 
                     'selector-dialog
                     :window-title title
                     :window-show nil
                     :view-position view-position
                     :view-size view-size))
      (let* ((new-button
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " New "
               #'(lambda (item) item 
                  (when (user-monitor-p *user-monitor*)
                    (incf (user-monitor-new-template *user-monitor*)))
                  (text-edit-template nil))))

             (edit-button 
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Edit "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-edit-template *user-monitor*)))
                       (text-edit-template (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             (delete-button
              (make-dialog-item
               'button-dialog-item #@(0 0) #@(60 20) " Delete "
               #'(lambda (item)
                   (let* ((item-selector (find-named-sibling item :item-selector))
                          (selected-cell (first (selected-cells item-selector))))
                     (when selected-cell
                       (when (user-monitor-p *user-monitor*)
                         (incf (user-monitor-delete-template *user-monitor*)))
                       (delete-template (cell-contents item-selector selected-cell)))))
               :dialog-item-enabled-p nil))
             
             (all-buttons (list new-button edit-button delete-button))
             (enable-only-when-selected-buttons (list edit-button delete-button)))
        
        (add-buttons-and-selector 
         window
         templates
         :message message
         :buttons all-buttons
         :enable-only-when-selected enable-only-when-selected-buttons
         :print-function #'(lambda (template stream) (format stream "~S" (example-template-name template)))
         :name-function #'(lambda (template) (symbol-name (example-template-name template))))))
    (window-select window)
    window))

(defun delete-template (template)
  (when (example-template-p template)
    (setf *example-templates* (delete template *example-templates*)))
  (update-templates))


;;;_____________________________________
;;;  update-templates

(defun update-templates ()
  (let ((window (find-window "Example Templates")))
    (when (window-open? window)
      (let* ((selector (view-named :item-selector window))
             (sp (scroll-position  selector)))
        (dolist (cell (selected-cells selector))
          (cell-deselect selector cell))
        (set-table-sequence selector *example-templates*)
        (scroll-to-cell selector sp)
        (update-selector-dialog window selector)
        ))))

(defun get-typein-item-from-user (message
                                  &key (initial-string "")
                                       (ok-text "OK")
                                       (cancel-text "Cancel")
                                       (size #@(365 100))
                                       (position :centered))
  (catch-cancel
    (let ((string (get-string-from-user message
                                        :initial-string initial-string
                                        :position position
                                        :size size
                                        :ok-text ok-text
                                        :cancel-text cancel-text)))
      (multiple-value-bind (value error) (catch-error-quietly (read-from-string string nil nil))
        (cond (error
               (message-dialog (format nil "~%~a is ill-formed." string) :position :centered)
               (get-typein-item-from-user message :initial-string string :cancel-text cancel-text))
              (t
               value))))))


(defun select-item (list &key
                         (window-title "Select an Item")
                         (table-print-function #'princ)
                         (table-name-function (when (symbolp (first list)) #'symbol-name))
                         (selection-type :single)
                         (action-function #'(lambda (list) list))
                         (modeless nil)
                         (message "Select an Item")
                         (ok-text "OK")
                         (cancel-text "Cancel")
                         (type-in-text nil) 
                         (view-position :centered)
                         (view-size #@(400 160)))
  (let* ((window (make-instance 
                   'selector-dialog
                   :window-title window-title
                   :window-show nil
                   :view-position view-position
                   :view-size view-size
                   :close-box-p modeless))

         (ok-button
          (make-dialog-item
           'button-dialog-item #@(0 0) #@(60 20) ok-text
           (if modeless
             #'(lambda (item) item
                (let* ((item-selector (find-named-sibling item :item-selector))
                       (selected-cells (selected-cells item-selector)))
                  (funcall action-function (mapcar #'(lambda (cell) (cell-contents item-selector cell)) selected-cells))))
             #'(lambda (item) item
                (let* ((item-selector (find-named-sibling item :item-selector))
                       (selected-cells (selected-cells item-selector)))
                  (return-from-modal-dialog (funcall action-function (mapcar #'(lambda (cell) (cell-contents item-selector cell)) selected-cells))))))
           :dialog-item-enabled-p nil))

         (type-in-button
          (when type-in-text
            (make-dialog-item
             'button-dialog-item #@(0 0) #@(60 20) type-in-text
             (if modeless
               #'(lambda (item) item
                  (let ((new-item (get-typein-item-from-user message)))
                    (unless (eq new-item :cancel)
                      (funcall action-function (list new-item)))))
               #'(lambda (item) item
                  (let ((new-item (get-typein-item-from-user message)))
                    (unless (eq new-item :cancel)
                       (return-from-modal-dialog (funcall action-function (list new-item)))))))
             :dialog-item-enabled-p t)))

         (cancel-button
          (unless modeless
            (make-dialog-item
             'button-dialog-item #@(0 0) #@(60 20) cancel-text #'(lambda (item) item (return-from-modal-dialog :cancel))
             :dialog-item-enabled-p t)))
         )
        
    (add-buttons-and-selector   
     window
     list
     :message message
     :selection-type selection-type
     :buttons (delete nil (list ok-button type-in-button cancel-button))
     :enable-only-when-selected (list ok-button)
     :default-when-selected ok-button
     :print-function table-print-function
     :name-function table-name-function)
    (cond (modeless
            (window-select window)
            window)
          (t
           (modal-dialog window t)))))


(defun select-a-relation (&optional (r-structs *r-structs*) (message "Select a relation") (title "Select Relation"))
  (first (select-item r-structs
                      :window-title title
                      :selection-type :single
                      :message message
                      :table-print-function #'pretty-print-r-struct-name
                      :table-name-function #'(lambda (r) (symbol-name (r-name r))))))

(defun select-a-rule (&optional (message "Select a rule") (title "Select Rule"))
  (first (select-item (remove-if-not #'rule-p *r-structs*)
                      :window-title title
                      :selection-type :single
                      :message message
                      :table-print-function #'pretty-print-r-struct-name
                      :table-name-function #'(lambda (r) (symbol-name (r-name r))))))

(defun select-a-fact (&optional (message "Select a fact") (title "Select Fact"))
  (first (select-item (remove-if-not #'pred-p *r-structs*)
                      :window-title title
                      :selection-type :single
                      :message message
                      :table-print-function #'pretty-print-r-struct-name
                      :table-name-function #'(lambda (r) (symbol-name (r-name r))))))


(defun select-a-type (&optional (types *all-types*) (message "Select a type") (title "Select Type"))
  (first (select-item types
                      :window-title title
                      :message message
                      :table-print-function #'pretty-print-type-name
                      :table-name-function #'(lambda (x) (symbol-name x)))))


(defun select-a-clause (clauses message title)
  (first (select-item clauses
                      :window-title title
                      :selection-type :single
                      :message message
                      :table-print-function #'(lambda (c s) 
                                                (let ((body (rest c)))
                                                  (if (and (null (rest body)) (symbolp (first body)))
                                                    (format s "~a" (first body))
                                                    (format s "(~a)" (clause-body-string (cdr c))))))
                      :table-name-function nil)))

(defun select-a-literal (literals message title)
  (first (select-item literals
                      :window-title title
                      :selection-type :single
                      :message message
                      :table-print-function #'(lambda (l s)
                                                (if (symbolp l)
                                                  (format s "~a" l)
                                                  (format s "~a" (clause-body-string (list l)))))
                      :table-name-function nil)))
