;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.   

(in-package :user)

(defvar *clean-tuples* nil)

;;;_______________________________________
;;;  CLEAN-TUPLES
;;;
;;;  Removes duplicated tuples and tuples containing variables

(defun clean-tuples (tuples)
  (if *clean-tuples*
    (delete-if #'(lambda (tuple) (some #'variable-p tuple)) (remove-duplicates tuples))
    tuples))

;;;_______________________________________
;;;  DUMP-PRED

(defun dump-pred (pred stream &optional (loud nil) (no-pos-or-neg nil) (reduction-pred? (reduction-pred-p pred)))
  (when loud (format t "  ~s~%" (r-name pred)))
  (format stream "~%(def~A-pred ~s~%" (if reduction-pred? "-reduction" "") (r-name pred) (if no-pos-or-neg ";;; incomplete" "" ) )
  (format stream "  :vars  (~{~s~^ ~})~%" (r-vars pred))
  (format stream "  :type  (~{~s~^ ~})~%" (r-type pred))
  (format stream "  :mode  (~{~s~^ ~})~%" (r-mode pred))
  (if no-pos-or-neg (format stream "#|~%"))
  (let* ((element-length (length (format nil "~S" (or (first (r-pos pred)) (first (r-neg pred))))))
         (elements-per-line (max 1 (floor 70 element-length)))
         (pos-format (format nil "  :pos ( ~~{~~~A@{~~s ~~}~~^~~%         ~~})~~%" elements-per-line))
         (neg-format (format nil "  :neg ( ~~{~~~A@{~~s ~~}~~^~~%         ~~})~~%" elements-per-line)))
    (format stream pos-format  (clean-tuples (r-pos pred)))
    (if (r-compute-neg pred)
      (format stream "  :neg   :computed~%")
      (format stream neg-format (clean-tuples (r-neg pred)))))
  (if no-pos-or-neg (format stream "|#~%"))
  (when reduction-pred?
    (format stream "  :reduction-type             ~s~%" (reduction-pred-reduction-type pred))
    (format stream "  :reduction-variabilization  ~s~%" (reduction-pred-reduction-variabilization pred)))
  (format stream "  :induction    ~s~%" (r-induction pred))
  (format stream "  :constraint   ~s~%" (r-constraint pred))
  (format stream "  :commutative  ~s~%" (r-commutative pred))
  (when (r-determinacy pred) (format stream "  :determinacy  ~s~%" (r-determinacy pred)))
  (when (r-questions pred) (format stream "  :questions  ( ~{~s~^~%                ~} )~%" (r-questions pred)))
  ;(when (r-try-constants pred) (format stream "  :try-constants ~s~%" (r-try-constants pred)))
  ;(when (r-sort-fn-def  pred) (format stream "  :sort-fn       ~s~%" (r-sort-fn-def pred)))
  (format stream "   )~%" ))

;;;_______________________________________
;;;  DUMP-PREDS

(defun dump-preds (preds stream &optional (loud nil) (short-preds))
  (when preds
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Predicates - Extensionally Defined Relations~%")
    (when loud (format t "~%Preds _____________~%"))
    (dolist (pair preds)
     (dump-pred (rest pair) stream loud (member (rest pair) short-preds)))))


;;;_______________________________________
;;;  DUMP-BUILTIN

(defun dump-builtin (builtin stream &optional (loud nil))
  (when loud (format t "  ~s~%" (r-name builtin)))
  (format stream "~%(def-builtin ~s~%" (r-name builtin))
  (format stream "  ~s~%" (r-function-def builtin))
  (when (r-vars builtin) (format stream "  :vars  (~{~s~^ ~})~%" (r-vars builtin)))
  (format stream "  :type  (~{~s~^ ~})~%" (r-type builtin))
  (format stream "  :mode  (~{~s~^ ~})~%" (r-mode builtin))
  (format stream "  :induction     ~s~%" (r-induction builtin))
  (format stream "  :constraint    ~s~%" (r-constraint builtin))
  (format stream "  :commutative   ~s~%" (r-commutative builtin))
  (when (r-determinacy builtin) (format stream "  :determinacy  ~s~%" (r-determinacy builtin)))
  (when (r-questions builtin) (format stream "  :questions    ~s~%" (r-questions builtin)))
  (format stream "  :try-constants ~s~%" (r-try-constants builtin))
  (when (r-sort-fn-def builtin) (format stream "  :sort-fn       ~s~%" (r-sort-fn-def builtin)))
  (format stream "  :create-prolog-function   T~%")
  (format stream "   )~%" ))

;;;_______________________________________
;;;  DUMP-BUILTINS

(defun dump-builtins (builtins stream &optional (loud nil))
  (when builtins
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Built-Ins - Functionally Defined Relations~%")
    (when loud (format t "~%Built-Ins _____________~%"))
    (dolist (pair builtins)
      (let ((builtin (rest pair)))
        (unless (member builtin *special-r-structs*)
          (dump-builtin builtin stream loud))))
    (dump-builtin-flags builtins stream)))
  
;;;_______________________________________
;;;  DUMP-BUILTIN-FLAGS

(defun dump-builtin-flags (builtins stream)
  (let ((predefined-builtins (mapcan #'(lambda (pair) (if (member (rest pair) *special-r-structs*) (list (rest pair)))) builtins)))
    (format stream "~%(set-builtin-flags '~S" (mapcar #'r-name predefined-builtins))
    (format stream "~%      '~S    ;; induction" (mapcar #'r-induction predefined-builtins))
    (format stream "~%      '~S )  ;; try-constants~%" (mapcar #'r-try-constants predefined-builtins))))

;;;_______________________________________
;;;  DUMP-ARITHMETIC-OP

(defun dump-arithmetic-op (arithmetic-op stream &optional (loud nil))
  (when loud (format t "  ~s~%" (r-name arithmetic-op)))
  (format stream "~%(def-arithmetic-op ~s~%" (r-name arithmetic-op))
  (format stream "  ~s~%" (r-function-def arithmetic-op))
  (when (r-vars arithmetic-op) (format stream "  :vars  (~{~s~^ ~})~%" (r-vars arithmetic-op)))
  (format stream "  :type  (~{~s~^ ~})~%" (r-type arithmetic-op))
  (format stream "  :mode  (~{~s~^ ~})~%" (r-mode arithmetic-op))
  (format stream "  :induction     ~s~%" (r-induction arithmetic-op))
  (format stream "  :constraint    ~s~%" (r-constraint arithmetic-op))
  (format stream "  :commutative   ~s~%" (r-commutative arithmetic-op))
  (when (r-determinacy arithmetic-op) (format stream "  :determinacy  ~s~%" (r-determinacy arithmetic-op)))
  (when (r-questions arithmetic-op) (format stream "  :questions    ~s~%" (r-questions arithmetic-op)))
  (format stream "  :try-constants ~s~%" (r-try-constants arithmetic-op))
  (when (r-sort-fn-def arithmetic-op) (format stream "  :sort-fn       ~s~%" (r-sort-fn-def arithmetic-op)))
 (format stream "   )~%" ))

;;;_______________________________________
;;;  DUMP-ARITHMETIC-OPS

(defun dump-arithmetic-ops (arithmetic-ops stream &optional (loud nil))
  (when arithmetic-ops
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Arithmetic-Operators~%")
    (when loud (format t "~%Arithmetic-Operators _____________~%"))
    (dolist (pair arithmetic-ops)
      (dump-arithmetic-op (rest pair) stream loud))))

;;;_______________________________________
;;;  DUMP-RULE

(defun dump-rule (rule stream  &optional (loud nil))
  (when loud (format t "  ~s~%" (r-name rule)))
  (format stream "~%(def-rule ~s~%" (r-name rule))
  (format stream "  :vars  (~{~s~^ ~})~%" (r-vars rule))
  (format stream "  :type  (~{~s~^ ~})~%" (r-type rule))
  (format stream "  :mode  (~{~s~^ ~})~%" (r-mode rule))
  (format stream "  :clauses~%     ( ~{~s~^~%       ~}~%     )~%" (or (get-clauses (r-name rule)) (r-clauses rule)))
  (format stream "  :induction    ~s~%" (r-induction rule))
  (format stream "  :constraint   ~s~%" (r-constraint rule))
  (format stream "  :commutative  ~s~%" (r-commutative rule))
  (when (r-determinacy rule) (format stream "  :determinacy  ~s~%" (r-determinacy rule)))
  (when (r-questions rule) (format stream "  :questions    ~s~%" (r-questions rule)))
  ;(when (r-try-constants rule) (format stream "  :try-constants ~s~%" (r-try-constants rule)))
  ;(when (r-sort-fn-def  rule) (format stream "  :sort-fn       ~s~%" (r-sort-fn-def rule)))
  (format stream "  )~%" ))

;;;_______________________________________
;;;  DUMP-RULES

(defun dump-rules (rules stream &optional (loud nil))
  (when rules
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Rules - Intensionally Defined Relations~%")
    (when loud (format t "~%Rules _____________~%"))
    (dolist (pair rules)
      (dump-rule (rest pair) stream loud))))

;;;_______________________________________
;;;  DUMP-TYPE

(defun dump-type (type stream &optional (loud nil))
  (let ((properties (remove-if-not #'(lambda(p) (get type p)) *type-properties*)))
    (when loud (format t "  ~a~%" type))
    (format stream "(def-type ~s~{~%   ~(~s~)~} )~%~%"
            (if properties (cons type properties) type)
            (get-type-instances type))))

;;;_______________________________________
;;;  DUMP-TYPES

(defun dump-types (types stream &optional (loud nil))
  (when types
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Types~%~%")
    (when loud (format t "~%Types _____________~%"))
    (dolist (type types)
      (dump-type type stream loud))))

;;;_______________________________________
;;;  DUMP-CLICHE

(defun dump-cliche (cliche stream  &optional (loud nil))
  (when loud (format t "  ~s~%" (cliche-name cliche)))
  (format stream "~%(def-cliche ~s~%" (cliche-name cliche))
  (format stream "  :pred-restrictions  (~{~s~^~%                       ~})~%" (cliche-pred-restrictions cliche))
  (format stream "  :var-restrictions   (~{~s~^~%                       ~})~%" (cliche-var-restrictions cliche))
  (format stream "  :cache?  ~s~%" (cliche-cache? cliche))
  (format stream "  )~%" ))

;;;_______________________________________
;;;  DUMP-CLICHES

(defun dump-cliches (cliches stream &optional (loud nil))
  (when cliches
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Cliches~%")
    (when loud (format t "~%Cliches _____________~%"))
    (dolist (pair cliches)
      (dump-cliche (rest pair) stream loud))))

;;;_______________________________________
;;;  DUMP-PROBLEMS

(defun dump-problems (problems current-problem stream &optional (loud nil))
  (declare (ignore loud))
  (when problems
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Learning Problem Specification and Parameter Settings~%")
    (dolist (problem problems)
      (unless (equal problem current-problem)
        (format stream "~%(def-focl-problem ~s~%~{  ~s ~s~%~}  )~%" (first problem) (rest problem))))
    (when current-problem
      (format stream "~%(def-focl-problem ~s~%~{  ~s ~s~%~}  )~%" (first current-problem) (rest current-problem)))))

;;;_______________________________________
;;;  DUMP-USER-MONITOR

(defun dump-user-monitor (user-monitor stream)
  (when (user-monitor-p user-monitor)
    (let ((file-names (push (format nil "~A" *kb-file*) (user-monitor-file-names user-monitor)))
          (times (push (list *load-time* (get-time)) (user-monitor-times user-monitor))))
      (setf (user-monitor-file-names user-monitor) nil
            (user-monitor-times user-monitor) nil)
      (let ((ums (format nil "~S" user-monitor)))
        (format stream "~%~%(set-user-monitor (make-user-monitor ~A :file-names '~S :times '~S))"
                (subseq ums 16 (- (length ums) 27)) file-names times))
      (setf (user-monitor-file-names user-monitor) file-names
            (user-monitor-times user-monitor) times))))

;;;_______________________________________
;;;  DUMP-KB

(defun dump-kb (stream &optional (loud nil) (example-templates nil))
  (format stream "~%;;;======================================================")
  (format stream "~%;;; FOCL Knowledge Base  (~A Format)~%" (if example-templates "Example"  "Relation"))
  (format stream "~%; (reset-relations)~%")
  (dump-types (nconc (reverse *all-types*) (mapcar #'car *domain*)) stream loud)
  (dump-preds (reverse *extensional-preds*) stream loud
              (remove-duplicates (mapcan #'(lambda (template)
                                             (cons (get-pred (example-template-name template))
                                                   (mapcar #'(lambda (fact) (get-pred (first fact))) (example-template-facts template))))
                                         example-templates)))
  (when example-templates (dump-examples example-templates stream loud))
  (dump-rules (reverse *intensional-preds*) stream loud)
  (dump-arithmetic-ops (reverse *arithmetic-ops*) stream loud)
  (dump-builtins (reverse *builtin-preds*) stream loud)
  (dump-cliches (reverse *all-cliches*) stream loud)
  (when *example-templates* (dump-example-templates *example-templates* stream loud))
  (dump-problems *focl-problems* *focl-problem* stream loud)
  (dump-user-monitor *user-monitor* stream)
  (setf *facts-changed* nil
        *rules-changed* nil)
  (values))

;;;_______________________________________
;;;  DUMP-EXAMPLE

(defun dump-example (example template stream
                             &optional
                             (loud nil)
                             (name (example-template-name template))
                             (pred (get-pred name))
                             (negative? (member example (r-neg pred) :test #'equal)))
  (when loud (format t "  ~:[(~s ~{~s~^ ~})~;(not (~s ~{~s~^ ~}))~]~%" negative? name example))
  (format stream "~%(def-example ~:[~s~;(not ~s)~] ~s~%" negative? name example)
  (let* ((facts (example-template-facts template))
         (vars (example-template-vars template))
         (defining-facts (return-object-var-defining-facts facts vars))
         (mapping (direct-mapping vars example)))
    (dolist (fact (nreverse (return-example-facts mapping facts defining-facts)))
      (format stream "   ~s~%" fact))
    (format stream "   )~%" )))

;;;_______________________________________
;;;  DUMP-EXAMPLES

(defun dump-examples (templates stream &optional (loud nil))
  (when templates
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Examples - data that will complete some predicates~%")
    (when loud (format t "~%Examples _____________~%"))
    (dolist (template templates)
      (when (example-template-p template)
        (let* ((pred-name (example-template-name template))
               (pred (get-pred pred-name)))
          (dolist (p (remove-duplicates (r-pos pred) :test #'equal))    ;;; duplicate removal
            (dump-example p template stream loud pred-name pred nil))
          (dolist (n (remove-duplicates (r-neg pred) :test #'equal))    ;;; duplicate removal
            (dump-example n template stream loud pred-name pred t)))))))

;;;_______________________________________
;;;  DUMP-EXAMPLE-TEMPLATE

(defun dump-example-template (template stream &optional (loud nil))
  (declare (ignore loud))
  (when (example-template-p template)
    (format stream "~%(def-example-template ~s (~{~s~^ ~})~%" (example-template-name template) (example-template-vars template))
    (format stream "~{  (~{~s~^ ~})~%~}" (example-template-facts template))
    (format stream "  )~%")))

;;;_______________________________________
;;;  DUMP-EXAMPLE-TEMPLATES

(defun dump-example-templates (templates stream &optional (loud nil))
  (when templates
    (format stream "~%;;;______________________________________________________")
    (format stream "~%;;; Example Templates~%")
    (dolist (template templates)
      (dump-example-template template stream loud))))

;;;_______________________________________
;;;  DUMP-KB-TO-FILE

(defun dump-kb-to-file (&optional (kb-file *kb-file*) (example-templates nil))
  (unless kb-file
    (setf kb-file (choose-new-file-name)))
  (unless (eq kb-file :cancel)
    (setf *kb-file* kb-file)
    (with-open-file (stream kb-file :direction :output :if-exists :supersede)
      (dump-kb stream t example-templates)
      (format t "Knowledge Base saved in ~s" kb-file))))

;;;_______________________________________
;;;  DUMP-KB-TO-FILE-AS

(defun dump-kb-to-file-as (&optional (kb-file *kb-file*) (example-templates nil))
  (declare (ignore kb-file))
  (setf *kb-file* nil)
  (dump-kb-to-file *kb-file* example-templates))

;;;_______________________________________
;;;  CHOOSE-NEW-FILE-NAME

(defun choose-new-file-name ()
#-:ccl-2  "new-kb-file"
#+:ccl-2 (catch-cancel (choose-new-file-dialog :button-string "Save KB")))




