;*------------------------------------------------------------------03/Jun/89-*
;* 
;*   ATTEND2.LSP:   A version with a different "Halt-Test". We are now looking 
;*       at two worst-case scenarios: one is as before -- adding the new
;*       instance into the second best class. The other supposes that the 
;*       new instance should really be a disjunct. See Halt-Test for more.
;*
;*    ALSO: each node now has two lists of attributes stored -- the atts 
;*       associated with that concept description, and the summary over the
;*       the children of that node. (I no longer do on-the-fly summaries.)
;*---------------------------------------------------------------------JHG----*

;*------------------------------------------------------------------08/Mar/89-*
;* 
;*                  Classit/2, or Classweb + attention
;*                           John Gennari
;* 
;*      This is code built ontop of Classweb: see below for documentation 
;*  and history. 
;*      The principle modification is the ability to look at selected 
;*  attributes: an attention mechanism. This entails modification of the 
;*  main loop, so that each attribute is chosen sequentially, and the 
;*  stopping condition is tested for. Also, the low level stuff where an
;*  obj is integrated into a concept must be modified (see node-atts.lsp).
;*  
;*  Currently (3/89) attributes are not inspected incrementally. This is
;*  for ease in implementation: I believe it is perfectly possible to build
;*  a more efficient, but more complex, incremental version.
;*      Finally, of course, there are stylistic differences. These are my 
;*  comments, and there may be more use of the loop macro.
;*  
;*     WARNING: This file must be used with some top-level calling file,
;*  such as "maintst.lsp" or "mult-test.lsp". Here, there is only code for
;*  classifying an instance. ("run" and "mainloop" are defined only in
;*  maintst.lsp.)
;*---------------------------------------------------------------------JHG----*

#|*****************************************************************************

IMPLEMENTATION OF CLASSIFY (COBWEB/CLASSIT); 
Patrick Young, and Kevin Thompson 5-9-88.
Plus a few additions by John Allen, 2-89.

History:
5/13  added missing attributes information.
6/3   salience on missing attributes.                                
6/3   extra value for outfile "t" to default name.                   
6/4   print in outfile some version, what the infile was, and time.  
6/4   return tree again.                                             
6/15  changed so that missing attributes (e.g. those with acount=0) aren't
      used, so I average by only those that have values. prob-att=value.
6/15  param for how nodes are printed.                               
6/15  ability to name each attribute yourself.                       
6/15  add ability for comments in input files.                       

2/14/89 Added :TREE to keyword list of run.  Allow the user to pass in a 
        pre-built tree.  JAA.
2/14/89 Added :learning to keyword list of run.  Allows you to specify learning
        or testing modes. Note that testing mode returns a list of lists, 
        each list containing the name of the test object, the node it matches,
        and a list of all the predicted values specified by *PRED-ATTS* JAA.
2/15/89 Added *SPLIT* and *MERGE* to control the availability of the split
        and merge operators.  JAA.
*****************************************************************************|#

(provide "attend2")

(require "struct")
(require "node-compute")
(require "node-atts")
(require "read-inst")
(require "get-time")
(require "globals")

(require "loop" "/ci/ci/worldm/bin/loop")
(use-package "LOOP")


; (proclaim '(optimize (safety 0) (speed 3)))


#|*****************************************************************************

  T O P   L E V E L   P R O C E D U R E   A N D   U S E R  I N T E R F A C E

*****************************************************************************|#


#|----------------------------------------------------------------15/Jun/88----
   Function  - USAGE
    Prints out a message describing the current usage of the function run.
-------------------------------------------------------------------KThompso--|#

(defun usage ()
  (mapc #'(lambda (str) (format t (concatenate 'string "--- " str "~%")))
  '(
  "USAGE: run <infile> <num-atts>"
  "and optional keyword arguments:"
  "OUTFILE: the name of an outfile for the results"
  "            (defaults to standard-io), giving t as argument will create"
  "            <infile>.out, e.g. infile soybean.data --> soybean.data.out"
  "PRINT-EACH: whether to print out the tree in between each iteration"
  "            (defaults to nil)"
  "PRINT-FUNCTION: how to print out nodes; can be :short or :long"
  "            (defaults to :short)"
  "ATT-TYPE: whether the attributes are :nominal or :numeric or :mixed"
  "            (defaults to :nominal)"
  "          if set to :mixed, infile is expected to have on first line of"
  "          data a line of n or s, for numeric and symbolic, for each att"
  " "
  "ATT-NAMES: list of att-names (length num-atts) given.  Must not have"
  "           duplicates.  Names of form (att1 att2 ...) generated"
  "           automatically otherwise."
  "BREAKER: whether to stop inside the main loop of run to inspect the tree"
  "            (defaults to nil)"
  "DEBUG: whether to print out debug statements about the program's innards"
  "            (defaults to nil)"
  ""
  "TREE: Allows you to pass in a pre-built tree. (defaults to nil)"
  ""
  "LEARNING: Allows user to specify learning (t) or testing (nil) mode."
  "          (defaults to t)"
  ""
  "PRED-ATTS: List of attribute to be predicted during testing mode.  Every"
  "           element must be found in ATT-NAMES (defaults to *PRED-ATTS*)"
  ""
  "ACUITY:  Sets the Acuity parameter (defaults to 1.0)"
  ""
  "THRESHOLD:  Sets the Threshold parameter (defaults to 1.0, meaning"
  "           no cutoff at all)"
  ""
  "MIN-CHANCE:  Sets the Min-Chance parameter (defaults to 0.010)"
  "         This is the base probability that an attribute will be inspected"
  ""
  "Note that each line the infile is expected to be in the form"
  "      A1 A2 A3 A4 A5 . . . An cat-info"
  "      where A1..An are attribute values, and cat-info is a label"
  "   Lines that begin with a % are ignored."
  ))
  (values))


#|----------------------------------------------------------------04/Jun/88----
   Function  - PRINT-OUT-BANNER
       
   Inputs    -> infile, outfile
      prints out banner to *OUTPUT-STREAM*
-------------------------------------------------------------------KThompso--|#

(defun print-out-banner (infile outfile)
  (pr-out "CLASSIFY running at ~A~%INFILE:  ~A" (get-time) infile)
  (if outfile 
     (pr-out "~%OUTFILE: ~A~3%" outfile)
     (pr-out "~3%"))
  (values))


#|*****************************************************************************

        C O N T R O L   S T R U C T U R E   F O R   C L A S S W E B

  Algorithm:
  The action performed depends on the tree handed to store in.

  (1) The tree handed to classify is a single node with no children.
      In this case, classify will generalize the current node and give
      it two children, the original node and the new obj
  (2) The tree has children, in which case, we consider the category utility
      of
         (i)   place the obj in a new child node,
         (ii)  place the obj in one of the child nodes,
         (iii) merge the best two children in (ii) and place the obj
               with them, or
         (iv)  consider splitting the children of the best node to
               the current level, placing the obj with the best of
               the children.
 
  Unless the obj is placed in a new child node, the algorithm is
  called recursively.
 
  As "classfy" is called recursively, the root given will already
  have the new instance integrated. However, if the new root has no
  children, then root will not have obj incorporated (see ModifyTree and
  fork-node). This avoids having to "subtract" an instance from a concept.
  

*****************************************************************************|#

;*------------------------------------------------------------------13/Mar/89-*
;*  Function  - CLASSIFY & CLASSRATT
;* 
;*  Inputs    -> root, the list of attribute names, (implicitly, the 
;*      instance, obj), and the level.
;* 
;*  Returns   -> Either the new tree, or a list of predicted attributes,
;*      depending on the *LEARNING* switch.
;*
;*  Actions   -> Here lies the basic control structure for Classit/2,
;*      particularly the attention mechanism. 
;*      
;*      These two functions represent the double (recursive) loop over levels
;*      in the tree, "classify", and over attributes in the att-names list,
;*      "classRatt". As the hierarchy is descended, classify recurses with
;*      the "level" parameter. (The recursive call is inside "ModifyTree".)
;*      Additionally, for a given level, there is recursion over the attribute
;*      list, as "choose-att" successively selects the best attribute. 
;*        The latter recursion (in "classRatt") halts if the score returned
;*      by Halt-test is good enough, while the level recusion halts when a 
;*      new disjunct is created, or when the root has no children.
;*
;*  NOTE: for this (3/89) non-incremental version, classRatts needs to 
;*     build up the list of known atts (k-atts) as well as remove atts from
;*     the list of unknown atts. If the alg were incremental, all that
;*     Class-One would need is bestAtt.
;*  MODIFIED 5/89 to add threshold.
;*  MODIFIED 5/89 to add efficiency scoring ("update-costs").
;*  MODIFIED 8/89 to add *ATTENTION* switch.
;*---------------------------------------------------------------------JHG----*

(defun classify (root att-names level)
  
  (if *GLOBAL-DEBUG* (pr-out "== At level ~D~%" level))
  (cond
   ((null (node-children root))   ; root shouldn't have 'obj incorporated!
    (if *LEARNING*                ; and attnames is the old list!
       (fork-node root att-names)
       (predicted-values root))
   )
   ((> (node-attscores root) *THRESHOLD*)
       (if *GLOBAL-DEBUG* (pr-out "        choosing not to continue..."))
       (if *LEARNING*
 	 root
 	(predicted-values root))
   )
   (*ATTENTION*
    (classRatt root att-names nil level))
   ( t
    (class-allatt root att-names level))     ; Just like Classweb
  )
)

;*------------------------------------------------------------------May/89-*
;*  Function  - CLASSRATT
;* 
;*  Inputs    -> the root, lists of unknown and known attributes, and the 
;*               level.
;*  Actions   -> This is the recursive function through the list of observed
;*      attributes. Here is where "choose-att" (selects the next attribute)
;*      and "Halt-test" are called from. Also, see documentation for 
;*      "Classify", above. 
;*      This function has gotten a little messy because of pr-out messages,
;*      and for prediction of missing atts (no learning). The inheritance
;*      test means that if you are predicted attribute x, and the chosen
;*      child has no information about attribute x, then you stop proceeding
;*      down the tree and make your prediction based on that parent. This
;*      logic breaks down in a number of ways (predicting more than one 
;*      attribute? what if x re-appears in the grand-children?), but it's the
;*      best I currently have. 
;*---------------------------------------------------------------------JHG----*

(defun classRatt (root unkn-atts known-atts level)

   (let* ((bestAtt (choose-att unkn-atts root))
	  (k-atts (cons bestAtt known-atts))
	  (new-alist (if bestAtt (remove bestAtt unkn-atts)
		                 nil ))
	   score newnode newparent oldnode second action)

     (if *GLOBAL-DEBUG* 
	 (pr-out "considering attribute ~A of list ~A~%" bestAtt k-atts))
     (multiple-value-setq 
	    (score newparent newnode oldnode second action)
	    (Class-One root k-atts))

     (cond ((or (null new-alist)            ;* No attributes left....
                (Halt-test root oldnode second new-alist action))
            (if (null new-alist)
		     (pr-out "No attributes left..~%")
		     (pr-out "Ignoring remaining attributes..~%"))
	    (cond (*LEARNING*
	            (ModifyTree newparent newnode oldnode level k-atts))

      ;* Learning is OFF:
		  ( (or 
		     (eq 1 (node-count newnode))  ; I forget, why this?
		     (null (find-att (car *pred-atts*) (node-atts newnode)))
;* the "inheritance" test....
		    )
		    (if (< 1 (length *pred-atts*))
			(break "don't know what to do if many pred-atts!"))
		    (update-costs :attr (length k-atts)
				  :nodes (length (node-children root)))
	            (predicted-values root))
		  ( t 
		    (update-costs :attr (length k-atts)
				  :nodes (length (node-children root)))
	            (classify newnode *ATT-NAMES* (1+ level))))
	    )
	   (  t                     ;* Need to look at the next attribute.
	     (classRatt root new-alist k-atts level))
     )
))

;*------------------------------------------------------------------10/Aug/89-*
;*  Function  - CLASS-ALLATT
;* 
;*  Inputs    -> The parent node, a list of all attributes, and the level.
;* 
;*  Actions   -> This implements the no-attention version of Classit-2
;*       Here, Class-One is called immediately with all the attributes.
;*---------------------------------------------------------------------JHG----*

(defun class-allatt (parent att-names level)

  (multiple-value-bind 
	(score newparent newnode oldnode)
	(Class-One parent att-names)
  (cond (*LEARNING*
	    (ModifyTree newparent newnode oldnode level att-names))
	( t
	    (update-costs :attr (length att-names)
			  :nodes (length (node-children parent)))
	    (classify newnode att-names (1+ level)))
  )

))

;*------------------------------------------------------------------16/Apr/89-*
;*  Function  - CHOOSE-ATT
;* 
;*  Inputs    -> The list of unseen attribute names, and the parent node.
;*        (*MIN-CHANCE* is the minimum chance for looking at a new attribute.)
;* 
;*  Returns   -> The next attribute to inspect. This is based on the 
;*     "childscore" values in the parent node. Each attribute has a chance
;*     of being chosen proportional to their scores.
;*     NOTE: exactly how this is done can be tweaked. One can use a linear
;*     proportion, a quadratic (by squaring the score), or even an exponential
;*     proportion.
;* MODIFIED 6/25/89
;*     The above work now occurs in "C-att". Choose-att is now an outer loop
;*     that picks another attribute if the chosen attribute is unknown. 
;*     There's no sense trying to classify something with an attribute that
;*     has no value!
;*---------------------------------------------------------------------JHG----*

(defun choose-att (att-list parent)

  (if (null att-list) nil ;else...

  (let* ((attribute (C-att att-list parent)))
      (if (eq (get 'obj attribute) '?)
			;RECURSE, we've choosen an unknown att.
          (choose-att (remove attribute att-list) parent) 
	;Else:
	  attribute)
  )
))


(defun C-att (att-list parent)

   (loop for att-name in att-list
	 for att = (find-att att-name (node-childatts parent))
	 collect (if att
		     (basic-att-childscore att)
		      *MIN-CHANCE*)
	 into Scorelist

	 finally
	    (if *GLOBAL-DEBUG* 
		(progn
		   (pr-out "   CHOOSE-ATT: attribute list was ~A~%" att-list)
		   (pr-out "   score list was (")
		   (loop for score in Scorelist do
		         (pr-out "~6,3F" score))
		   (pr-out ")~%"))
	    )
	    (let* ((sum (apply '+ Scorelist))
		   (rand-val (if (equal sum 0.0) 
				  -0.1 
				 (random (float sum))))
		   (totalscore 0.0))
               (return
	          (loop for score in Scorelist
		     for att-name in att-list do
		     (setq totalscore (+ totalscore score))
		     (if (> totalscore rand-val) (return att-name))
	       ))
	    )
))

;*------------------------------------------------------------------04/Apr/89-*
;*  Function  - HALT-TEST
;* 
;*  Inputs    -> root, the oldnode that was chosen by compute-cu-best,
;*      the second best node (child), lists of known and unknown attributes,
;*      and action, a flag says which operator we just tried.
;* 
;*  Returns   -> T or nil, indicating whether or not to stop inspecting
;*      unseen attributes. (T means stop looking.)
;*
;*  Actions   -> This compares three different scores in two steps:
;*      First, OBJ is modified ("change-OBJ") so that the unknown attributes
;*      are set to the second best node (child). 
;*      Then we imagine if this instance (with all attributes known) were
;*      added into the best node ("compute-tentative") vs. the second best
;*      node (the call to determine-CU).
;*      If this comparison says we can ignore the remaining attributes, 
;*      then we make an additional test -- we modify OBJ again (inside of 
;*      "compute-disjunct"), and imagine if the unknown atts. were to 
;*      indicate a disjunct. Again, this OBJ must be compared to the current
;*      best know (another call to "compute-tentative"). 
;*      Finally, the attributes must be restored.
;*---------------------------------------------------------------------JHG----*

(defun Halt-test (root oldnode child unkn-atts action)

  (let* ((unseen (change-OBJ unkn-atts child))
	 (pos (position child (node-children root)))
	 (testatts (intersect child oldnode))
	 
	  Decision
	)
       (setq Decision
	 (and
	      testatts   ;* if no intersection, give up...
	      (> (add-to-best action oldnode root testatts)
		 (determine-cat-utility
		        (new-partition root child pos testatts) testatts))
	      (or (eq action 'DISJUNCT)
		(progn
	          (setq testatts (loop for att in (node-atts oldnode) 
				   collect (basic-att-name att)))
	          (< (compute-disjunct unkn-atts root oldnode testatts)
		     (add-to-best action oldnode root testatts))))
         ))

; Restore attributes:
      (loop for attr in unkn-atts
	    for old-value in unseen
	    do
	      (putprop 'OBJ old-value attr)
      )

      Decision
))

;*------------------------------------------------------------------03/Jun/89-*
;*  Function  - COMPUTE-DISJUNCT
;* 
;*  Inputs    -> The old action taken, the unknown attributes, the parent
;*               and a list of all the attributes.
;* 
;*  Returns   -> The score associated if the worst-case scenario occurs
;*       and the instance should really be a disjunct. To do this, we modify
;*       the attribute values associated with OBJ. Used only by Halt-test.
;*
;*  Actions   -> ATTN:  This modifies the values associated with 'OBJ.
;*---------------------------------------------------------------------JHG----*

(defun compute-disjunct (unkn-atts parent child allatts)

  (loop for attr in unkn-atts
	with tmp
	for parentatt = (find-att attr (node-childatts parent))
	for childatt = (find-att attr (node-atts child))
	do
	  (putprop 'OBJ (worst-val parentatt (best-val childatt)) attr)
	finally
	  (setq tmp (compute-CU-self parent allatts))
	  (if *GLOBAL-DEBUG*
	      (pr-out "disjunct -- ~5,3F (score for continuing)~%" tmp)
	      (pr-out "D "))
	  (return tmp)
  )
)

;*------------------------------------------------------------------03/Jun/89-*
;*  Function  - ADD-TO-BEST
;* 
;*  Inputs    -> The old action, the "best" node without the instance, 
;*               the root, and a list of all the attributes. ALSO OBJ.
;* 
;*  Returns   -> The score if the new, modified OBJ (with all attributes
;*       filled in) were added to the old best node. Used only by Halt-test.
;*   NOTE: the commented out stuff is what might be done for merge and split.
;*   currently, it forces a nil return, meaning that if we want one of these
;*   operators, we have to look at all attributes.
;*---------------------------------------------------------------------JHG----*

(defun add-to-best (action best root allatts)

(let ((score) (pos))
  (cond ((eq action 'BEST)
	 (setq pos (position best (node-children root)))
	 (setq score
	       (determine-cat-utility
		  (new-partition root best pos allatts) allatts)))
	((eq action 'DISJUNCT)
	 (setq score (compute-CU-self root allatts)))
	((eq action 'MERGE)
	 (setq score -999))
;		   (compute-CU-merge root (integrate-obj best allatts)
;						child best allatts)
	((eq action 'SPLIT)
	 (setq score -999))
;		   (compute-CU-split root (integrate-obj best allatts)
;				     best allatts)
  )
  (if *GLOBAL-DEBUG*
       (pr-out "            ~5,3F (score for halting)~%" score))

  score

))

;*------------------------------------------------------------------03/Jun/89-*
;*  Function  - CHANGE-OBJ
;* 
;*  Inputs    -> the list of unknown atts and the second-best node.
;* 
;*  Returns   -> a list of the actual `unseen' attribute values (yes, this
;*           is cheating).
;*  Actions   -> Changes the attribute values so that they match the
;*           mean values of the second best concept. (See best-val in 
;*           node-atts). Used only by Halt-test.
;*     NOTE: if the second best node ("child") does not have "attr", (childatt
;*           is null), then "best-val" will return '?.
;*---------------------------------------------------------------------JHG----*

(defun change-OBJ (unkn-atts child)

(loop for attr in unkn-atts
      for childatt = (find-att attr (node-atts child))
          collect (get 'OBJ attr) into old-values 
	  do
	    (putprop 'OBJ (best-val childatt) attr)
    finally 
      (return old-values)
))

;*------------------------------------------------------------------16/Jun/89-*
;*  Function  - INTERSECT
;* 
;*  Inputs    -> two att lists
;* 
;*  Returns   -> a list of attribute *names*, the intersection of
;*      the two lists. Used by HALT-TEST. L2 is -999 when action is DISJUNCT.
;*---------------------------------------------------------------------JHG----*

(defun intersect (L1 L2)

  (if (eq L2 '-999)
     (loop for attr in (node-atts L1) collect (basic-att-name attr))
  ;ELSE:
     (intersection 
       (loop for attr in (node-atts L1) collect (basic-att-name attr))
       (loop for attr in (node-atts L2) collect (basic-att-name attr)))
))

#|----------------------------------------------------------------04/May/88----
Function  - CLASS-ONE             (documented and modified 3/14/89, -JHG)
				 (This used to be "store")

Inputs    -> root, the parent node, the best attribute, (or a list of 
   the known atts -- this is needed for the non-incremental version).

Returns   -> Five values: the score, the newparent node, the new (modified)
  child node, the old (unmodified) "best" child, and the second best 
  child. 
  If learning is off,the tree is not modified, and a set of predictions
  are returned.

Actions -> This function choses among the four options, (merge, split, 
  new disjunct, or best class), based on a singe attribute, best-att.
  For each option, a "compute-CU-?"  function is called (see
  node-compute.lsp). These return many things, including a pointer
  to a modified tree and a score.  These scores are compared to
  choose an action.

  Note that the tree is *Not* modified here. That is reserved for 
  ClassRatt. 

-------------------------------------------------------------------KThompso--|#

(defun Class-One (root att-names)

  (let
    (CU-self CU-best-child     ;category-utility scores
     (CU-merge -999) (CU-split -999)
     winner                    ; winner from ordered list of CU scores

     self-tree best-child-tree ;resultant trees
     merge-tree split-tree
     
     self-node                 ;new node with only obj in it
     best-node second-node     ;best (with obj integrated) and 
			       ;  second best nodes to place obj in.
     old-best-node             ;best node without obj integrated
     oldsplit-node             ;best node of split w/o obj integrated
     merge-node split-node     ;child nodes with obj for merge and
			       ;  split
     )

    (multiple-value-setq (CU-self self-tree self-node)
      (compute-CU-self root att-names))

    (multiple-value-setq 
     (CU-best-child best-child-tree best-node second-node old-best-node)
     (compute-CU-best root att-names))

    (if *LEARNING*        ;dont do these if testing.
      (progn
	(if *MERGE*
	  (multiple-value-setq (CU-merge merge-tree merge-node)
            (compute-CU-merge root best-node second-node old-best-node
		 att-names)))
	(if *SPLIT*
	  (multiple-value-setq (CU-split split-tree split-node oldsplit-node)
            (compute-CU-split root best-node old-best-node att-names)))
    ))
    (setq winner (max CU-self CU-best-child CU-merge CU-split))
    (if *GLOBAL-DEBUG*
	(pr-out
	    "===== Scores:Self = ~5F  Best = ~5F  Merge = ~5F  Split = ~5F~%"
	    CU-self CU-best-child CU-merge CU-split))

    (cond

     ((eq CU-self winner)
      (if (or *GLOBAL-DEBUG* *LEARNING*)
       (pr-out "putting instance by itself~%"))
      (values winner self-tree self-node -999 old-best-node 'DISJUNCT)
     )

     ((eq CU-best-child winner)
      (if (or *GLOBAL-DEBUG* *LEARNING*)
        (pr-out "putting with child ~A~%" (node-members old-best-node)))
      (values winner best-child-tree best-node old-best-node 
	      second-node 'BEST)
     )

     ((eq CU-merge winner)
      (if (or *GLOBAL-DEBUG* *LEARNING*)
       	(pr-out "merging children ~A and ~A~%"
	 (node-members old-best-node) (node-members second-node)))
      (values winner merge-tree merge-node -999 old-best-node 'MERGE)
     )                        ; See ModifyTree for "-999" explanation.

     ((eq CU-split winner)
      (if (or *GLOBAL-DEBUG* *LEARNING*)
        (pr-out "splitting child ~A~%" (node-members old-best-node)))
      (values winner split-tree split-node oldsplit-node old-best-node 'SPLIT)
     )
   )
))

;*------------------------------------------------------------------15/Mar/89-*
;*  Function  - MODIFYTREE
;* 
;*  Inputs    -> the tree (current root node), the node (the chosen child of
;*     tree), the oldnode (the node w/o 'OBJ incorporated), the modified
;*     list of parent attributes, and the level.
;* 
;*  Returns   -> the new modified tree.
;*  Actions   -> This is where the hierarchy actually gets modified (via a 
;*     setf statement). It is called only from store, and only if the system
;*     is about to recurse (not in the disjunct case). The If clause is a
;*     look-ahead: if the node we are about to recurse on has no children,
;*     then recurse on a node without the new obj incorporated, since we
;*     are about to do a "fork-node". Note that this should never happen 
;*     if Merge is chosen (hence -999, above).
;* MODIFIED 4/89:
;*     The "newatts" parameter is the parent list of attributes with the
;*     appropriate childscore values.
;*     Also, as we recurse down a level with the classify call, we start
;*     over with the original (global) list of *ATT-NAMES*.
;*   NOTE: if we about to fork-node, that we only use the known-attributes
;*     (Familiarization....)
;*     Finally, the clause for disjuncts (do NOT recurse) is here.
;*---------------------------------------------------------------------JHG----*

(defun ModifyTree (tree node oldnode level known-atts)

  (cond ((eq 1 (node-count node))   ;node is a new disjunct
         (if (not *LEARNING*)
	      (break "what are you doing in Modify tree??")))

        ((null (node-children node))      ; node has no children
         (setf (elt (node-children tree) 
		(position node (node-children tree)))
	 (classify oldnode known-atts (1+ level))))

	( t
         (setf (elt (node-children tree) 
		(position node (node-children tree)))
	 (classify node *ATT-NAMES* (1+ level))))
  )
  tree  ;return the new root
)

;*----------------------------------------------------------------------------
;* 
;*            CODE FOR MEASURING EFFICIENCY and ACCURACY.
;* 
;*---------------------------------------------------------------------JHG----*

;*------------------------------------------------------------------26/May/89-*
;*  Function  - PREDICTED-VALUES
;* 
;*  Inputs    -> The node (and globally, the attributes you want to predict).
;* 
;*  Returns   -> A list of attributes from the node that correspond to the
;*          attribute names in *PRED-ATTS*.
;*---------------------------------------------------------------------JHG----*

(defun predicted-values (node)

  (loop for att-name in *PRED-ATTS*
	for att = (find-att att-name (node-atts node)) 
	if att collect att into answer
	else do (pr-out "can't predict about attribute ~A~%" att-name)
   finally 
        (return answer)
  )
)

;*------------------------------------------------------------------26/May/89-*
;*  Function  - UPDATE-COST
;* 
;*  Inputs    -> the number of nodes or the number of attributes inspected
;* 
;*  Actions   -> updates the global variables *COST-A* and *COST-N*.
;*---------------------------------------------------------------------JHG----*

(defun update-costs (&key (attr nil) (nodes nil) (clear nil))

   (cond ( clear
	   (setq *COST-A* 0)
	   (setq *COST-N* 0)
	   (setq *WORK* 0))
	 ( t
	   (setq *COST-A* (+ *COST-A* attr))
	   (setq *COST-N* (+ *COST-N* nodes))
	   (setq *WORK* (+ *WORK* (* attr nodes))))
   )
)
