;;; -*- Mode:Common-Lisp; Package:user; Base:10 -*-

;;;; Copyright (c) 1992 by Hwee Tou Ng. This program may be freely
;;;; copied, used, or modified provided that this copyright notice 
;;;; is included in each copy of this code and parts thereof.

(in-package :user)

(proclaim '(optimize (speed 3) (compilation-speed 0)))

(export '(make-brule 
	  kb-brule-consequent kb-brule-antecedents
	  get-facts get-brules
	  index-fact index-brule
	  add-kb-facts remove-kb-facts
	  add-kb-brules remove-kb-brules
	  remove-current-indices
	  uniquify-fact uniquify-brule
	  final-goal-name-count final-goal-name-count?
	  final-goal-datum final-goal-datum?
	  assumable?
	  load-kb unload-kb))

;--------------------------------
;     KB management
;--------------------------------

(defvar *domain*)
(defvar *brules*)
(defvar *facts*)
(defvar *fact-predicates* nil)
(defvar *brule-predicates* nil)

(defvar *coherence-hash-table*)  ; used for computing the coherence metric
(defvar *sort-hierarchy*)
(defvar *unique-slot-value-predicates*)
(defvar *plan-steps*)
(defvar *plan-step-predicates*)

(defstruct (fact (:print-function print-fact))
  id
  datum)

(defstruct (brule (:print-function print-brule))
  id
  conseq
  antes)

(defun print-fact (fact stream depth)
  (declare (ignore depth))
  (format stream "~s: ~s" (fact-id fact) (fact-datum fact))
  (values))

(defun print-brule (brule stream depth)
  (declare (ignore depth))
  (format stream "~s: ~s <- ~s"
	  (brule-id brule) (brule-conseq brule) (brule-antes brule))
  (values))

(defun kb-brule-consequent (kb-brule)
  (second kb-brule))

(defun kb-brule-antecedents (kb-brule)
  (cddr kb-brule))

(defun retrieve-facts (predicate)
  (cond ((get predicate 'facts))
	(t (push predicate *fact-predicates*)
	   (setf (get predicate 'facts) (make-dtree)))))
  
(defun remove-kb-facts ()
  (dolist (predicate *fact-predicates*)
    (setf (get predicate 'facts) nil))
  (setf *fact-predicates* nil))

(defun index-fact (fact)
  (dtree-index (fact-datum fact) fact
	       (retrieve-facts (predicate (fact-datum fact)))))

(defun get-facts (datum)
  (dtree-fetch datum (retrieve-facts (predicate datum))
	       nil 0 nil most-positive-fixnum))

(defun add-kb-facts (kb-facts &aux fact (fact-count 0))
  (dolist (kb-fact kb-facts)
    (setf fact (make-fact :id (cons (fact-name-count (incf fact-count))
				    (collect-vars kb-fact))
			  :datum kb-fact))
    (setf (get (fact-name-count fact-count) 'fact) fact)
    (index-fact fact)))


(defun retrieve-brules (predicate)
  (cond ((get predicate 'brules))
	(t (push predicate *brule-predicates*)
	   (setf (get predicate 'brules) (make-dtree)))))

; Remove all facts and brules that are indexed for
; the current problem.

(defun remove-current-indices ()
  (loop
    (setf (get (first *brule-predicates*) 'brules) nil)
    (when (eq (pop *brule-predicates*) 'G.1)
      (return)))
  (loop
    (setf (get (first *fact-predicates*) 'facts) nil)
    (when (eq (pop *fact-predicates*) 'G.1)
      (return))))
  
(defun remove-kb-brules ()
  (dolist (predicate *brule-predicates*)
    (setf (get predicate 'brules) nil))
  (setf *brule-predicates* nil))

(defun index-brule (brule)
  (dtree-index (brule-conseq brule) brule
	       (retrieve-brules (predicate (brule-conseq brule)))))

(defun get-brules (datum)
  (dtree-fetch datum (retrieve-brules (predicate datum))
	       nil 0 nil most-positive-fixnum))


(defun add-kb-brules (kb-brules &aux brule (rule-count 0))
  (dolist (kb-brule kb-brules)
    (setf brule (make-brule :id (cons (rule-name-count (incf rule-count))
				      (collect-vars kb-brule))
			    :conseq (kb-brule-consequent kb-brule)
			    :antes (kb-brule-antecedents kb-brule)))
    (setf (get (rule-name-count rule-count) 'brule) brule)
    (index-brule brule)))


(defun uniquify-fact (fact &aux n-f)
  (setf n-f (uniquify (list (fact-id fact) (fact-datum fact))))
  (values (first n-f) (second n-f)))

(defun uniquify-brule (brule &aux n-c-a)
  (setf n-c-a (uniquify (list (brule-id brule)
			      (brule-conseq brule)
			      (brule-antes brule))))
  (values (first n-c-a) (second n-c-a) (third n-c-a)))

(defun rule-name-count (count)
  (intern (format nil "R.~d" count) 'user))

(defun fact-name-count (count)
  (intern (format nil "F.~d" count) 'user))

(defun final-goal-name-count (count)
  (intern (format nil "G.~d" count) 'user))

(defun final-goal-name-count? (predicate &aux string)
  (and (> (length (setf string (string predicate))) 2)
       (char= (char string 0) #\G)
       (char= (char string 1) #\.)))

(defun final-goal-datum (count vars)
  (cons (final-goal-name-count count) vars))

(defun final-goal-datum? (datum)
  (and (consp datum)
       (final-goal-name-count? (predicate datum))))

(defvar *predicate-specific-abduction*)
(defvar *assumable-predicates*)

(defun assumable? (a)
  (and (not (final-goal-datum? a))
       (not (equal a '(falsity)))
       (or (not *predicate-specific-abduction*)
	   (member (predicate a) *assumable-predicates*))))

; An instance-constraining rule is of the form:
; (<- (inst ?y ?b) (inst ?x ?a) (role-predicate ?x ?y))

(defun inst-constrain-rule? (r)
  (and (= (length r) 4)
       (eq (predicate (second r)) 'inst)
       (eq (predicate (third r)) 'inst)
       (= (length (fourth r)) 3)
       (eq (second (third r)) (second (fourth r)))
       (eq (second (second r)) (third (fourth r)))))

(defvar *behavior-mode-predicates*)
(defvar *free-assumption-predicates*)

(defun load-kb (kb)
  (load kb)
  (setf *brules* (nreverse *brules*))
  (add-kb-brules *brules*)
  (add-kb-facts *facts*)
  (when (or (eq *domain* 'qsim-diagnosis)
	    (eq *domain* 'circ-diagnosis))
    (setf *behavior-mode-predicates*
	  (cons 'norm (set-difference *assumable-predicates*
				      *free-assumption-predicates*))))
  (when (eq *domain* 'plan-recognition)
    (defhierarchy *sort-hierarchy*)
    (dolist (p *unique-slot-value-predicates*)
      (setf (get p 'uniq-pred) t))
    (setf *plan-step-predicates* nil)
    (dolist (p-steps *plan-steps*)
      (setf (get (first p-steps) 'plan) t)
      (dolist (p-step (rest p-steps))
	(pushnew p-step *plan-step-predicates*)))
    (dolist (p *plan-step-predicates*)
      (setf (get p 'plan-step) t))
    (dolist (r *brules*)
      (when (inst-constrain-rule? r)
	(setf (get (predicate (fourth r)) 'role-pred) t)
	(push (list (third (third r))
		    (third (second r)))
	      (get (predicate (fourth r)) 'inst-constr)))))
  (setf *coherence-hash-table*
	(make-hash-table :test #'equal :size 100))
  (values))

(defun unload-kb ()
  (remove-kb-brules)
  (remove-kb-facts)
  (when (or (eq *domain* 'qsim-diagnosis)
	    (eq *domain* 'circ-diagnosis))
    (setf *behavior-mode-predicates* nil))
  (when (eq *domain* 'plan-recognition)
    (undefhierarchy)
    (dolist (p *unique-slot-value-predicates*)
      (setf (get p 'uniq-pred) nil))
    (dolist (p-steps *plan-steps*)
      (setf (get (first p-steps) 'plan) nil))
    (dolist (r *brules*)
      (when (inst-constrain-rule? r)
	(setf (get (predicate (fourth r)) 'role-pred) nil)
	(setf (get (predicate (fourth r)) 'inst-constr) nil)))
    (dolist (p *plan-step-predicates*)
      (setf (get p 'plan-step) nil))))
