% PENDCON Fuzzy controller for inverted pendulum
%   This example is refered to in Chapter 5.
%
% USES: pendmod, pendplot

% Mark Beale 6-24-93
% Copyright (c) 1993 by PWS Publishing Company

% DEFINITION OF VARIABLE RANGES
% =============================

deg = pi/180; % (a degree in radians)

% FUZZY SETS
% ==========

% POSITION
PosL = eqshape(3,[-75 75],5,'extend');

% VELOCITY
VelL = eqshape(3,[-20 20],5,'extend');

% ANGLE
AngleL = eqshape(3,[-90, 90]*deg,7,'extend');

% ANGULAR VELOCITY
AngleVelL = eqshape(3,[-4 4],7,'extend');

% REFERENCE ANGLE
RefS = [-55:(55/2):55]*deg;

% CONTROL FORCE
ForceS = -400:(400/3):400;

% PLOT FUZZY SETS
% ===============

clc
disp('Hit any key to see Fuzzy Sets for Position...')
pause
PosS = -75:(75/24):75;
PosG = shapeval(PosL,PosS);
plotfs(PosS,'Fuzzy Sets for Position', 'Position (meters)');
plot(PosS,PosG);
pause

clc
disp('Hit any key to see Fuzzy Sets for Base Velocity...')
pause
VelS = -20:(20/24):20;
VelG = shapeval(VelL,VelS);
plotfs(VelS,'Fuzzy Sets for Velocity', 'Velocity (meters/second)');
plot(VelS,VelG);
pause

clc
disp('Hit any key to see Fuzzy Sets for Angle...')
pause
AngleS = (-pi/2):(pi/48):(pi/2);
AngleG = shapeval(AngleL,AngleS);
plotfs(AngleS/deg,'Fuzzy Sets for Angle','Angle (Degrees)');
plot(AngleS/deg,AngleG);
pause

clc
disp('Hit any key to see Fuzzy Sets for Angular Velocity...')
pause
AngleVelS = -4:4/24:4;
AngleVelG = shapeval(AngleVelL,AngleVelS);
plotfs(AngleVelS,'Fuzzy Sets for Angular Velocity', ...
  'Angular Velocity (Rad/sec)');
plot(AngleVelS,AngleVelG);
pause

clc
disp('Hit any key to see Fuzzy Sets for Reference Angle...')
pause
RefG = ones(1,length(RefS));
plotfs(RefS*deg,'Fuzzy Sets for Reference Angle','Ref Angle (Deg)');
plot(RefS*deg,RefG,'*');
pause

clc
disp('Hit any key to see Fuzzy Sets for Control Force...')
pause
ForceG = ones(1,length(ForceS));
plotfs(ForceS,'Fuzzy Sets for Control Force','Force (Kg*m/sec^2)');
plot(ForceS,ForceG,'*');
pause

% DEFINITION OF CONTROLLER 1
% ==========================
% Decides reference angle based on position and velocity:
  
% ANTECEDANT TABLE
A1 = table(1:5,1:5);

% WEIGHT TABLE, Rows represent Velocity, Columns represent Position

W1 = [1.0;  1.0;  1.0;  0.5;  0.0;
		1.0;  1.0;  0.5;  0.0; -0.5;
		1.0;  0.5;  0.0; -0.5; -1.0;
		0.5;  0.0; -0.5; -1.0; -1.0;
		0.0; -0.5; -1.0; -1.0; -1.0];

% GROUP OF FUZZY SETS
Z1 = group(PosL,VelL);

% DEFINITION OF CONTROLLER 1
% ==========================
% Determines Force based on Angle and Angular velocity:

% ANTECEDANT TABLE
A2 = table(1:7,1:7);

% WEIGHT TABLE, Rows represent Angle, Cols represent Angular Velocity
W2 = [-400; -400; -400; -400; -266; -133;   0;
		-400; -400; -400; -266; -133;    0; 133;
		-400; -400; -266; -266;    0;  133; 266;
		-400; -266; -133;    0;  133;  266; 400;
		-266; -133;    0;  266;  266;  400; 400;
		-133;    0;  133;  266;  400;  400; 400;
			0;  133;  266;  400;  400;  400; 400];

% GROUP OF FUZZY SETS
Z2 = group(AngleVelL,AngleL);

% SIMULATION OF CONTROLLED PENDULUM
% =================================

% INITIAL STATE
x0 = [4; 0; 0*deg; 0*deg];

% TIME
t0 = 0;
t1 = 35;
dt= 0.05;
T = (t0+dt):dt:t1;
steps = length(T);

X = zeros(4,steps);
R = zeros(1,steps);
U = zeros(1,steps);

clc
disp('Hit any key to see the controller at work...')
pause

x = x0;
pendplot(x,0,0);
for i=1:steps

  % EVALUATE CONTROLLER EVERY 3 TIMESTEPS
  if rem(i,3) == 1
	 r = frules(A1,W1,Z1,[x(1); x(2)]); if isnan(r), return, end
	 u = frules(A2,W2,Z2,[x(4); x(3)-r]); if isnan(u), return, end
  end

  % UPDATE STATE
  x = x + dt * pendmod(x,u);

  % CHECK FOR COLLISION OF PENDULUM WITH BASE
  if abs(x(3)) > pi/2, x(3) = sign(x(3))*pi/2; end

  % PLOT EVERY 3 TIME STEPS
	 if rem(i,3) == 1
	 pendplot(x,u,i*dt);
	 end

  % SAVE VALUES
	 X(:,i) = x;
  R(i) = r; 
  U(i) = u;
  
  % FINISH IF BALANCED AND CENTERED
  if all(abs(x) < 0.01)
	 steps = i;
	 X = X(:,1:steps);
	 R = R(1:steps);
	 U = U(1:steps);
	 T = T(1:steps);
	 t1 = t0+steps*dt;
	 break
  end
end

% PLOT RESPONSES
% ==============

% POSITION RESPONSE
clc
disp('Hit any key to see the Position Response...')
pause
plot([t0 T],[x0(1) X(1,:)])
hold on
plot([t0 t1],[0 0],':')
hold off
title('Position Response')
xlabel('Time (Seconds)')
ylabel('Position (Meters)')
pause

% ANGULAR RESPONSE
clc
disp('Hit any key to see the Angular Response...')
pause
plot([t0 T],[x0(3) X(3,:)]/deg)
hold on
plot([t0 t1],[0 0],':')
hold off
title('Angular Response')
xlabel('Time (Seconds)')
ylabel('Angle (Degrees)')
pause

% FORCE RESPONSE
clc
disp('Hit any key to see the Force Response...')
pause
plot(T,U)
hold on
plot([t0 t1],[0 0],':')
hold off
title('Control Force Response')
xlabel('Time (Seconds)')
ylabel('Force (Kg*m/s^2)')
pause
clear

