//---------------------------------------------------------------------------
//
// CONFINT.CC
//
//    Confidence interval type implementation.
//
//    Based on A. Kaufmann, M. Gupta, "Introduction to Fuzzy Arithmetic",
//    Van Nostrand Reinhold, New York, 1991.
//
// Contents:
//
//    double min4( double opnd1, double opnd2, double opnd3, double opnd4)
//    double max4( double opnd1, double opnd2, double opnd3, double opnd4)
//
//    ConfInt::ConfInt( double lBound, double uBound)
//
//    ConfInt ConfInt::operator*( ConfInt opnd2)
//    ConfInt ConfInt::operator*=( ConfInt opnd2)
//
//    ConfInt operator/( double opnd1, ConfInt opnd2)
//
//    ConfInt sin( ConfInt opnd)
//    ConfInt cos( ConfInt opnd)
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

#include "confint.h"

#include <stdlib.h>


//---------------------------------------------------------------------------
//
// double min4( double opnd1, double opnd2, double opnd3, double opnd4)
//
//    Calculate the minimum from 4 double values.
//
// Arguments:
//
//    double opnd1
//    double opnd2
//    double opnd3
//    double opnd4
//       the input values
//
// Return value:
//
//    the computed minimum
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 05.03.94)
//
//---------------------------------------------------------------------------

double min4( double opnd1, double opnd2, double opnd3, double opnd4)

{
   double min = ( opnd1 < opnd2) ? opnd1 : opnd2;

   if ( opnd3 < min) min = opnd3;
   if ( opnd4 < min) min = opnd4;

   return min;
}


//---------------------------------------------------------------------------
//
// double max4( double opnd1, double opnd2, double opnd3, double opnd4)
//
//    Calculate the maximum from 4 double values.
//
// Arguments:
//
//    double opnd1
//    double opnd2
//    double opnd3
//    double opnd4
//       the input values
//
// Return value:
//
//    the computed maximum
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 05.03.94)
//
//---------------------------------------------------------------------------

double max4( double opnd1, double opnd2, double opnd3, double opnd4)

{
   double max = ( opnd1 > opnd2) ? opnd1 : opnd2;

   if ( opnd3 > max) max = opnd3;
   if ( opnd4 > max) max = opnd4;

   return max;
}


//---------------------------------------------------------------------------
//
// ConfInt::ConfInt( double lBound, double uBound)
//
//    Confidence interval constructor.
//
//    Confidence intervals require upper bound greater than (or equal to)
//    lower bound.
//
// Arguments:
//
//    double lBound
//    double uBound
//       lower and upper bound for the confidence interval
//
// Return value:
//
//    a new confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.04.94)
//
//---------------------------------------------------------------------------

ConfInt::ConfInt( double lBound, double uBound)

   :  lower( lBound),
      upper( uBound)

{
   if ( lBound > uBound) {

      cerr << "\n\nConfidence Interval Error:"
           << "\n\nLower bound greater than upper bound in\n\n\t[ "
           << lBound << " ," << uBound << "].\n";

      abort();
   }
}


//---------------------------------------------------------------------------
//
// ConfInt ConfInt::operator*( ConfInt opnd2)
//
//    Multiplication between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to multiply for
//
// Return value:
//
//    the resulting confidence value
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

ConfInt ConfInt::operator*( ConfInt opnd2)

{
   double tmp1 = lower * opnd2.lower;
   double tmp2 = lower * opnd2.upper;
   double tmp3 = upper * opnd2.lower;
   double tmp4 = upper * opnd2.upper;

   return ConfInt( min4( tmp1, tmp2, tmp3, tmp4),
                   max4( tmp1, tmp2, tmp3, tmp4));
}


//---------------------------------------------------------------------------
//
// ConfInt ConfInt::operator*=( ConfInt opnd2)
//
//    Multiplication and assignment between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to multiply for
//
// Return value:
//
//    the resulting confidence value
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

ConfInt ConfInt::operator*=( ConfInt opnd2)

{
   double tmp1 = lower * opnd2.lower;
   double tmp2 = lower * opnd2.upper;
   double tmp3 = upper * opnd2.lower;
   double tmp4 = upper * opnd2.upper;

   lower = min4( tmp1, tmp2, tmp3, tmp4);
   upper = max4( tmp1, tmp2, tmp3, tmp4);

   return *this;
}


//---------------------------------------------------------------------------
//
// ConfInt operator/( double opnd1, ConfInt opnd2)
//
//    Division of a singleton by a confidence interval.
//
// Arguments:
//
//    double opnd1
//    ConfInt opnd2
//       division operands
//
// Return value:
//
//    the resulting confidence value
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

ConfInt operator/( double opnd1, ConfInt opnd2)

{
   if ( opnd2.lower * opnd2.upper <= 0) {

      cerr << "\n\nConfidence Interval Error:"
           << "\n\nDivision by zero error in\n\n\t"
           << opnd1 << "/" << opnd2 << ".\n";

      abort();
   }

   return opnd1 * ConfInt( 1 / opnd2.upper, 1 / opnd2.lower);
}


//---------------------------------------------------------------------------
//
// ConfInt sin( ConfInt opnd)
//
//    Calculate sine of confidence intervals.
//
// Arguments:
//
//    ConfInt opnd
//       a confidence interval (expressed in radiants, not degree)
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.04.94)
//
//---------------------------------------------------------------------------

ConfInt sin( ConfInt opnd)

{
   if ( opnd.upper - opnd.lower >= ( 2 * M_PI))
      return ConfInt( -1, 1);

   int offset = int( opnd.lower / ( 2 * M_PI));
   double max = ( M_PI / 2) + offset * ( 2 * M_PI);
   double min = ( 3 * M_PI / 2) + offset * ( 2 * M_PI);
   double low = sin( opnd.lower);
   double upp = sin( opnd.upper);

   double lower = (( opnd.lower <= min && opnd.upper >= min)
		   || opnd.upper >= ( min + ( 2 * M_PI)))
                ? -1
                : ( low < upp) ? low : upp;

   double upper = (( opnd.lower <= max && opnd.upper >= max)
                   || opnd.upper >= ( max + ( 2 * M_PI)))
                ? 1
                : ( low > upp) ? low : upp;   

   return ConfInt( lower, upper);
}


//---------------------------------------------------------------------------
//
// ConfInt cos( ConfInt opnd)
//
//    Calculate cosine of confidence intervals.
//
// Arguments:
//
//    ConfInt opnd
//       a confidence interval (expressed in radiants, not degree)
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.04.94)
//
//---------------------------------------------------------------------------

ConfInt cos( ConfInt opnd)

{
   if ( opnd.upper - opnd.lower >= ( 2 * M_PI))
      return ConfInt( -1, 1);

   int offset = int( opnd.lower / ( 2 * M_PI));
   double max = offset * ( 2 * M_PI);
   double min = M_PI + offset * ( 2 * M_PI);
   double low = cos( opnd.lower);
   double upp = cos( opnd.upper);

   double lower = (( opnd.lower <= min && opnd.upper >= min)
                   || opnd.upper >= ( min + ( 2 * M_PI)))
                ? -1
                : ( low < upp) ? low : upp;

   double upper = (( opnd.lower <= max && opnd.upper >= max)
                   || opnd.upper >= ( max + ( 2 * M_PI)))
                ? 1
                : ( low > upp) ? low : upp;   

   return ConfInt( lower, upper);
}
