#include "global.h"


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 */ 
static int domains[N][K];
static int conflicts[N][N];
static int checking[N][N];


/*
 * Clear the setup data structures.
 */
static void clear_setup(n, k)
int n, k;
{
    int i, j;

    for (i = 1; i <= n; i++) {
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
        for (j = 1; j <= n; j++) {
            conflicts[i][j] = 0;
            checking[i][j] = 0;
        }
    }
}


/*
 * Create the union of two sets which hold where the conflicts took place.  As
 * described by Prosser, this is necessary to be able to continue graph-based
 * backjumping correctly to ensure the correct variable will not be missed.
 */
static void union_checking(i, j)
int i, j;
{
    int n;

    for (n = 1; n < i; n++)
        conflicts[i][n] = conflicts[i][n] || checking[n][j];
}


/*
 * Create the union of two sets which hold where the conflicts took place.  As
 * described by Prosser, this is necessary to be able to continue graph-based
 * backjumping correctly to ensure the correct variable will not be missed.
 * The maximum in the set for variable "i" is retained in position
 * "conflicts[i][i]" for future reference.
 */
static void union_conflicts(i, j)
int i, j;
{
    int n;

    for (n = 1; n < i; n++) {
        conflicts[i][n] = conflicts[i][n] || conflicts[j][n];
        if (conflicts[i][n] && conflicts[i][i] < n)
            conflicts[i][i] = n;
    }
}


/*
 * Reset the sets for the given index to be empty.  This allows for new
 * information to be stored in these sets without the old information
 * misleading the algorithm.
 */
static void empty_conflicts(index)
int index;
{
    int i;

    for (i = 1; i <= index; i++)
        conflicts[index][i] = 0;
}


/*
 * This function checks the edge between variables "i" and "j" to see which
 * values in the domain of "j" can be eliminated due to the current
 * instantiation of "i" as "value".  Note that given a variable "i" and an
 * instantiation "v", domains[i][v] is 0 if the value is still allowed,
 * otherwise it contains the variable which caused it to be eliminated.  If any
 * value in "j" is deleted, this is noted in the "checking" matrix.  The number
 * of values in the domain of "j" after all deletions is returned.
 */
static int check_forward(k, current, j, value)
int k, current, j, value;
{
    int i, delete_count, old_count = 0;

    checks++;
    for (i = 0; i < k; i++)
        if (domains[j][i] == 0)
            old_count++;
    if (domains[j][value] == 0) {
        delete_count = 1;
        domains[j][value] = current;
        checking[current][j] = 1;
    }
    return(old_count - delete_count);
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(i, n, k)
int i, n, k;
{
    int j, l;

    for (j = i + 1; j <= n; j++)
        if (checking[i][j]) {
            for (l = 0; l < k; l++)
                if (domains[j][l] == i)
                    domains[j][l] = 0;
            checking[i][j] = 0;
        }
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int current, n, k, value;
{
    int i;

    for (i = current + 1; i <= n; i++) {
        if (C[current][i] == U)
            continue;
        else if (check_forward(k, current, i, value) == 0)
            return(i);
    }
    return(0);
}


/*
 * Solve the constraint network using the FORWARD CHECKING with GRAPH-BASED
 * BACKJUMPING (FC-GBJ) method.  If this is the first variable then initialize
 * the data structures.  If a solution is found then update the "found"
 * variable, call the function to process the solution, and return the value
 * according to whether the first solution is desired or all solutions.  Then,
 * check if the timer has expired, and if it has, return immediately.
 * Otherwise, begin checking each possible instantiation of the variable.  For
 * each domain value, perform the following steps.  First, if preprocessing
 * eliminated the value, disregard the rest of the loop.  Otherwise,
 * instantiate the variable, check if the network is still consistent, and then
 * call the backtracking routine recursively.  After checking each domain
 * value, restore the domains which were eliminated by the instantiation.
 * After checking all possible domain values, determine the variable to jump
 * back to, update the data structures, and return.
 */
int FC_GBJ(C, n, k, solution, current, found)
NETWORK C;
int n, k, current, *found;
SOLUTION solution;
{
    int i, jump, fail;

    if (current == 1) {
        clear_setup(n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        return(0);
    }
    if (time_expired())
        return(0);
    for (i = 0; i < k; i++) {
        if (domains[current][i])
            continue;
        solution[current] = i;
        fail = consistent(C, n, k, current, solution[current]);
        if (!fail) {
            jump = FC_GBJ(C, n, k, solution, current + 1, found);
            if (jump != current)
                return(jump);
        }
        if (fail)
            union_checking(current, fail);
        restore(current, n, k);
    }
    jump = 0;
    for (i = 1; i < current; i++)
        if (conflicts[current][i])
            jump = i;
    for (i = jump + 1; i < current; i++)
        if (checking[i][current])
            jump = i;
    union_checking(current, current);
    union_conflicts(jump, current);
    for (i = current; i > jump; i--) {
        empty_conflicts(i);
        restore(i, n, k);
    }
    restore(jump, n, k);
    return(jump);
}
