#include "global.h"


/*
 * Data structures to hold the required information.  See the accompanying
 * research articles for complete descriptions of the structures.  The
 * "domains" structure works as follows.  If an entry contains a 0, then the
 * value is still allowed.  Otherwise, it contains the variable whose
 * instantiation eliminated it.  Note that this data structure has no
 * information from preprocessing (which is kept directly in the "C" matrix).
 */
static int checking[N][N];
static int domains[N][K];
static int parents[N][N];
static int P[N];


/*
 * Return whether there exists some constraint between two variables, which is
 * simple searching an edge for a "0" entry.
 */
static int constraint(con, k)
CONSTRAINT con;
int k;
{
    int i, j;

    for (i = 0; i < k; i++)
        for (j = 0; j < k; j++)
            if (con[i][j] == 0)
                return(1);
    return(0);
}


/*
 * Clear the setup data structures.  Then, compute the parent tree and store
 * this information in the data structure to be used when backjumping is to be
 * done.  See the "Dechter" paper for a full description of the data structure.
 */
static void clear_setup(C, n, k)
NETWORK C;
int n, k;
{
    int i, j, l;

    for (i = 1; i <= n; i++) {
        P[i] = 0;
        for (j = 1; j <= n; j++)
            parents[i][j] = 0;
            checking[i][j] = 0;
        for (j = 0; j < k; j++)
            domains[i][j] = 0;
    }
    for (i = 1; i <= n; i++)
        for (j = 1; j < i; j++)
            if (constraint(C[i][j], k)) {
                for (l = 1; l < j; l++)
                    parents[i][l] = parents[i][l] || parents[j][l];
                parents[i][j] = 1;
                parents[i][i] = j;
            }
}


/*
 * Create the union of two parent sets.  This is necessary to be able to
 * continue graph-based backjumping correctly to ensure the correct variable
 * will not be missed.
 */
static int union_parents(i)
int i;
{
    int j, temp = 0;

    for (j = 1; j < i; j++)
        if (P[j] |= parents[i][j])
            temp = j;
    P[i] = 0;
    return(temp);
}


/*
 * Update the variable to jump back to by looking at the lists of parents.  As
 * described by Dechter, this is necessary to be able to continue graph-based
 * backjumping correctly to ensure the correct variable will not be missed.
 */
static void update_parents(i, j)
int i, j;
{
    int k;

    for (k = 1; k < i; k++)
        P[k] = P[k] || parents[j][k];
}


/*
 * This function checks the edge between variables "i" and "j" to see which
 * values in the domain of "j" can be eliminated due to the current
 * instantiation of "i" as "value".  Note that given a variable "i" and an
 * instantiation "v", domains[i][v] is 0 if the value is still allowed,
 * otherwise it contains the variable which caused it to be eliminated.  If any
 * value in "j" is deleted, this is noted in the "checking" matrix.  The number
 * of values in the domain of "j" after all deletions is returned.
 */
static int check_forward(C, k, i, j, value)
NETWORK C;
int k, i, j, value;
{
    int n, old_count = 0, delete_count = 0;

    for (n = 0; n < k; n++)
        if (C[j][j][n][n] && (domains[j][n] == 0)) {
            old_count++;
            checks++;
            if (C[i][j][value][n] == 0) {
                domains[j][n] = i;
                delete_count++;
            }
        }
    if (delete_count)
        checking[i][j] = 1;
    return(old_count - delete_count);
}


/*
 * Function to restore the domain of variables which were eliminated due to the
 * instantiation of variable "i".  A variable is known to have had a value
 * deleted by looking at the "checking" matrix.
 */
static void restore(i, n, k)
int i, n, k;
{
    int j, l;

    for (j = i + 1; j <= n; j++)
        if (checking[i][j]) {
            for (l = 0; l < k; l++)
                if (domains[j][l] == i)
                    domains[j][l] = 0;
            checking[i][j] = 0;
        }
}


/*
 * Check if the current instantiation of the variables is consistent by
 * checking if the current instantiation of a variable will reduce some future
 * variable's domain to the empty set, in which case the appropriate result is
 * returned.
 */
static int consistent(C, n, k, current, value)
NETWORK C;
int n, k, current, value;
{
    int i;

    for (i = current + 1; i <= n; i++) {
        if (check_forward(C, k, current, i, value) == 0) {
            update_parents(current, i);
            return(0);
        }
    }
    return(1);
}


/*
 * Solve the constraint network using the FORWARD CHECKING with Dechter's
 * GRAPH_BASED BACKJUMPING (FC-GBJ) method.  If this is the first variable
 * then initialize the data structures.  If a solution is found then update
 * the "found" variable, call the function to process the solution, and return
 * the value according to whether the first solution is desired or all
 * solutions.  Then, check if the timer has expired, and if it has, return
 * immediately.  Otherwise, begin checking each possible instantiation of the
 * variable.  For each domain value, perform the following steps.  First, if
 * preprocessing eliminated the value, disregard the rest of the loop.
 * Otherwise, instantiate the variable, check if the network is still
 * consistent, and then call the backtracking routine recursively.  After
 * checking each domain value, restore the domains which were eliminated by the
 * instantiation.  After checking all possible domain values, determine the
 * variable to jump back to, update the data structures, and return.
 */
int FC_GBJ(C, n, k, solution, current, number, found)
NETWORK C;
int n, k, current, number, *found;
SOLUTION solution;
{
    int i, jump, curr = count;

    if (current == 1) {
        clear_setup(C, n, k);
        *found = 0;
    } else if (current > n) {
        process_solution(C, n, solution);
        *found = 1;
        count++;
        return(number == 1 ? 0 : n);
    }
    if (time_expired())
        return(0);
    for (i = 0; i < k; i++) {
        if (C[current][current][i][i] == 0 || domains[current][i])
            continue;
        solution[current] = i;
        if (consistent(C, n, k, current, solution[current])) {
            jump = FC_GBJ(C, n, k, solution, current + 1, number, found);
            if (jump != current)
                return(jump);
        }
        restore(current, n, k);
    }
    jump = (curr == count) ? union_parents(current) : current - 1;
    for (i = current; i > jump; i--)
        restore(i, n, k);
    return(jump);
}
