/*
Copyright (C) 1992 University of Maryland.

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies, and
that both that copyright notice and this permission notice appear
in supporting documentation.  The author makes no representations
about the suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty. Your use of
this software signifies that you are willing to take the risks of
using this software by yourself.

THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
USE OR PERFORMANCE OF THIS SOFTWARE.
*/


/* trans.c */

/* author    : Steve Armentrout
 *             Dept of Computer Science
 *             University of Maryland, College Park
 */

/*
 * trans is a program which attempts to make one transition from a given
 * cellular atomaton state (passed as standard input) using a given set of 
 * rules (passed as a text file).  The calling format is: 
 * $ trans [-m|v] [-s 4|8] [-r 'dir_symbols'] rulefile
 * If a complete transition can be made, then the resulting state is printed 
 * to standard out.  Otherwise, the rules that are needed along with the input 
 * state (embellished with markers denoting cells for which no rule applied) 
 * are printed.  Additionally, an error code will be returned signifying 
 * whether a successful transition occurred, so that a Unix script can call 
 * trans repeatedly until it fails.
 */

/* possible improvements:
 *
 * - the errlist which manages skeleton rules could be a hash table.  this
 *   would greatly improve its performance, but its unecessary at present.
 *   fyi - the reason I did not take advantage of hsearch is that it is
 *   limited to one hash table at a time :(
 */

/* to compile trans.c into an executable file called trans type
 * $ cc trans.c -o trans
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <search.h>
#include <varargs.h>


/* for simplicity we allocate a large cellular array and use only a large
 * enough section to accomodate the input state
 */

#define MAXROWS  800
#define MAXCOLS  800
#define MAXRULES 2000

#define FALSE 0
#define TRUE  1 

#define BLANK ' ' 
#define NOMATCH '~' /* a char denoting a cell for which no rule applies */
#define STAR '*' 

#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#define MAX(a,b) (((a) > (b)) ? (a) : (b))

/* function prototypes */
ENTRY *hsearch();

struct state {
  int grid [MAXROWS] [MAXCOLS];
  int rows, cols;
};


/* global vars */
/*************************************************************************/
char *progname;      /* used for error msgs */
char *rulefile;      /* the name of rulefile passed via cmd line */
char *dir_syms;      /* a string containing the direction-oriented symbols */
int   dir_syms_len;  /* the length of dir_syms */

int rule_perms;      /* the number of symmetric permutations of a rule */
enum {Moore, vonNeumann} nbhd_type;

/* the following four vars are required for the hash table library routines */
char rule_lhs_space[MAXRULES*8];
char rule_rhs_space[MAXRULES];
char *rule_lhs_ptr = rule_lhs_space;
char *rule_rhs_ptr = rule_rhs_space;

/* this struct is used to manage the list of rules" that are needed for
 * a successful transition
 */
struct {
  char list[500][10];  
  int  cnt;
} errlist;


/* error management */
/*************************************************************************/

/* error prints an error message and exits the program with errcode 1.
 * incidently, this fctn accepts a variable len arg list.  It employs
 * the macros defined in varargs.h to manage the args.  Exactly HOW
 * this occurs is FM.
 */
void
error(fmt, va_alist)
  char *fmt;
  va_dcl
{
  va_list ap;

  va_start(ap);

  fprintf(stderr, "%s: ", progname);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  exit(1);
}

void
dbg(fmt, va_alist)
  char *fmt;
  va_dcl
{
  va_list ap;

  va_start(ap);

  fprintf(stderr, "%s: ", progname);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
}



/* i/o management */
/*************************************************************************/

/* read a line and return its len */
int
fgetline(line, max, fptr)
  char  *line;
  int    max;
  FILE 	*fptr;
{
  if (fgets(line, max, fptr) == NULL)
    return 0;
  else
    return strlen(line);
}




/* string manipulation */
/*************************************************************************/

/* strrotater rotates the string s one char to the right */
void
strrotater(s)
  char *s;
{
  char c;
  int  n;

  if (*s == '\0') return;  /* do nothing on empty strings */

  n = strlen(s) - 1;
  c = s[n];

  while (n > 0)
    s[n] = s[--n];

  s[0] = c;
} 


void
rotate(c)
  char *c;
{
  char *p;
  int  offset;

  if ((p = strchr(dir_syms, *c)) != NULL) {
    offset = ((p - dir_syms) + 1) % dir_syms_len; 
    *c = dir_syms[offset];
  }
}


/* permute takes a rule and returns the next clockwise permutation of it
 * depending upon the symmetry option specified by the user.
 */
void
permute(s, c)
  char *s, *c;
{
  int n=0;

  /* first for the rotations */
  switch (nbhd_type) {
    case (vonNeumann):
      strrotater(s+1);  /* rotate the substring s[2]..s[n] to the right */
      break;

    case (Moore):
      switch (rule_perms) {
		  /* two rotations are required if 4-fold symmetry is used */
	case (4): strrotater(s+1); strrotater(s+1); break;
	case (8): strrotater(s+1); 	   	    break;
      		  break;
      } 
  }

  /* now we rotate the direction oriented symbols if there are any */
  if (dir_syms != NULL) {
    while (s[n]) 
      rotate(&s[n++]);
    rotate(c);
  }
}


/* errlist management*/
/*************************************************************************/

void
list_init()
{
  errlist.cnt=0;
}

/* list_mem(s) returns TRUE iff s (or any permutation of s) is in errlist */
int
list_mem(s)
  char *s;
{
  int i, j;
  char teststr[10], dummy;

  for (i=0; i < errlist.cnt; i++) {
    strcpy(teststr, errlist.list[i]);
    for (j=0; j < rule_perms; j++) {
      if ( !strcmp(teststr, s) ) return (TRUE);
      else permute(teststr, &dummy);
    }
  }

  return (FALSE);
}


/* list_insert(s) attempts to insert s into errlist.  if s (or any permutation
 * of s is already in errlist, then no insert occurs. 
 */
void
list_insert(s)
  char *s;
{
  if (!list_mem(s) )
    strcpy(errlist.list[errlist.cnt++], s);
}


/* list_print prints the entire contents of errlist to stdout */
void
list_print()
{
  int i;

  for (i=0; i < errlist.cnt; i++)
    printf("%s -> ?\n", errlist.list[i]);
}



/* usage routines */ 
/*************************************************************************/

  
/* improper_usage() aborts with usage msg */
void
improper_usage()
{
  error("usage: %s [-m|v] [-s 4|8] [-r 'dir_symbols'] rulefile\n", progname);
}

/* usage determines whether the command line arguments are sufficient and
 * assigns the arguments to variables for convenience.  usage expects
 * trans [-m|v] [-s 4|8] [-r 'dir_symbols'] rulefile
*/
void
usage(argc, argv)
  int argc;
  char *argv[];
{
  int i;

  progname = argv[0];
  rulefile = NULL;

  if (argc <= 1) improper_usage();

  nbhd_type = vonNeumann; 	/* default value */
  rule_perms = 1; 		/* default value */
  dir_syms = NULL;		/* default value */
  dir_syms_len = 0;		/* default value */

  for (i=1; i < argc; i++) {
    if (argv[i][0] == '-') {
      switch (argv[i][1]) {
        case 'm': nbhd_type = Moore;      break;
        case 'v': nbhd_type = vonNeumann; break;

/* ??? when trans begins supporting 8-fold rule symmetries, this -s and -r
 * code will need to be seriously upgraded
 */
        case 's': if (i+1 == argc) error("-s requires additional argument\n");
		  if ((rule_perms = atoi(argv[++i])) != 4)
		    error("only 4-fold rule symmertry is supported\n"); 
		  break;
	case 'r': if (i+1 == argc) error("-r requires additional argument\n");
 		  dir_syms = argv[++i];
		  if ((dir_syms_len = strlen(dir_syms)) != 4)
		    error("only 4 symbols may be direction-oriented\n");
		  break;
        default: improper_usage(); 
      }
    } else {
        if (rulefile == NULL) rulefile = argv[i];
        else error("expects only one rulefile\n");
    }
  } 
/* ??? some consistency checks will need to occurr here.  eg s -4 requires
 * that if the -r option is used, its arg must be of len 4.
 */
}
 
 
/* state management */
/*************************************************************************/

/* init_state reads the state from stdin and sets up an internal representation
 * in currstate.  it can be the case that the length of the rows in the
 * input state varies, so init_state assumes the length of the shortest row
 * is the number of cols in the input state - in effect clipping the rhs of 
 * the input state.
 */
void
init_state(currstate) struct state *currstate; {
  char ch;
  register int r=0, c=0; /* array indicies */
  int row_len=-1;  	 /* when this value goes non-neg, we know that */
			 /* a legitimate row len has been established  */

  while (( ch = getchar() ) != EOF) {
    /* range check the state */
    if (c >= MAXCOLS-1) 
      error("maximum number of columns permitted is %d\n", MAXCOLS);

    if ( ch == '\n' )  /* reset indicies on newline chars */
		       /* and do some error checking */
    {
      if (row_len < 0) row_len = c;	/* set the row_len on the first row */

      if (c != row_len)
        error("all rows in the CA must be of equal length\n");

      r++; c=0;				/* reset indicies */

      if (r >= MAXROWS-1)
        error("maximum number of rows permitted is %d\n", MAXROWS);
    }
    else
      currstate->grid[r][c++] = ch;
  }

  currstate->cols = row_len;
  currstate->rows = r;
}


/* print_state(s) prints state s to stdout */
void
print_state(s)
  struct state *s;
{
  register int r, c; /* array indicies */

  for (r=0; r < s->rows; r++) {
    for (c=0; c < s->cols; c++) putchar(s->grid[r][c]);
    putchar('\n');
  }
}


/* file management */
/*************************************************************************/

/* efopen attempts to open the given file and aborts elegantly if it cannot */
FILE 
*efopen(file, mode)
  char *file, *mode;
{
  FILE *fp, *fopen();

  if ((fp=fopen(file,mode)) != NULL)
    return(fp);

  error("can't open file %s mode %s\n", file, mode);
}
      

/* hash related fctns */
/*************************************************************************/

/* hinsert(s, c) stores s and c in a hash table as an item - ie we can
 * subsequently lookup the value c with the key s
 */
void
hinsert(s, c)
  char *s, c;
{
  ENTRY item;

  /* begin by placing s in rule_lhs and c in rule_rhs */
  strcpy(rule_lhs_ptr, s);  
  *rule_rhs_ptr = c;

  /* construct item for insertion into the table */
  item.key  = rule_lhs_ptr;
  item.data = rule_rhs_ptr;

  /* plunk it in */
  hsearch(item, ENTER);

  /* update the space pointers */
  rule_lhs_ptr += strlen(s) + 1;
  rule_rhs_ptr++;
}


/* hfound returns TRUE iff s is a key in the hash table */
int
hfound(s)
  char *s;
{
  ENTRY item;

  item.key = s;

  return (hsearch(item, FIND) != NULL);
}

 

/* build_hash reads the rule file and creates a hash table of the rules it
 * contains.  it explicitly creates all symmetric permutations of each rule
 * and stores them in the hash table.  an error is reported if any redundant
 * or conflicting rules are detected.
 */
void
build_hash()
{
  char line[80], c, s[8];
  FILE *fptr;
  int  i;

  hcreate(MAXRULES);

  fptr = efopen(rulefile, "r");

  while (fgetline(line, sizeof(line), fptr) > 0) {
    if (sscanf(line, "%s -> %1s", s, &c) == 2) {
      /* begin with some consistency checks */
      switch (nbhd_type) {
        case (vonNeumann):
          if (strlen(s) != 5)
            error("von Neumann neighborhoods require rules with lhs of len 5\n");
          break;
  
        case (Moore):
          if (strlen(s) != 9)
            error("Moore neighborhoods require rules with lhs of len 9\n");
          break;
      }
  
      if (hfound(s))
        error("redundant or conflicting rule found: %s -> %s\n", s, &c);
      else {
        hinsert(s, c);
         
        for (i=1; i < rule_perms; i++) {
          permute(s, &c); 
          hinsert(s, c);
        }
      }

    } /* end if scan */
  }   /* end while */

  fclose(fptr);
}


/* this is a procedure for testing whether the hash table is being created
 * correctly.  it is for debugging purposes only.
 */
 void
 test()
 {
   ENTRY item, *found_item;
   char name_to_find[30];

   item.key = name_to_find;

   while (scanf("%s", item.key) != EOF) {
     if ((found_item = hsearch(item, FIND)) != NULL) 
        printf("found %s -> %c\n", found_item->key, *found_item->data);
     else 
        printf("no such rule %s\n", name_to_find);
   }
 }



/* miscellaneous */
/*************************************************************************/


/* here we define two macros to assist in extracting elements from the grid.
 * MOD(a,b) implements a restricted form of a real modulus fctn (analogous to
 * the %% operator in C*).  It is restricted in that it assumes that b is
 * non-negative, but this is sufficient for our purposes.  T(r,c) (which stands
 * for Toroidal(r,c) ) employs the MOD macro and exists primarily to reduce 
 * typing and improve readability.
 */

#define MOD(a, b)  ( ((a) < 0) ? ((b) + (a)) : ((a) % (b)) )
#define T(r,c) s->grid[MOD((r), s->rows)] [MOD((c), s->cols)]


/* build_lhs returns in lhs a string corresponding to the neighborhood of cell 
 * (r,c) of state s.
 */ 
void build_lhs(s, lhs, r, c)
  struct state *s;
  char *lhs;
  int r, c;
{
  switch (nbhd_type) {
    case (Moore): 
    		lhs[0] = T(r,   c);	/* center */
		lhs[1] = T(r-1, c);	/* north */
		lhs[2] = T(r-1, c+1);	/* northeast */
		lhs[3] = T(r,   c+1);	/* east */
		lhs[4] = T(r+1, c+1);	/* southeast */
		lhs[5] = T(r+1, c);	/* south */
		lhs[6] = T(r+1, c-1);	/* southwest */
		lhs[7] = T(r,   c-1);	/* west */
		lhs[8] = T(r-1, c-1);	/* northwest */
		lhs[9] = '\0';		/* end of str */
		break;

    case (vonNeumann): 
    		lhs[0] = T(r,   c);	/* center */
		lhs[1] = T(r-1, c);	/* north */
		lhs[2] = T(r,   c+1);	/* east */
		lhs[3] = T(r+1, c);	/* south */
		lhs[4] = T(r,   c-1);	/* west */
		lhs[5] = '\0';		/* end of str */
		break;
  }
}


/* transition attempts to use the given rule set to make a transition from 
 * state s1 * and record the result in s2.  if cells are detected for which 
 * no rule * applies, the lhs of the rule that is needed is stored in errlist 
 * for * later diagnostic output.  transition returns TRUE iff it makes a 
 * complete transition - ie every cell has a rule which applies.
 */
int
transition(s1, s2) 
  struct state *s1, *s2;
{
  register int r,c;
  int succ=TRUE;
  ENTRY item, *found_item;
  char lhs[10];

  /* make s2 have the same dimensions as s1 */
  s2->rows = s1->rows;
  s2->cols = s1->cols;

  for (r=0; r < s1->rows; r++) {
   for (c=0; c < s1->cols; c++) {
     build_lhs(s1, lhs, r, c); /* build a key for the hash table */
     item.key = lhs;

     if ((found_item = hsearch(item, FIND)) != NULL) 
       s2->grid[r][c] = *found_item->data; /* set cell according to rule */ 
     else {
       succ = FALSE; /* record that we failed */
       s2->grid[r][c] = NOMATCH; /* store a marker indicating the bad cell */
       list_insert(lhs); /* record the lhs of the rule that is required */
     };
    }
  }

  return(succ);
}


/* if a complete transition cannot be made, diagnostics are printed to
 * facilitate the creation of rules that will be sufficient to make the 
 * transition.
 */
void   
print_diagnostics(s1, s2)
  struct state *s1, *s2;
{
  register int r, c;
  int rowflag=FALSE, colflags[MAXCOLS];

  list_print(); /* print out the errlist */

  /* now print out the initial state with markers denoting which cells bombed */

  /* first we must initialize an array which will be used to print markers
   * beneath columns which have defective cells
   */
  for (c=0; c < s1->cols; c++)
    colflags[c] = FALSE;

  for (r=0; r < s1->rows; r++, rowflag=FALSE) {
    for (c=0; c < s1->cols; c++) {
      putchar(s1->grid[r][c]);
      if (s2->grid[r][c] == NOMATCH) {
        rowflag=TRUE;
        colflags[c]=TRUE;
      }
    }
    if (rowflag) putchar(STAR);
    putchar('\n');
  }

  /* now print the column indicators */

  for (c=0; c < s1 -> cols; c++) {
    if (colflags[c]) putchar(STAR);
    else putchar(BLANK); 
  }
  putchar('\n');
}


void
main(argc, argv)
  int argc;
  char *argv[];
{
  struct state currstate, nextstate;

  list_init(); /* initialize the errlist */

  usage(argc, argv); /* get cmd line args and check for correct usage */

  build_hash(); /* create hash table of rules */

  init_state(&currstate); /* initialize currstate to that on stdin */

  /* if a transition can be made then print the resulting state and return
   * success.  otherwise print diagnostics and return failure.
   */
  if ( transition(&currstate, &nextstate) ) {
    print_state(&nextstate);
    exit(0);
  }
  else {
    print_diagnostics(&currstate, &nextstate);
    exit(1);
  }
}




