#ifndef DATMO_UTILS_H
#define DATMO_UTILS_H

#include <utils/Linear.h>
#include <utils/SquareMatrix.h>

using namespace utils;

// Position vectors and covariances.
static const int POINT_ORDER = 2;
typedef SquareMatrix<double, POINT_ORDER> PointMat;
typedef Vec2d PointVec;

// Position + velocity vectors, covariances and systems.
static const int FILTER_ORDER = 6;
typedef LinearVector<double, FILTER_ORDER> FilterVec;
typedef SquareMatrix<double, FILTER_ORDER> FilterMat;


static inline double fsqr (double x) {
  return x * x;
}

static inline double signum (double x) {
  return x < 0 ? -1 : 1;
}

// Return length squared.  More efficient than length when we are just
// comparing magnitudes, and sometimes we just want the squared magnitude.
static inline double length2 (const Vec2d &x) {
  return x.dot(x);
}

// Return the point closest to p on the line defined by base and direction
// (normalized.)
static inline Vec2d closestPoint (const Vec2d &p, const Vec2d &base, const Vec2d &direction) {
  Vec2d p1p = p - base;
  return (direction * direction.dot(p1p)) + base;
}    

// return the point on segment p1 p2 that is closest to p
static inline Vec2d segClosestPoint
    (const Vec2d &p, const Vec2d &p1, const Vec2d &p2)
{
  Vec2d p1p = p - p1;
  Vec2d displacement = p2 - p1;
  if (displacement.dot(p1p) < 0)
    return p1;
  if (displacement.dot(p - p2) > 0)
    return p2;
  else {
    double mag = displacement.length();
    if (mag < 1e-6)
      return p1;
    else {
      Vec2d dir = displacement * (1/mag);
      return (dir * dir.dot(p1p)) + p1;
    }
  }
}    

// Return vector rotated by 90 degrees counter-clockwise.
static inline Vec2d normal (const Vec2d &v) {
  return Vec2d(-v[1], v[0]);
}

// Set columns of PointMat from two basis vectors.
static inline void set_cols (PointMat &m,
			     const PointVec &c0, const PointVec &c1) 
{
  double *m_dat = m.getValue();
  const double *c0_dat = c0.getValue();
  ((double (*)[2])m_dat)[0][0] = c0_dat[0];
  ((double (*)[2])m_dat)[1][0] = c0_dat[1];
  const double *c1_dat = c1.getValue();
  ((double (*)[2])m_dat)[0][1] = c1_dat[0];
  ((double (*)[2])m_dat)[1][1] = c1_dat[1];
}

double angle_diff (const Vec2d &d1, const Vec2d &d2);
void set_rotation (PointMat &m, double rot_by);
void set_rotated_covariance (PointMat &res,
			     const Vec2d &dir, const Vec2d &norm,
			     double along_sigma, double norm_sigma,
			     double max_elong);
void calc_eigenvalues (double A[2][2], double W[2]);
void calc_eigenvectors (double A[2][2], double Z[2][2], double W[2]);
void print_evals (ostream &out, const PointMat &m);
void get_submatrix_eigenvalues (double W[2], const FilterMat &m, int offset = 0);

#endif
