#include "utils.h"

// Return the angular distance from direction vector d1 to d2.
double angle_diff (const Vec2d &d1, const Vec2d &d2) {
  double dp = normal(d1).dot(d2);
  // Prevent float trap in case asin arg slightly exceeds 1.
  double d_theta = fabs(dp) > 0.99999 ? M_PI/2 * signum(dp) : asin(dp);
  if (d1.dot(d2) < 0)
    return (signum(d_theta) * M_PI - d_theta);
  else
    return d_theta;
}


// Init M to be a 2d rotation matrix.
void set_rotation (PointMat &m, double rot_by) {
  double ct = cos(rot_by);
  double st = sin(rot_by);
  m.at(0, 0) = ct;
  m.at(0, 1) = -st;
  m.at(1, 0) = st;
  m.at(1, 1) = ct;
}


// Create a rotated covariance matrix given the basis vectors for the
// unrotated coordinate system and the uncorrelated standard deviations.
// max_elongation sets a limit on how skinny the distribution can be.
//
void set_rotated_covariance (PointMat &res,
			     const Vec2d &dir, const Vec2d &norm,
			     double along_sigma, double norm_sigma,
			     double max_elong) 
{
  // rot * cov * rot'
  PointMat rot;
  set_cols(rot, dir, norm);
  double as2 = fsqr(along_sigma);
  double ns2 = max(fsqr(norm_sigma), as2/max_elong);
  res.clear();
  res.at(0, 0) = as2;
  res.at(1, 1) = ns2;
  res.mult(rot, res);
  rot.transpose(rot);
  res.mult(res, rot);

#if 0
  PointMat dum;
  ASSERT(dum.spd_inverse(res), "Covariance not SPD!");
#endif
}



//// 2d eigenvalues/eigenvectors

// This was adapted from the Gandalf vision libary, and should be put in a
// file with the right license.

// Set W to the eigenvalues of A.  W is sorted in increasing order.
void calc_eigenvalues (double A[2][2], double W[2]) {
  double fDisc = sqrt(4.0F*A[1][0]*A[1][0] + fsqr(A[0][0]-A[1][1]));
  W[0] = 0.5F*(A[0][0] + A[1][1] - fDisc);
  W[1] = 0.5F*(A[0][0] + A[1][1] + fDisc);
  if (W[0] > W[1]) {
    double temp = W[0];
    W[0] = W[1];
    W[1] = temp;
  }
}

// Find eigenvalues and eigenvectors of A.
void calc_eigenvectors
    (double A[2][2], double Z[2][2], double W[2])
{
  calc_eigenvalues(A, W);

  double dTheta, dCosT, dSinT;

  if ( A[1][0] == 0.0 && A[0][0] == A[1][1] )
    dTheta = 0.0;
  else
    dTheta = 0.5*atan2(-2.0*A[1][0], A[1][1]-A[0][0]);
      
  dCosT = cos(dTheta);
  dSinT = sin(dTheta);
  Z[0][0] = dCosT;
  Z[0][1] = -dSinT;
  Z[1][0] = dSinT;
  Z[1][1] = dCosT;
}


// Print the root of the trace, the elongation in the direction of the
// principal component, and the direction of that component.
void print_evals (ostream &out, const PointMat &m) {
  double cov[2][2];
  m.getValue(cov);
  double evec[2][2];
  double eval[2];
  calc_eigenvectors(cov, evec, eval);
  out <<sqrt(cov[0][0] + cov[1][1]) <<" "
      <<(fabs(eval[0]) < 1e-9 ? 0 : eval[1]/eval[0])
      <<" [" <<evec[0][1] <<" "  <<evec[1][1] <<"]";
}


// Get eigenvalues of a 2x2 submatrix.
void get_submatrix_eigenvalues
    (double W[2], const FilterMat &m, int offset)
{
  double cov[2][2];
  for (int i = 0; i < 2; i++)
    for (int j = 0; j < 2; j++)
      cov[i][j] = m.at(i + offset, j + offset);
  
  calc_eigenvalues(cov, W);
}


// for use in gdb.
void print_mat (FilterMat *m) {
  cerr.precision(5);
  cerr <<*m;
}
