// Detection And Tracking of Moving Objects (in LIDAR data)
//
// Author: Robert A MacLachlan
//
// Earlier DATMO implementations were written by Justin Carlson and 
// Chieh-Chih Wang.  

#ifndef DATMO_H
#define DATMO_H

#include <VehState/VehState.h>
#include <ConfigSource/ConfigSource.h>
#include <Display/Display.h>
#include <vector>
#include <fstream>
#include "ObjectTrack.h"
#include "Segment.h"

using namespace std;
using namespace utils;

/*
 * Bounding box in vehicle coordinates of an area we want to ignore
 */
typedef struct {
        double x0;
        double y0;
        double x1;
        double y1;
} datmo_bounding_box_t;
  

class Datmo
{
  friend class ObjectBase;
  friend class ObjectTrack;
  friend class Segment;
  friend class TrackLogger;
  friend class PointLogger;

public:
  Datmo (VehState *vs, ConfigSource *config,
	 const vector<datmo_bounding_box_t> &ignored_spaces,
	 vector<ScannerInfo *> &scanners,
	 Display *display, ostream *scan_log);
  ~Datmo();

  // Group a bunch of add_scan calls on distinct scanners that occur at
  // approximately the same time
  void begin_scans ();

  // Add a single scan.
  bool add_scan(ScannerInfo *scanner);

  // Bookend to begin_scans()
  void end_scans ();

  vector<ObjectTrack *> &get_tracks();

  static void render_callback(double e_o, double n_o, double low_x, double low_y,
			      double high_x, double high_y,
			      utils::SymbolTable *st, void *data);
  void render(double e_o, double n_o);

private:
  // VehState source.
  VehState *_vs;

  // Debug display.
  Display *_display;

  // Set after initialization on first call to add_scan.
  bool _inited;

  // Areas (in vehicle coordinates) that we just want to ignore
  vector<datmo_bounding_box_t> _ignored_spaces;

  // Vector of all scanners.
  vector<ScannerInfo *> _scanners;

  // ScannerInfo for the scanner used in current add_scan call.
  ScannerInfo *_scanner;

  // Our iteration number.
  unsigned int _iteration;

  // Time of first scan, previous scan.  First scan is used as time 0 in debug
  // output.  Previous scan is used to find the time increment to advance by
  // in ObjectTrack::update.
  double _first_scan_time;
  double _prev_scan_time;

  // Time of current scan, and also (after update has run) the timestamp for
  // all track state, since we update the state to this time.
  double _scan_time;

  // Counter used to generate track IDs.
  unsigned int _next_track_id;

  // Temporary data structures we reuse on each iteration, _max_scan_size in
  // size.

  // Time of each reading in this scan.
  utils::Vector<Time> _times;

  // State at each reading.
  utils::Vector<VehStateStruct> _states;

  // Temporary used during segmentation.  For each point, 1 if it has been
  // assigned to some segment yet, 0 otherwise.
  unsigned char *_used;

  // Represents segmented sets of points to be associated with a track.
  list<Segment *> _segments;

  // At the end of each iteration, the contents of _segments is appended to
  // this list, which is cleared out at datmo::time_step().  This is done for
  // visualization (so we can see segments from all scanners) but also affects
  // allocation, since segments can't be deleted until then.
  list<Segment *> _segment_log;

  // Segment sets are used during model-based segmentation.
  vector<SegmentSet *> _segment_sets;

  // Current tracks.  This list is sorted by increasing track_id because new
  // tracks are always added at the end.
  list<ObjectTrack *> _tracks;

  // Tracks that are effectively dead, and are going to be deleted on the next
  // iteration.
  list<ObjectTrack *> _dead_tracks;

  // Output tracks.
  vector<ObjectTrack *> _output_tracks;

  // If we are displaying ignored points, save them here.
  vector<PointData> *_ignored_points;

  // If non-null, log debug info related to scanner delta_t.
  ostream *_scan_log;

  void init();
  bool in_ignore_region (const LineScanElem &p);
  void ignore_point (const PointData &point);
  void ignore_points (const vector<PointData> &points);
  void segment ();
  bool import_data (ScannerInfo *scanner);
  void find_segment_sets ();
     
  Segment *closest_segment (Segment *prev_seg, unsigned int prev_ix,
			    const PointVec &p, SegmentSet *sset);
  void model_segment ();
  void model_segment_1 (SegmentSet *sset);
  bool model_segment_ok (SegmentSet *sset);

  void split_tracks (SegmentSet *sset, Segment *seg, unsigned int gap_pos);
  void find_overlaps ();
  void associate_tracks ();

  void reap_tracks ();
  void update_tracks ();

  // If non-zero, then we didn't validate all moving tracks on the last pass,
  // and this is the ID of the last validated.  Start after this one on the
  // next cycle.
  unsigned int _last_validated;
  void validate_tracks ();

  void log_ignored ();
  void ft_kill_track (ObjectTrack *track);

  /* not really */ public: 
  void _dump_data ();

};
#endif
