#include "Segment.h"
#include "gsl_fit.h"
#include "params.h"

Segment::Segment (Datmo *datmo_arg, ScannerInfo *scanner_arg,
		  const vector<PointData> &points_arg) :
  ObjectBase(datmo_arg, scanner_arg, SEGMENT),
  squared_fit_quality(0),
  split_from(NULL),
  corner_ix(0)
{
  points = points_arg;
}

Segment::Segment (ObjectBase *track) :
  ObjectBase(track->datmo, track->scanner, SEGMENT),
  squared_fit_quality(0),
  split_from(NULL),
  corner_ix(0)
{
  copy_features(track, true);
  best_match = track;
  track->best_match = this;
}

Segment::~Segment () {
  kind = DELETED;
};


// Buffers used by line/corner fitting (see fit_linear/fit_corner.)  These are
// preallocated at _max_scan_size in Segment::static_init.
//
// Points to fit.
PointVec *Segment::_fit_buf;
//
// Point weights, normalized so that the sum of the weights is the number of
// points.
double *Segment::_fit_weight;
//
// Unweighted fit errors for each point, and buffer for sorting so that we can
// find the outlier points.
double *Segment::_fit_error;
double *Segment::_fit_error_sorted;

// Init Segment static vars.
void Segment::static_init () {
  _fit_buf = new PointVec[_max_scan_size];
  _fit_error = new double[_max_scan_size];
  _fit_weight = new double[_max_scan_size];
  _fit_error_sorted = new double[_max_scan_size];
}



//// find_gap
//
// Find the biggest gap in a segment (a potential split point.)  gap_pos is
// index in points of the first point after the gap.  gap_size is the size of
// the gap normalized by the appropriate limit parameter.  Values >1 indicate
// that the limit is exceeded.
void Segment::find_gap (unsigned int &gap_pos, double &gap_size) {
  ASSERT(points.size() >= 2, "Less than 2 points?");
  gap_pos = 0;
  gap_size = 0;

  for (unsigned int seg_ix = 1; seg_ix < points.size(); seg_ix++) {
    const PointData &pd_prev = points[seg_ix-1];
    const PointData &pd = points[seg_ix];
    float this_rng = min(pd_prev.range, pd.range);
    if (pd.scan_pos != pd_prev.scan_pos + 1) {
      bool hole = false;
      bool occlusion = false;
      for (unsigned int i = pd_prev.scan_pos + 1; i < pd.scan_pos; i++) {
	float rng = scanner->points[i].range;
	if (rng < _range_limit) {
	  if (rng < this_rng)
	    occlusion = true;
	  else
	    hole = true;
	}
      }
      double size = (pd_prev.pos - pd.pos).length();
      
      if (hole) {
	size /= _max_hole_size;
      } else if (occlusion) {
	size /= _max_occlusion_size;
      } else {
	size /= _max_dropout_size;
      }
      
      if (size > gap_size) {
	gap_size = size;
	gap_pos = seg_ix;
      }
    }

    double step = fabs(pd_prev.range - pd.range) / _max_step_size;
    if (step > gap_size) {
      gap_pos = seg_ix;
      gap_size = step;
    }
  }
}


//// good_new_track
//    
// Return true if this segment is a good basis for a new track.  We require
// that the track:
//  1] have enough non-occluded points.  This rejects some mangy tracks that
//     may cause problems later.
//  2] usefully localize in two dimensions.  If due to occlusion of vague line
//     ends, a track doesn't localize, then it will later on fail the info
//     increment test in data association.  But if we drop it now, we avoid a
//     lot of churn in creating and deleting new tracks.  The main reason for
//     doing this test is for the esthetics of debug display, and not churning
//     through lots of track IDs.
//
bool Segment::good_new_track () {
  int np = _min_track_points;
  for (unsigned int i = 0; i < points.size(); i++)
    if (!points[i].occluded && !--np) break;
  if (np) return false;

  // ### DEBUG
  if (_model_based_segmentation) return true;

  switch (shape_class) {
  case COMPLEX:
    {
      const PointMat &minc = features[MIN_BOUND].residue_covariance;
      const PointMat &maxc = features[MAX_BOUND].residue_covariance;
      double vague_cov = fsqr(_vague_pos_noise);
      for (int i = 0; i < 2; i++)
	if (minc.at(i, i) < vague_cov || maxc.at(i, i) < vague_cov)
	  return true;
      return false;
    }
      
  case LINE:
    return !(features[FIRST].vague && features[LAST].vague);

  case CORNER:
    return true;

  default:
    return true; // can't happen, keeps gcc happy.
  }
}

//// Segment::init -- feature extraction:

// Find the bounding box of a segment and the covariances of the bounds.
// After finding covariance based on whether an occluded point contributes to
// the position of a bound, we then add a contribution from the scanner
// angular resolution limit.
//
// We assume a point spacing along a line normal to the scan ray (which is
// actually the best case, not conservative.)  In principle we could use the
// line info if this is a linear segment, but that probably isn't desirable,
// as this would have the effect of increasing the noise when a segment goes
// from COMPLEX to linear, which seems wrong.
//
void Segment::find_bounds () {
  ASSERT(!points.empty(), "No points in segment?");
  FeaturePoint &minf = features[MIN_BOUND];
  FeaturePoint &maxf = features[MAX_BOUND];
  minf.residue_covariance.clear();
  maxf.residue_covariance.clear();
  minf.init(points[0]);
  maxf.init(points[0]);

  for (unsigned int j = 1; j < points.size(); j++) {
    double var = (points[j].occluded
		  ? fsqr(_max_segment_distance * _point_spacing_sigma_factor)
		  : fsqr(_disoriented_pos_noise));
    for (int i = 0; i < 2; i++) {
      double val = points[j].pos[i];
      if (val < minf.state[i]) {
	minf.state[i] = val;
	minf.residue_covariance.at(i, i) = var;
      }

      if (val > maxf.state[i]) {
	maxf.state[i] = val;
	maxf.residue_covariance.at(i, i) = var;
      }
    }
  }

  // Bound vague not really used currently.  We set it if an occluded point
  // determined part of that bound.
  for (int i = 0; i < 2; i++) {
    if (minf.residue_covariance.at(i, i) >= fsqr(_vague_pos_noise))
      minf.vague = true;
    if (maxf.residue_covariance.at(i, i) >= fsqr(_vague_pos_noise))
      maxf.vague = true;
  }

  // Angular resolution related uncertainty.
  Vec2d scan_dir =  points[0].pos - scanner->location; // scanner ray
  double range = scan_dir.length();
  scan_dir.normalize();
  Vec2d scan_norm = normal(scan_dir);
  // point spacing on line normal to scan ray
  double lateral = range * sin(scanner->resolution());
  double along_sigma = lateral * _point_spacing_sigma_factor;

  PointMat res_cov;
  set_rotated_covariance(res_cov, scan_norm, scan_dir, along_sigma, 0,
			 _max_disoriented_pos_noise_elongation);
  minf.residue_covariance.add(minf.residue_covariance, res_cov);
  maxf.residue_covariance.add(maxf.residue_covariance, res_cov);
}


// Find the point between the points at FIRST and LAST which lies farthest off
// the line defined by those two points.  Return the index of the farthest point.
void Segment::find_knuckle (int first, int last, int &max_ix) {
  assert(first+1 < last); // at least three points.
  double max_err_square = -1;
  max_ix = -1;
  Vec2d p1 = points[first].pos;
  Vec2d p2 = points[last].pos;
  Vec2d displacement = p2 - p1;
  Vec2d dir(displacement);
  dir.normalize();

  int npoints = 0;
  for (int i = first + 1; i < last; i++) {
    Vec2d p = points[i].pos;
    Vec2d p1p = p - p1;
    Vec2d cp = (dir * dir.dot(p1p)) + p1;
    Vec2d err_vec = p - cp;
    double err_square = err_vec.dot(err_vec);
    npoints++;
    if (err_square > max_err_square) {
      max_err_square = err_square;
      max_ix = i;
    }
  }
}


// Fill fit_buf with npoints points starting at start, then find weights.  
// The fit is weighted by the distance between points so that we are not
// influenced by varying point density.  We normalize the result weights so
// that the weight sum is the number of points.  Normalizing the weights makes
// the magnitude of the fit error independent of the point spacing.
//
// So that the weights are not inflated for outlier points, we want to
// only consider the distance along the line when finding the weight.
// But of course, we don't know the line yet.  So we first do an
// equal-weight line fit.
//
void Segment::fill_fit_buf (int start, int npoints) {
  // fill buffer
  for (int i = 0; i < npoints; i++) {
    _fit_buf[i] = points[i + start].pos;
    _fit_weight[i] = 1;
  }

  double ignore;
  Vec2d base, dir;
  fit_linear_aux(npoints, ignore, base, dir);

  // find weights
  double weight_sum = 0;
  double prev_delta = 0;
  for (int i = 0; i < npoints - 1; i++) {
    Vec2d nd2 = _fit_buf[i + 1] - _fit_buf[i];
    double next_delta = nd2.dot(dir);
    double w = max(next_delta, prev_delta);
    _fit_weight[i] = w;
    weight_sum += w;
    prev_delta = next_delta;
  }
  _fit_weight[npoints - 1] = prev_delta;
  weight_sum += prev_delta;

  double norm = npoints/weight_sum;
  for (int i = 0; i < npoints; i++)
    _fit_weight[i] *= norm;
}


// Fit a line to the npoints points in _fit_buf, returning the result as a
// base and dir(ection).  The sign of the direction points from the first to
// last point in the buffer.  The base always lies on the X or Y axis, and
// doesn't reflect the actual point location.
//
// gsl_fit_wlinear does badly when the correct result is near vertical, so we
// try to hack around this by swapping X and Y when the feature seems more
// vertical than horizontal, as determined by the slope of the line from start
// to end.
//
void Segment::fit_linear_aux
    (int npoints, double &sumsq, Vec2d &base, Vec2d &dir) 
{
  assert(npoints >= 2);
  Vec2d disp = _fit_buf[npoints-1] - _fit_buf[0];
  bool swapped = false;
  double *px = &_fit_buf[0][0];
  double *py = &_fit_buf[0][1];
  if (fabs(disp[1]) > fabs(disp[0])) {
    swapped = true;
    double *tmp = py;
    py = px;
    px = tmp;
  }
    
  double c0, c1, cov_00,  cov_01, cov_11;
  int stride = sizeof(Vec2d)/sizeof(double);
  assert(sizeof(Vec2d) % sizeof(double) == 0);
  gsl_fit_wlinear(px, stride, _fit_weight, 1, py, stride, npoints,
		  &c0 , &c1, &cov_00,  &cov_01, &cov_11, &sumsq);

  base = Vec2d(0, c0);
  dir = Vec2d(1, c1);
  dir.normalize();

  if (swapped) {
    base = Vec2d(base[1], base[0]);
    dir = Vec2d(dir[1], dir[0]);
  }

  if (disp.dot(dir) < 0)
    dir = dir * -1;
}
  

static int sort_order(const void *x, const void *y) {
  return *(double *)x > *(double *)y ? -1 : 1;
}


// Given line defined by base and normal, drop the points with the highest fit
// error w.r.t. that line.  npoints is the new number of points in the fit
// buffer (after dropping.)  We renormalize the weights after dropping.
//
// When deciding which points to drop, we use the non-weighted error so
// that we aren't biased in favor of throwing out the most important points.
// 
// Normally npoints == full_npoints - ndrop, but we might drop fewer points
// when there are points with exactly the same error.
//
void Segment::fit_drop_outliers
    (const Vec2d &base, const Vec2d &norm, int full_npoints, int &npoints)
{
  int ndrop = (int)(full_npoints * _line_discard_fraction);
  if (ndrop == 0) {
    npoints = full_npoints;
    return;
  }

  // find squared fit error for each point.
  for (int i = 0; i < full_npoints; i++) 
    _fit_error[i] = fsqr(norm.dot(_fit_buf[i] - base));

  // sort in descending order and find cut point.
  memcpy(_fit_error_sorted, _fit_error, full_npoints * sizeof(double));
  qsort(_fit_error_sorted, full_npoints, sizeof(double), sort_order);
  double max_ok = _fit_error_sorted[ndrop];

  npoints = 0;
  double w_sum = 0;
  for (int i = 0; i < full_npoints; i++)
    if (_fit_error[i] <= max_ok) {
      _fit_buf[npoints] = _fit_buf[i];
      double w = _fit_weight[i];
      _fit_weight[npoints] = w;
      w_sum += w;
      npoints++;
    }
  
  double nf = npoints/w_sum;
  for (int i = 0; i < npoints; i++)
    _fit_weight[i] *= nf;
}


// Fit a line to the subsequence of our points from start to end.  To improve
// fitting with outliers, we after an initial fit, we throw out the worst x%
// of the points and then refit.
//
// sumsq is the squared error of the final fit, and npoints is the number of
// points in the final fit.
void Segment::fit_linear
    (int start, int end, double &sumsq, int &npoints, Vec2d &base, Vec2d &dir)
{
  int full_npoints = end - start;
  fill_fit_buf(start, full_npoints);

  // trial linear fit.
  fit_linear_aux(full_npoints, sumsq, base, dir);

  // drop outlier points.
  fit_drop_outliers(base, normal(dir), full_npoints, npoints);

  // Fit on point subset.
  fit_linear_aux(npoints, sumsq, base, dir);
}


// To find the side leg position, we take the weighted mean position of the
// other leg's points in the coordinate system of the main leg.  The component
// along the main leg locates the corner, and the component normal to the main
// leg determines which direction the other leg points in.  The length of the
// leg is the maximum component in that direction.  How I'm doing the
// transform is by taking the dot product of each point's offset with the
// corner's two basis vectors.  This is equivalent to 2x2 matrix
// multiplication.
//
// The direction of the side is determined by the sign of the mean normal
// component of the point positions.  The sign of norm is flipped to point in
// this direction.  corner_pos is the output corner position.  side_mag is the
// length of the side, as determined by the maximal point in the direction
// that the side points.
//
void Segment::fit_side
    (int npoints, const Vec2d &big_pos, const Vec2d &dir_big, Vec2d &norm,
     Vec2d &corner_pos, double &side_mag)
{
  double along_mean = 0;
  double norm_sum = 0;
  double norm_min = 0;
  double norm_max = 0;
  for (int i = 0; i < npoints; i++) {
    Vec2d translated = _fit_buf[i] - big_pos;
    double w = _fit_weight[i];
    double along = translated.dot(dir_big);
    along_mean += along * w;

    double np = translated.dot(norm);
    norm_sum += np * w;
    if (np < norm_min) norm_min = np;
    if (np > norm_max) norm_max = np;
  }
  along_mean /= npoints;
  corner_pos = big_pos + dir_big * along_mean;
  side_mag = norm_max;

  // flip normal if it points the wrong way.
  if (norm_sum < 0) {
    side_mag = -norm_min;
    norm = norm * -1;
  }
}


// To fit a corner, we do a linear fit on the big side, then find out which
// right-angle leg would minimize the least-square error with the other leg's
// points.
//
// Before finding the side leg right-angle fit, we do a linear fit on the
// short side to see what the best-fit corner angle would be.  If the best_fit
// angle is not far enough from parallel, then it is a bad fit, and we fail.
//
// As for fit_linear, we do a trial fit of the side, then drop outlier points,
// then refit.
//
// If successful, we also set the direction and norm_dir in the Segment to
// our basis vectors.
//
// mean_square_err is the mean-square error (already normalized by the
// number of points used.)
// 
void Segment::fit_corner 
    (int big_start, int big_end, int small_start, int small_end,
     Vec2d &corner_pos, Vec2d &big_pos, Vec2d &small_pos, double &mean_square_err,
     bool &success)
{
  if ((big_end - big_start) < 2 || (small_end - small_start) < 2) {
    success = false;
    return;
  }
  // first do linear fit on the big side.
  Vec2d base_big, dir_big;
  double sumsq_corner;
  int npoints_corner;
  fit_linear(big_start, big_end, sumsq_corner, npoints_corner, base_big, dir_big);
  if (big_start < small_start) // flip to point away from corner
    dir_big = dir_big * -1;
  big_pos = closestPoint(big_pos, base_big, dir_big);

  // linear fit the small side for angle test.
  Vec2d best_base_small, best_direction_small;
  double ignore;
  int small_full_npoints = small_end - small_start;
  fill_fit_buf(small_start, small_full_npoints);
  fit_linear_aux(small_full_npoints, ignore, best_base_small,
		 best_direction_small);

  // good is < because it's the cos(min)
  success = fabs(best_direction_small.dot(dir_big)) < _cos_min_corner_angle;
  if (!success) return;
    
  // Now figure out where the other right-angle side should be.
  Vec2d norm = normal(dir_big);
  
  // Trial fit side.  Also, this determines the final side length, since we
  // want to use all points for the max operation.
  double side_mag;
  fit_side(small_full_npoints, base_big, dir_big, norm, corner_pos, side_mag);

  // drop outlier points.
  int small_npoints;
  fit_drop_outliers(corner_pos, dir_big, small_full_npoints, small_npoints);

  // Re-fit side.
  fit_side(small_npoints, base_big, dir_big, norm, corner_pos, ignore);
  small_pos = corner_pos + norm * side_mag;

  // find weighted squared fit error for each point.
  for (int i = 0; i < small_npoints; i++)
    sumsq_corner += fsqr(dir_big.dot(_fit_buf[i] - corner_pos));

  npoints_corner += small_npoints;

  mean_square_err = sumsq_corner/npoints_corner;

  // direction is always for center->first, regardless of big/small.
  if (big_start > small_start) {
    direction = norm;
    norm_dir = dir_big;
  } else {
    direction = dir_big;
    norm_dir = norm;
  }
}


// Try to fit as a corner, and if the fit is reasonable do so.  Return true if
// we succeed.  To fit as a corner, we must pass these tests:
//  -- Angle between linear fit on sides sharp enough (see fit_corner).
//  -- Corner mean-squared error less than line.
//  -- Corner convex.
//
// The corner is disoriented if either:
//  -- The fit error is too large, or
//  -- there are not enough points, or
//  -- line/corner fit differ significantly in direction, and the corner MSE
//     is not much better than line.
//
// What is a convex corner?  It means that the corner is closer to the scanner
// than any point on the two sides.  This is not a necessary criterion for
// being a convex corner, but is necessary for *seeing* both sides of a convex
// corner (as long as it is a 90 degree corner.)  The equivalent test
// used here is that the line from the scanner to the corner makes an obtuse
// angle with both sides.
//
// Why only convex?  Well, in this application moving concave corners are very
// rare.
bool Segment::try_fit_corner (unsigned int eff_ignore,
			      double mse_line, const Vec2d &dir_line) 
{
  // Find corner fit.  First find the knuckle point, then fit the two sides.
  find_knuckle(eff_ignore, points.size() - 1 - eff_ignore, corner_ix);

  // After some massaging, these will be the ends and corner of the corner
  // feature.
  PointData pd_first_corner = points[eff_ignore];
  PointData pd_last_corner = points[points.size() - 1 - eff_ignore];
  PointData pd_corner = points[corner_ix];

  double mse_corner;
  bool success;
  unsigned int big_npoints;
  // The bigger side gets to determine the orientation of the corner, so find
  // it.
  if (length2(pd_corner.pos - pd_first_corner.pos)
      > length2(pd_corner.pos - pd_last_corner.pos)) {
    fit_corner(eff_ignore, corner_ix, corner_ix, points.size() - eff_ignore,
	       pd_corner.pos, pd_first_corner.pos, pd_last_corner.pos, mse_corner,
	       success);
    big_npoints = corner_ix + 1;
  } else {
    fit_corner(corner_ix, points.size() - eff_ignore, eff_ignore, corner_ix,
	       pd_corner.pos, pd_last_corner.pos, pd_first_corner.pos, mse_corner,
	       success);
    big_npoints = points.size() - corner_ix;
  }
  if (!success) return false;

  mse_corner *= _corner_weight;

  Vec2d p1p = scanner->location - pd_corner.pos; // dir of scanner at corner
  bool corner_ok =
    (success && mse_corner < mse_line
     && (pd_first_corner.pos - pd_corner.pos).dot(p1p) < 0
     && (pd_last_corner.pos - pd_corner.pos).dot(p1p) < 0);
  
  if (corner_ok && mse_corner < fsqr(_disoriented_line_tolerance)) {
    shape_class = CORNER;
    squared_fit_quality = mse_corner;
    disoriented
      = (mse_corner > fsqr(_line_tolerance)
	 || big_npoints < _min_line_points
	 || (fabs(dir_line.dot(direction)) <= _cos_fit_big_difference
	     && fabs(dir_line.dot(norm_dir)) <= _cos_fit_big_difference
	     && mse_line/mse_corner < fsqr(_fit_much_better)));
    features[FIRST].init(pd_first_corner);
    features[CORNER_F].init(pd_corner);
    features[LAST].init(pd_last_corner);
    return true;
  } else {
    return false;
  }
}


// Classify according to shape.
// 
// Fit as line, and as corner (if applicable), then see which is better, and
// if either is good enough.  If the line fit is ridiculously good (1mm),
// don't even think about fitting a corner to avoid numeric problems.  Also,
// if only two points, don't try to corner fit.  Line fit error should be zero
// in this case, so this test is redundant, and included only so that it is
// clear that corner fit can assume >= 3 points.
//
// If corner fit fails, we can still fit as line if the fit is good enough.
//
// Finally, determine if the segment is compact according to size and point
// density.
//
void Segment::shape_classify () {
  if (points.size() < 2) {
    if (_use_bounding_box) {
      shape_class = COMPLEX;
      disoriented = true;
      compact = true;
      return;
    } else {
      ERROR("Segment with less than 2 points.");
    }
  }

  unsigned int eff_ignore =
    ((_ignore_end_points * 2 + 2) > points.size()
     ? 0
     : _ignore_end_points);

  find_bounds();

  Vec2d base, dir_line;
  double mse_line;
  {
    double sumsq;
    int npoints;
    fit_linear(eff_ignore, points.size() - eff_ignore, sumsq, npoints,
	       base, dir_line);
    mse_line = sumsq/npoints;
  }
  PointData pd_first = points[eff_ignore];
  pd_first.pos = closestPoint(pd_first.pos, base, dir_line);
  PointData pd_last = points[points.size() - 1 - eff_ignore];
  pd_last.pos = closestPoint(pd_last.pos, base, dir_line);

  if (mse_line > fsqr(0.001)
      && points.size() >= 3
      && try_fit_corner(eff_ignore, mse_line, dir_line)) {
    // do nothing; already done as side-effect of try_fit_corner.
  } else if (!_use_bounding_box || mse_line < fsqr(_disoriented_line_tolerance)) {
    // Otherwise if line fit is good enough, fit as line.
    shape_class = LINE;
    squared_fit_quality = mse_line;
    disoriented
      = (mse_line > fsqr(_line_tolerance) || points.size() < _min_line_points);
    direction = dir_line * -1;
    // norm_dir must point away from scanner.
    norm_dir = normal(direction);
    if (norm_dir.dot(scanner->location - pd_first.pos) > 0)
      norm_dir = norm_dir * -1;
    features[FIRST].init(pd_first);
    features[LAST].init(pd_last);
  } else {
    shape_class = COMPLEX;
    disoriented = true;
    direction = Vec2d(0, 0); // these were clobbered by fit_corner
    norm_dir = Vec2d(0, 0);
  }

  double diag = (features[MAX_BOUND].pos() - features[MIN_BOUND].pos()).length();
  if (diag > _compact_size) {
    compact = false;
  } else {
    disoriented = true;
    int min_npoints = (int)rint(diag * _compact_min_density);
    for (unsigned int i = 0; i < points.size() && min_npoints > 0; i++)
      if (!points[i].occluded) min_npoints--;
    compact = !min_npoints;
  }
}


// Find the position of the next point before (increment = -1) or after
// (increment = 1) this segment that has any range return.  If there is no
// such point, return the scanner location.
const Vec2d &Segment::adjacent_point (int increment) {
  int start_ix, end_ix;
  if (increment == 1) {
    start_ix = points.back().scan_pos + 1;
    end_ix = scanner->num_points;
  } else {
    start_ix = points.front().scan_pos - 1;
    end_ix = -1;
  }

  for (int ix = start_ix; ix != end_ix; ix += increment) 
    if (scanner->points[ix].range < _no_return_range)
      return scanner->points[ix].pos;

  return scanner->location;
}

// Segment::check_1_scan_angle
//
// Determine whether a line end is vague and find the position covariance
// derived from angular resolution limits.  dir points from other to end, and
// norm is a normal pointing away from the scanner.  Increment increments an
// index in the points in the direction of the end.  end_ix is the index in
// points of the point that establishes the end, and npoints is the number of
// points in the line or corner side
//
// Due to the scanner angular resolution limit, the feature position
// uncertainty is strongly asymmetrical.  We determine the point spacing at
// the end of the line by finding the max spacing along the line for the last
// several points, then use this to generate a longitudinal uncertainty.  This
// is combined with the lateral uncertainty, and then rotated into the world
// coordinate frame to create an appropriately correlated position covariance.
// Using the actual inter-point spacing works better than the theoretical
// spacing due to angular resolution because it penalizes ends where we are
// missing returns.
//
// An end is non-vague if there is good evidence that the line does not
// continue.  The line might continue beyond this segment without our
// otherwise realizing it if an object in front is occluding our view, or if
// due to scanner resolution limits or signal dropout (specularity, etc.), the
// next point is not close enough.  Our test has two parts:
//
// In the first test, the end-point in a scan is vague if the adjacent point
// does not lie far enough behind the line, where "far enough" is
// _max_segment_distance * _vague_range_margin.  This handles both the case
// where the next point is on the line but too far away, or where there is an
// occluding object.  The adjacent point is the closest neighboring point that
// has any range return.
//
// This is somewhat redundant with the front part of the "occluded" test done
// in Datmo::segment, but also gets a few more cases because the next point on
// the line will often be at a greater range than the apparent end point, so
// it could possibly be shadowed by an object that is at a greater range than
// the end.  In practice this is not entirely distinct from the case where the
// next point is on the line, as we have to allow some slop in the line
// matching.
//
// The second test is a side effect of the position covariance computation.
// If the point spacing is so large we infer a covariance larger than
// _vague_pos_noise, then the end is marked vague.  This handles the case
// where we have a bad line fit, and the line is nearly parallel to the scan
// ray.  In this case, being "behind the line" is not well defined.
//
// The max point spacing test is waived for compact segments.  It is too
// conservative for pedestrians because we see the legs seperately.  Compact
// segments have to meet a separate point density requirement.  If the ends
// are vague, then a track never gets created.
//
// If the segment is disoriented, then we bound how asymmetrical the
// distribution can be by increasing the lateral noise.  We don't want the
// line randomly flipping around to create the impression that we are
// correctly localizing in 2D when we combine multiple measurements.
//
void Segment::check_1_scan_angle
    (FeaturePoint &end, const FeaturePoint &other, int increment,
     const Vec2d &dir, const Vec2d &norm,
     unsigned int end_ix, unsigned int npoints,
     double &along_sigma, double &lateral_sigma)
{
  if (end.vague) {
    along_sigma = _max_segment_distance * _point_spacing_sigma_factor;
  } else {
    Vec2d E = end.pos();
    const Vec2d &adj_pos = adjacent_point(increment);

    // Not far enough behind the line?
    if ((adj_pos - E).dot(norm) < _max_segment_distance * _vague_range_margin)
      end.vague = true;

    // Find max inter-point spacing near the end of the line.
    Vec2d prev(points[end_ix].pos);
    double max_space = 0;
    unsigned int ix = end_ix - increment;
    for (int count = min(_spacing_npoints, (int)npoints) - 1;
	 count > 0;
	 count--) {
      Vec2d pos = points[ix].pos;
      double space = fabs((pos - prev).dot(dir));
      if (space > max_space) max_space = space;
      ix = ix - increment;
      prev = pos;
    }
      
    along_sigma = max_space * _point_spacing_sigma_factor;
    if (along_sigma > _vague_pos_noise && !compact)
      end.vague = true;
  }

  lateral_sigma
    = max((disoriented ? _disoriented_pos_noise : _pos_noise),
	  sqrt(squared_fit_quality));

  double elong
    = (disoriented ? _max_disoriented_pos_noise_elongation : 1e3);

  set_rotated_covariance(end.residue_covariance,
			 dir, norm, along_sigma, lateral_sigma,
			 elong);
}

// Check if scan angle on ends is too oblique, and mark as vague.
// Note that the end may already be vague due to occlusion.
void Segment::check_scan_angle () {
  unsigned int npoints = points.size();
  double ignore1, ignore2;
  if (shape_class == LINE) {
    check_1_scan_angle(features[FIRST], features[LAST], -1,
		       direction, norm_dir,
		       0, npoints, ignore1, ignore2);
    check_1_scan_angle(features[LAST], features[FIRST], 1,
		       direction*-1, norm_dir,
		       npoints-1, npoints, ignore1, ignore2);
  } else if (shape_class == CORNER) {
    features[CORNER_F].vague = false;
    double f_along, f_lateral, l_along, l_lateral;
    check_1_scan_angle(features[FIRST], features[CORNER_F], -1,
		       direction, norm_dir,
		       0, corner_ix, f_along, f_lateral);
    check_1_scan_angle(features[LAST], features[CORNER_F], 1,
		       norm_dir, direction,
		       npoints-1, npoints - corner_ix, l_along, l_lateral);

    double elong
      = (disoriented ? _max_disoriented_pos_noise_elongation : 1e3);

    set_rotated_covariance(features[CORNER_F].residue_covariance,
			   direction, norm_dir,
			   1/(1/f_along + 1/l_lateral),
			   1/(1/f_lateral + 1/l_along),
			   elong);
  }
}

// If end is vague, and line from other to end is shorter than _linear_size,
// then extend end to that limit.
void Segment::maybe_extend 
    (int feature_ix, Vec2d &end, const Vec2d &other) {
  if (features[feature_ix].vague) {
    Vec2d line_incr = end - other;
    double cur_len = line_incr.length();
    if (cur_len < _linear_size)
      end = line_incr * (_linear_size/cur_len) + other;
  }
}

// Position is the middle of the object, taken from the ends if linear or from
// the box otherwise.  When finding the center of large objects, we extend
// half-vague lines to be _linear_size long.  If line, offset position
// estimate normal to the line away from the scanner by 1/2 the default
// object, or 1/2 actual size, whichever is less.
//
void Segment::find_center () {
  Vec2d p1, p2;
  double offset = 0;
  if (shape_class == LINE || shape_class == CORNER) {
    p1 = features[FIRST].pos();
    p2 = features[LAST].pos();
    double size = (p1 - p2).length();
#if 0
    if ((size > _compact_size) && !vague_line()) {
      if (shape_class == LINE) {
	maybe_extend(FIRST, p1, p2);
	maybe_extend(LAST, p2, p1);
      } else {
	const Vec2d &corner = features[CORNER_F].pos();
	maybe_extend(FIRST, p1, corner);
	maybe_extend(LAST, p2, corner);
      }
    }
#endif
    if (shape_class == LINE)
      offset = min(_linear_size, size)*0.5;
  } else {
    p1 = features[MIN_BOUND].pos();
    p2 = features[MAX_BOUND].pos();
  }

  Vec2d position = (p1 + p2)*0.5;

  position = position + norm_dir * offset;

  PointData pd;
  pd.pos = position;
  pd.occluded = false;
  features[CENTER].init(pd);
}


// An object is considered significantly occluded if it is compact and
// any point is occluded, or if the first or last point is occluded.
// If disoriented and occluded, jack up the position noise.
void Segment::detect_occlusion () {
  if (compact) {
    for (unsigned int i = 0; i < points.size(); i++) {
      if (points[i].occluded) {
	occlusion = PARTIAL;
	return;
      }
    }
  } else {
    if (points[0].occluded || points.back().occluded)
      occlusion = PARTIAL;
  }

  if (occlusion != NONE && disoriented) {
    for (int f_ix = 0; f_ix < NUM_FEATURES; f_ix++) 
      if (valid_feature(f_ix))
	features[f_ix].residue_covariance *= _occluded_noise_factor;
  }
}

	
// Call this after all points are added to a newly created segment to its
// extract features.
void Segment::init () {
  check_scan_angle();
  find_center();
  if (_detect_occlusion)
    detect_occlusion();
}


void Segment::dump_state (ostream &out) {
  out <<endl;
  dump_time(out);
  out <<"fit_quality " <<sqrt(squared_fit_quality) <<endl;
}
