///////////////////////////////////////////////////////////////////////////////
//
//                               UnpackAction.h
//
// Implements an action which takes a flat buffer of data with no alignment 
// padding and builds it into the correct data structure
//
// Classes defined for export:
//     UnpackAction
//     
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_unpack_action_h
#define utils_unpack_action_h

#include <utils/Basic.h>
#include <string.h>

#include <utils/formatting/DataAction.h>
#include <utils/formatting/FormConfig.h>

__UTILS_BEGIN_NAMESPACE

class UnpackAction : public DataAction {
    UTILS_BASE_HEADER(UnpackAction);
  public:
    // Create action to unpack from a buffer.  If buf_size is 0, then all data
    // must be allocated.  If it is non-zero, then all data possible will be 
    // used "in place," i.e., pointers will reference the data in buffer 
    // directly, if possible. align and byte_order specify the alignment and 
    // byte order of the data in buffer.
    UnpackAction(unsigned char* buffer=0, int buf_size=0, int index=0,
                 void* data=NULL,int align=ALIGN, int byte_order=BYTE_ORDER);

    void setBuffer(unsigned char* buffer, int buf_size, int index=0,
                   int align=ALIGN, int byte_order=BYTE_ORDER);

    int getIndex() const { return _buf_index; }
    unsigned char* getBuffer() const { return _buffer; }
    int getBufSize() const { return _buf_size; }
    int getInt();
    int getAlignment() const { return _align; }
    int getByteOrder() const { return _byte_order; }
    void setIndex(int i) { _buf_index = i; }

    virtual bool actInt();
    virtual bool actShort();
    virtual bool actLong();
    virtual bool actFloat();
    virtual bool actDouble();
    virtual bool actChar();
    virtual bool actLength(int size);

    virtual bool actString();
    virtual bool actPtr(FormatSpec*);

    virtual bool actFixedArray(const Vector<int>& sizes, FormatSpec* fmt);
    virtual bool actVarArray(StructSpec*,
                               const Vector<int>&, FormatSpec*);

    static bool unpack(FormatSpec*, unsigned char*, int, void*,
                         int=0,int=ALIGN,int=BYTE_ORDER);
    static void* unpack(FormatSpec*, unsigned char*, int, 
                        int=ALIGN,int=BYTE_ORDER);
    static void initClass();

    // transfer from the buffer to the dest
    void fromBuffer(void* dest, int size) {
       if (dest != _buffer)
           memcpy((char*) dest, (char*) _buffer+_buf_index, size);
       _buf_index += size;
    }

    // advance the buffer index by length
    void advanceBuffer(int length) { _buf_index += length; }

    // copy size bytes from src into the current data 
    void toData(void* src, int size) { memcpy((char*) curData(), (char*) src, 
                                             size); }

  private:
    // return True if the current data is actually in the buffer
    bool dataInPlace() const {
        unsigned char* cur_data = curData();
        return (_buf_size &&
                cur_data >= _buffer && cur_data < _buffer+_buf_size);
    }

    // return TRUE if the packed buffer has the same byte order
    bool inNativeFormat() const { return _byte_order == BYTE_ORDER; }

    // return TRUE if the data is to be unpacked "inline," i.e. using the
    // the buffer data as much as possible
    bool isInline() const { return _buf_size != 0; }
    bool transferArrayData(FormatSpec*, int);

    // formatting routines for swapping between byte orders

#if ((BYTE_ORDER == LITTLE_ENDIAN) || (BYTE_ORDER == BIG_ENDIAN))

    /* The little endian is just reversed from the big endian, so we can
     * do the exchange when we do the copy and save calling hton*
     */

    void bytesToShort(unsigned char* dest) {
        if (_byte_order == BYTE_ORDER)
            fromBuffer(dest, sizeof(short));
        else {
            unsigned char* src = _buffer+_buf_index;
            if (src != dest) {
                dest[0] = src[1];
                dest[1] = src[0];
            } else {
                unsigned char s = src[0];
                dest[0] = src[1];
                dest[1] = s;
            }
            _buf_index += 2;
        }
    }

    void revIntBytes(unsigned char* dest) {
        unsigned char* src = _buffer+_buf_index;
        if (src == dest) {
            unsigned char s0 = src[0];
            unsigned char s1 = src[1];
            unsigned char s2 = src[2];
            dest[0] = src[3];
            dest[1] = s2;
            dest[2] = s1;
            dest[3] = s0;
        } else {
            dest[0] = src[3];
            dest[1] = src[2];
            dest[2] = src[1];
            dest[3] = src[0];
        }
    }

    void bytesToInt(unsigned char* dest) {
        if (_byte_order == BYTE_ORDER) 
            fromBuffer(dest, sizeof(int));
        else {
            revIntBytes(dest);
            _buf_index += 4;
        }
    }

#else /* not little endian or big endian, must be pdp endian.*/

    void bytesToShort(unsigned char* dest) {
        fromBuffer(dest, sizeof(short));
        if (_byte_order != BYTE_ORDER) 
            *((short*) dest) = htons(*((short*) dest));
        _buf_index += sizeof(short);
    }

    void revIntBytes(unsigned char* dest) {
        *((int*) dest) = htonl(*(int*) (_buffer+_buf_index));
    }

    void bytesToInt(unsigned char* dest) {
        fromBuffer(dest, sizeof(int));
        if (_byte_order != BYTE_ORDER)
            *((int*) dest) = htonl(*(int*) dest);
    }

#endif /* LITTLE_ENDIAN || BIG_ENDIAN */

    void bytesToLong(unsigned char* dest) { bytesToInt(dest); }
    void bytesToFloat(unsigned char* dest) { bytesToInt(dest); }
    
    void bytesToDouble(unsigned char* dest) {
        if (_byte_order == BYTE_ORDER) {
            fromBuffer(dest, sizeof(double));
        } else {
            revIntBytes(dest);
            _buf_index += 4;
            revIntBytes(dest+4);
            _buf_index += 4;
        }
    }

  private:
    int _buf_size;             // size of the buffer
    int _buf_index;            // current index into the buffer
    unsigned char* _buffer;    // the buffer
    int _align, _byte_order;   // the alignment and byte order of the data
};

__UTILS_END_NAMESPACE

#endif
