///////////////////////////////////////////////////////////////////////////////
//
//                                 Token.h
//
// This header file defines classes that convert format string tokens into 
// tokens that can be manipulated by FormatParser
//
// Classes defined for export:
//    Token - a symbolic token representing a component of a format string
//    TokenReader - Reads tokens from text
//
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_formatting_token_h
#define utils_formatting_token_h

#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE

/* This structure is what token string specifiers get converted into for easier
   analysis */
struct Token {
    /* types of tokens */
    enum {
        LBRACE, RBRACE, COMMA, LT, GT,
        STAR, BANG, COLON, LBRACK, RBRACK, LPAREN, RPAREN,
        INT,
        STR,
        EOSTRING
    };
    
    Token(int t, int l) { type = t; loc = l; }
    ~Token();
    void print();

    int loc;          // index into string where token is found
    int type;         // type of token
    union {           // value of the token
        char *str;    // either a string if is a STR type token
        int num;      // or a number if it is a INT type token
    } value;
};

/* Class which gets initialized with a string by the Init member function, and
   which then nextToken is called on repeatedly to process the tokens as they
   are gotten from the string.  There is a one token undo buffer which can
   be invoked using ungetToken */
class TokenReader {
  public:
    TokenReader();
    ~TokenReader();

    void init(const char*);
    Token* nextToken();
    void ungetToken(Token*);

  private:
    Token* readString(const char* s);
    Token* readInt(const char* s);

  private:
    Token* _unget_token;       // the one token undo buffer
    int _current_location;       // index into the parse string
    const char* _current_string; // the parse string
};

__UTILS_END_NAMESPACE

#endif
