///////////////////////////////////////////////////////////////////////////////
//
//                               Vector.h
//
// Declares a variable sized vector list template
//
// Templates defined for export:
//     Vector<T>
//    
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_VECTOR_H
#define UTILS_VECTOR_H

#include <utils/Basic.h>

#include <string.h>
#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE
    
// the variable array template class
template <class T> class Vector {
  public:
    // create a variable size array containing elements of type T with a 
    // maximum number of elements max_elems
    Vector(int max_elems = 1) {
        _max_elems = max_elems;
        _num_elems = 0;
        if (_max_elems <= 0) {
            _max_elems = 0;
            _data = (T*) NULL;
        } else {
            _data = (T*) new char[max_elems*sizeof(T)];
        }
    }

    // destroy the variable size array
    virtual ~Vector() {
        if (_data)
            delete [] (char*) _data;
    }

    int numElems() const { return _num_elems; }  // get number of elements
    T* getData() const { return _data; } // get the element data
    void clear() { _num_elems = 0; }   // clear out the elements
    bool empty() const { return _num_elems == 0; }

    // set the maximum number of elements of the variable sized array, 
    // expanding the array if necessary
    void setMaxNum(int max_elems) {
        if (max_elems <= _max_elems)
            return;
        T* new_data = (T*) new char[max_elems*sizeof(T)];
        if (_data) {
            memcpy((char*) new_data, (char*) _data, _num_elems*sizeof(T));
            delete [] (char*) _data;
        }
        _data = new_data;
        _max_elems = max_elems;
    }

    int getMaxNum() const { return _max_elems; }

    void setNumElems(int num_elems) {
        setMaxNum(num_elems);
        _num_elems = num_elems;
    }

    // remove the element at the index, shift the remainder of the array down
    // to compensate.  Returns false if that index does not exist in the array
    bool remove(int index) {
        if (index < 0 || index >= _num_elems)
            return false;

        _num_elems--;
        if (_num_elems-index > 0) 
            memmove(_data+index, _data+index+1, (_num_elems-index)*sizeof(T));

        return true;
    }

    // copy a list
    Vector& operator =(const Vector& pl) { copy(pl); return *this; }

    // pop an element from the beginning of the list, removing it from the list
    bool pop(T& data) {
        if (_num_elems <= 0)
            return false;
        
        data = _data[0];
        remove(0);
        return true;
    }

    // get a reference to an element in the list, truncated by the list range
    T& operator [](int i) {
        if (i < 0)
            i = 0;
        if (i >= _num_elems)
            i = _num_elems-1;
        return _data[i];
    }

    T& last() { return _data[_num_elems-1]; }
    const T& last() const { return _data[_num_elems-1]; }

    // get a constant reference to an element, to satisfy some picky C++ code
    const T& operator[](int i) const { return (*((Vector*) this))[i]; }
    const T& getValue(int i) const { return (*((Vector*) this))[i]; }
    
    // set the i'th element to val.  If the variable array does not have i 
    // elements, expand it.
    bool setValue(int i, const T& val) {
        if (i<0)
            return false;
        if (i >= _num_elems) {
            setMaxNum(i+1);
            _num_elems = i+1;
        }
        _data[i] = val;

        return false;
    }

    // append data to the list, increasing the maximum size if necessary
    void append(const T& data) {
        if (_num_elems >= _max_elems)
            setMaxNum(2*_num_elems);
        _data[_num_elems] = data;
        _num_elems++;
    }
    // Create a new element at the end of the list and return a reference
    T& append() {
        if (_num_elems >= _max_elems)
            setMaxNum(2*_num_elems);
        _num_elems++;
        return _data[_num_elems-1];
    }

    // prepend data to the list
    void prepend(const T& data) {
        shift(0);
        _data[0] = data;
    }

    // insert data at an index into the list, creating the elements if 
    // necessary
    void insert(const T& data, int index) {
        if (index < 0)
            return;
        if (index == _num_elems) {
            append(data);
            return;
        }
        if (index > _num_elems) {
            setMaxNum(index+1);
            _num_elems = index+1;
        } else
            shift(index);
        _data[index] = data;
    }

    // set res to be the reverse of the list
    void reverse(Vector& res) {
        res.setMaxNum(numElems());
        for (int i=numElems()-1;i>=0;i--) 
            res.append(_data[i]);
    }

    // find element data in the list, returning the index if found, -1 if not
    int find(const T & data) const {
        for (int i=0;i<numElems();i++) 
            if (_data[i] == data)
                return i;
        return -1;
    }
    
    // zero backing store
    void zero() {
	    if (_data) {
		    memset(_data, 0, _max_elems * sizeof(T));
	    }
    }


  private:
    // shift all elements past and including index by one to make room for 
    // something new
    void shift(int index) {
        setMaxNum(_num_elems+1);
        if (_num_elems) {
            memmove(_data+index+1, _data+index, (_num_elems-index)*sizeof(T));
        }
        _num_elems++;
    }
    // copy the elements to another list
    void copy(const Vector& other) {
        int num_elems = other.numElems();
        setMaxNum(num_elems);
        _num_elems = num_elems;
        memcpy((char*)getData(), (char*)other.getData(), _num_elems*sizeof(T));
    }

  protected:
    int _num_elems;   // number of elements in the list
    T* _data;         // the array of elements in the list
    int _max_elems;   // the maximum number of elements that the array can
                      // hold without resizing
};

__UTILS_END_NAMESPACE

#endif
