#ifndef VEC3D_H
#define VEC3D_H
#include <math.h>
#include <utils/Basic.h>

#include <iostream>

__UTILS_BEGIN_NAMESPACE

class InvalidIndexException {
  //an invalid index was passed to the Vec3d
};

class Vec3d {
 public:
  union {
    double v[3]; 
    struct { 
      double x;
      double y;
      double z;
    };
    struct {
      double roll;  // around x axis
      double pitch; // around y axis
      double yaw;     // around z axis
    };
  };
  int dim(void) {return 3;};
  Vec3d(double p0,double p1, double p2=0);
  Vec3d(int i) {v[0]=0;v[1]=0;v[2]=0;};
  Vec3d(void) {v[0]=0;v[1]=0;v[2]=0;};
  Vec3d(double* array) {v[0]=array[0];v[1]=array[1];v[2]=array[2];}
  double length(void);
  double len(void) {return length();}
  Vec3d normalize(void);
  Vec3d cross(const Vec3d& other) const;
  double* getValue() { return &v[0]; }
  const double* getValue() const { return &v[0]; }
  void getValue(double* output) const;
  void negate() { v[0]=-v[0]; v[1]=-v[1]; v[2]=-v[2]; }
  void setValue(const double* array) {
    v[0]=array[0];v[1]=array[1];v[2]=array[2];
  }
  
  friend std::ostream &operator<<(std::ostream &str, Vec3d const &v);  
  friend std::istream &operator>>(std::istream &stream, Vec3d &in);

  Vec3d & operator +=(const Vec3d& rhs);
  Vec3d & operator -=(const Vec3d& rhs);

  Vec3d & operator/=(const double &r) {
    v[0]/=r; v[1]/=r; v[2]/=r;
    return *this;
  };

  Vec3d & operator*=(const double &r) {
    v[0]*=r; v[1]*=r; v[2]*=r;
    return *this;
  };

  double& operator[](int i);
  const double& operator[](int i) const;
  double  dot(const Vec3d& on) const;

  double & operator()(int ix) {return v[ix];};
  double operator()(int ix) const {return v[ix];};

  // rotates the vector about the Z axis by theta radians
  void rotateZ(double theta);

  /** this clips the vector to the value passed in clipper,
      i.e. any axis of the vector that is greater than clipper 
      is limited to clipper*/
  void clip(Vec3d& clipper);
  /** this clips the vector to the absolute value of clipper*/
  void clipabs(Vec3d &clipper);
  //Angle between vectors RAD

  double Vec3d::angle(const Vec3d &v1) const;
  Vec3d& operator=(const Vec3d rhs);
  Vec3d& operator=(const double rhs);

 private:
  static InvalidIndexException iie;
};


Vec3d operator +(const Vec3d& rhs);
Vec3d operator -(const Vec3d& rhs);

inline Vec3d operator+(const Vec3d &a, const Vec3d &b) {
  return Vec3d(a(0)+b(0), a(1)+b(1), a(2)+b(2));
};

inline Vec3d operator-(const Vec3d &a, const Vec3d &b) {
  return Vec3d(a(0)-b(0), a(1)-b(1), a(2)-b(2));
};

inline Vec3d operator/(const Vec3d &a, double b) {
  return Vec3d(a(0)/b, a(1)/b, a(2)/b);
};

inline Vec3d operator*(const Vec3d &a, double b) {
  return Vec3d(a(0)*b, a(1)*b, a(2)*b);
};

inline Vec3d operator*(double b, const Vec3d &a) {
  return Vec3d(a(0)*b, a(1)*b, a(2)*b);
};

inline Vec3d operator*(const Vec3d &a, const Vec3d &b) {
  return a.cross(b);
};

inline Vec3d operator-(const Vec3d &a) {
  return Vec3d(-a(0),-a(1), -a(2));
}


inline bool operator!=(const Vec3d &a, const Vec3d &b) {
  if (a.x!=b.x || a.y!=b.y || a.z!=b.z)
    return true;
  else
    return false;
}

inline bool operator==(const Vec3d &a, const Vec3d &b) {
  if (a.x!=b.x || a.y!=b.y || a.z!=b.z)
    return false;
  return true;
}

inline double& Vec3d::operator[](int i) {
  if (i>-1 && i<3)
    return v[i];
  else {
    std::cerr << "\nERROR: Vec3d index out of range: index = " <<  i
              << std::endl;;
    throw iie;
  }
}
inline const double& Vec3d::operator[](int i) const {
  if (i>-1 && i<3)
    return v[i];
  else {
    std::cerr << "\nERROR: Vec3d index out of range: index = " <<  i
              << std::endl;;
    throw iie;
  }
}

inline Vec3d Vec3d::cross(const Vec3d& other) const {
  Vec3d retval;
  retval.v[0] = (v[1]*other.v[2]) - (v[2]*other.v[1]);
  retval.v[1] = (v[2]*other.v[0]) - (v[0]*other.v[2]);
  retval.v[2] = (v[0]*other.v[1]) - (v[1]*other.v[0]);
  return retval;
}

inline void Vec3d::getValue(double* output) const {
  output[0] = v[0]; output[1] = v[1]; output[2] = v[2];
}

__UTILS_END_NAMESPACE

#endif //__R3VECTOR





