///////////////////////////////////////////////////////////////////////////////
//
//                               Type.h
//
// Defines the class for defining a type.  This class lets types be checked
// at run-time, not just at compile time.  Type names are stored in a static
// dictionary so that types can be looked up by name.
//
// Classes defined for export:
//   Type - the hierarchical type class
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_TYPE_H
#define UTILS_TYPE_H

#include <utils/Basic.h>
#include <utils/String.h>

__UTILS_BEGIN_NAMESPACE

struct TypeData;
template <class T> class StringDict;

//
// Type has no virtual functions to keep it small...
//
class Type {
#ifdef __DELTA
#  pragma __nondynamic_class
#endif
  public:
    Type(short key=0) { index = key; }
    // Get a type from a name
    static Type	fromName(Name name);

    // Get name
    Name		getName() const;

    // Get parent
    Type		getParent() const;

    // returns the "bad type" - used for type errors
    static Type	badType();

    // returns TRUE if the type is bad
    bool		isBad() const
	{ return (index == 0); }

    // returns TRUE if this type is derived from the argument type
    bool		isDerivedFrom(Type t) const;

    // returns TRUE if this type knows how to create an instance
    bool		canCreateInstance() const;

    // Get an instance of the given type, or NULL if such an instance
    // is unavailable.
    void * 		createInstance() const;

    // test equality / inequality
    int			operator ==(const Type t) const
	{ return (index == t.index);}
    int			operator !=(const Type t) const
	{ return (index != t.index);}

    // Less-than operator that can be used to sort types. Pretty
    // useless otherwise.
    int			operator <(const Type t) const
	{ return (index < t.index); }

  public:

    // Create a new type
    static Type	createType(Type parent, Name name,
				   void * (*createMethod)(Name, void*)=0L,
                                   void* clientData = NULL);

  public:
    // Initialize the type system
    static void		init();

    // Returns the type key as a short
    short		getKey() const		{ return index; }

  private:
    // Types are passed around on the stack a lot, and are cast to
    // void *'s; they MUST fit into a single word.
    unsigned int index;

    static StringDict<void*> *nameDict;   // the type name dictionary
    static int		nextIndex;       // the next available type slot
    static int		arraySize;       // the maximum number of types
    static TypeData	*typeData;   // array of all types

    static void		expandTypeData();  // expand typeData
};

__UTILS_END_NAMESPACE

#endif
    


