///////////////////////////////////////////////////////////////////////////////
//
//                               Time.h
//
// Declares a class for managing time.  An Time state instance holds
// a time, and gives methods for extracting it in a variety of forms.  Static
// methods are available to find the current time, and to control how that
// current time proceeds, either at a constant scaling of real time or 
// by discrete jumps.
//
// Classes declared for export:
//     Time
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_TIME_H
#define UTILS_TIME_H

#include <utils/Basic.h>
#include <math.h>

__UTILS_BEGIN_NAMESPACE

struct TimeState;

// the time class.  
class Time  {
  public:
    // Creators
    Time();
    Time( double sec );
    Time( long sec, long usec );

    static void init();   // initialize the Time static variables

    void setToTimeOfDay();  // set instance to be time of day
    // get time of day, given the time mode, which is either a scale factor
    // of real time or jumping by discrete steps whenever stepTime is called
    static Time getTimeOfDay();   

    // get time of day, ignoring time modes etc.
    static Time getRealTimeOfDay();

    // the result of getTimeOfDay will go at a constant factor of real time
    static void useContinuousTime();
    // set that factor
    static void setTimeConstant(float tc);

    // the result of getTimeOfDay will only change when stepTime is called
    static void useDiscreteTime();
    // amount time will step by
    static void setTimeStep(float time_step);
    // step time
    static void stepTime();

    // information about the time mode
    static int getTimeMode() { return _time_mode; }
    static float getTimeConstant() { return _time_constant; }
    static float getTimeStep() { return _time_step; }

    // allows the time state to be pushed and popped
    static void pushState();
    static void popState();

    static Time	zero();
    static Time	maxt();
    void			setValue( double sec );
    void			setValue( long sec, long usec );
    double			getValue() const;
    void			getValue( long &sec, long &usec ) const;
    void			getValue( int &sec, int &usec ) const;
    void	getValue( unsigned int &sec, unsigned int &usec ) const;
    int		operator ==( const Time &t ) const;
    int		operator !=( const Time &t ) const;
    int  operator <(const Time& t) const;
    int  operator >(const Time& t) const;
    int  operator <=(const Time& t) const;
    int  operator >=(const Time& t) const;

    bool isZero() const { return _sec == 0 && _usec == 0; }

    // Addition
    friend Time		operator +(const Time &t0, const Time &t1);

    // Subtraction
    friend Time		operator -(const Time &t0, const Time &t1);

    // Destructive addition
    const Time &	operator +=(const Time &tm)
	{ return (*this = *this + tm); }

    // Destructive subtraction
    const Time &	operator -=(const Time &tm)
	{ return (*this = *this - tm); }

    // Absolute difference
    Time        absDiff(const Time &tm) const;
    
    static void sleep(double);

    // Read only references to state
    const long int &sec;
    const long int &usec;

    // Conversion to/assignment from double
    operator const double () const 
        { return (double)_sec + ((double)_usec)/1000000; }
    const Time &operator =(double rhs) 
        {  _sec = (long)rhs;  _usec = (long)rint((rhs - _sec) * 1000000); return *this;}
    const Time &operator =(const Time &rhs) 
        {  _sec = rhs._sec; _usec = rhs._usec; return *this;}

  protected:
    int _sec;
    int _usec;

  private:
    static Time* _time_base;     // the real time of a mode change
    static Time* _time_start;    // the simulated time of a mode change
    static float _time_constant;  // continuous time flow factor
    static float _time_step;      // discrete time flow step size
    static int _time_mode;        // 0 - continuous, 1 - discrete
    static TimeState* _state_stack;  // stack used by push/pop operations
};

__UTILS_END_NAMESPACE

#endif

