/*****************************************************************************

                               StringDict.h

 Declares a string dictionary template, i.e., it associates string's 
 with values of any type T.  

 Classes and Templates implemented for export:
     BasicStringDict - a dictionary of strings and void*
     StringDict<T> - a template front end for BasicStringDict which
                       takes care of the necessary casting
    
*****************************************************************************/

#ifndef UTILS_STRING_DICT_H
#define UTILS_STRING_DICT_H

#include <utils/Basic.h>

#ifdef __cplusplus

__UTILS_BEGIN_NAMESPACE

struct StringDictEntry;

class BasicStringDict  {
  public:
    // create a dictionary with entries buckets
    BasicStringDict(int entries = 251);
    virtual ~BasicStringDict();

    // Calls given routine (passing value) for each entry in dictionary.
    //  The order of entries is not guaranteed to mean anything.
    void applyToAll(void (*rtn)(const char* key, void *value) );
    
    // Like above, but with a data entry for callback usage
    void applyToAll(void (*rtn)(const char* key, void *value, void *data), 
                    void *data );
    
    // Removes all entries from dictionary.
    void	clear();

    // Add an entry to the dictionary
    bool enter(const char* key, void *value);

    // Find an entry in the dictionary
    bool find(const char* key, void*& value) const;

    // Removes the entry with the given key. Returns FALSE if no such entry.
    bool remove(const char* key);

  private:
    StringDictEntry* findEntry(unsigned long key, const char* str) const;

  private:
    int _table_size;                // Number of entries in the table
    StringDictEntry** _buckets;   // Hash table

    friend class StringDictIteratorBase;
};

template <class T>
class StringDict : public BasicStringDict {
  public:
    // create a dictionary with entries buckets
    StringDict(int entries = 251) : BasicStringDict(entries) {}

    // Calls given routine (passing value) for each entry in dictionary.
    //  The order of entries is not guaranteed to mean anything.
    void	applyToAll(void (*rtn)(const char* key, T value) ) {
        BasicStringDict::applyToAll((void (*)(const char*, void*)) rtn);
    }
    
    // Like above, but with a data entry for callback usage
    void	applyToAll(void (*rtn)(const char* key, T value, void *data), 
                       void *data ) {
        BasicStringDict::
            applyToAll((void (*)(const char*, void*, void*)) rtn, data);
    }
    
    // Add an entry to the dictionary
    bool enter(const char* key, T value) {
        return BasicStringDict::enter(key, (void*) value);
    }

    // Find an entry in the dictionary
    bool find(const char* key, T& value) const {
        void* v = (void*) value;
        bool res = BasicStringDict::find(key, v);
        value = (T) v;
        return res;
    }
};

// StringDictIteratorBase - iterator for a linked list of void*
class StringDictIteratorBase {
 public:
  StringDictIteratorBase(BasicStringDict& dict);

  void* first();
  void* next();
  const char* key();

 private:
  BasicStringDict& _dict;   // the dict we are iterating over
  StringDictEntry* _next;   // the current element
  int _index;
};

// StringDictIterator<T> - iterator of a dict of T
template <class T> class StringDictIterator : public StringDictIteratorBase {
 public:
  StringDictIterator(StringDict<T>& dict) : StringDictIteratorBase(dict) {}

  T first() { return (T) StringDictIteratorBase::first(); }
  T next() { return (T) StringDictIteratorBase::next(); }
};

template <class T> class ManagedStringDict : public StringDict<T*> {
 public:
  ~ManagedStringDict() {
    clear();
  }

  void clear() {
    StringDictIterator<T*> iter(*this);
    for (T* cur=iter.first(); cur; cur=iter.next())
      delete cur;
    StringDict<T*>::clear();
  }
};

__UTILS_END_NAMESPACE

typedef __UTILS::BasicStringDict *UTL_StringDict;

extern "C" {

#else

typedef void* UTL_StringDict;

#endif

typedef void (*SDICT_CallbackFunc)(const char*, void*, void*);

extern UTL_StringDict SDICT_new(int buckets);
extern void SDICT_delete(UTL_StringDict);
extern void SDICT_applyToAll(UTL_StringDict sd, SDICT_CallbackFunc, void*);
extern void SDICT_clear(UTL_StringDict sd);
extern bool SDICT_enter(UTL_StringDict sd, const char* key, void* value);
extern bool SDICT_find(UTL_StringDict sd, const char* key, void** value);
extern bool SDICT_remove(UTL_StringDict sd, const char* key);

#ifdef __cplusplus
}
#endif


#endif
