///////////////////////////////////////////////////////////////////////////////
//
//                               String.h
//
// Declares classes for creating and using smart strings and unique strings
//
// Classes declared for export:
//     String - a smart string class
//     Name - a unique string class
//     NameEntry - representation of a unique string
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_STRING_H
#define UTILS_STRING_H

#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE

class String
{
  public:
    // Constructor/destructor
    String();
    String( const char *str );
    String( const char *str, int start, int end );
    String( const String &str );
    String( int digitString );
    ~String();

  public:
    // Operations
    unsigned long hash();
    int			getLength() const;
    void		makeEmpty( bool freeOld = true );
    const char*	getString() const;
    String	getSubString( int startChar, int endChar = -1 ) const;
    void		deleteSubString( int startChar, int endChar = -1 );

    String&	operator =( const char *str );
    String&	operator =( const String &str );

    String&	operator +=( const char *str );
    String&	operator +=( const String &str );

    int			operator !() const;

    friend int	operator ==( const char *s, const String &str );
    friend int	operator ==( const String &str, const char *s );
    friend int	operator ==( const String &str1, const String &str2 );

    friend int	operator !=( const char *s, const String &str );
    friend int	operator !=( const String &str, const char *s );
    friend int	operator !=( const String &str1, const String &str2 );

    static unsigned long	hash( const char *s );
    static char* copy(const char*);

private:
    void		expand( int size );

private:
    // Variables
    char		*_string;
    int			_storage_size;
    static const char	_null_string[];
};


//////////////////////////////////////////////
// class Name

struct NameChunk;

class NameEntry
{
    // Operations
public:
    bool isEmpty() const;
    bool isEqual( const char *s ) const;
    const char*					getString() const;

    // Static operations
    static const NameEntry*	insert( const char *s );

    // Constructor/initialization
private:
    NameEntry( const char *s, unsigned long h, NameEntry *n );
    static void initClass();
    static void _cleanClass();

    // Static pointers
private:
    // name_table holds all of the name entries, organized by hash numbers.
    static int _name_table_size;
    static NameEntry **_name_table;

	// chunk is a linked list of chunks of memory for allocating strings
    static struct NameChunk	*_chunk;

    // Variables
    const char *_string;
    unsigned long _hash_value;
    NameEntry *_next;

    friend class Name;
};


class Name
{
    // Constructor/destructor
public:
    Name();
    Name( const char *s );
    Name( const String &s );
    Name( const Name &n );
    static void		initClass();
    static void		_cleanClass();
    ~Name();

    // Operations
public:
    const char*		getString() const;
    int				getLength() const;

    static bool	isIdentStartChar( char c );
    static bool	isIdentChar( char c );

    int				operator !() const;

    friend int		operator ==( const char *s, const Name &n );
    friend int 		operator ==( const Name &n1, const Name &n2 );
    friend int		operator ==( const Name &n, const char *s );
    friend int		operator !=( const char *s, const Name &n );
    friend int 		operator !=( const Name &n1, const Name &n2 );
    friend int		operator !=( const Name &n, const char *s );

    static bool	isNodeNameStartChar( char c );
    static bool	isNodeNameChar( char c );

    // Variables
private:
    const NameEntry	*_entry;
    static const char _bad_characters[];
    static const NameEntry* _empty_string; // default empty string
};

__UTILS_END_NAMESPACE

#endif
