#ifndef SQUAREMATRIX_H

#define SQUAREMATRIX_H

#include <utils/Linear.h>

__UTILS_BEGIN_NAMESPACE

// This square matrix class was developed for use in Kalman filters, though
// could presumably be used more generally.  Since the matrices may be
// somewhat large, operations are generally in-place destructive, with
// *this being clobbered with the result.  *this may be identical to
// any of the arguments, which are passed as const references to avoid
// unnecessary copying.
//
// NOTE: the inverse operation doesn't work on general matrices, only
// symmetric, positive definite ones.

template <class T, int size>
class SquareMatrix {
 public:

  // Construct matrix filled with zeros.
  SquareMatrix () {
    memset(&_data, 0, sizeof(_data));
  }

  // Copy constructor
  SquareMatrix<T, size> (const SquareMatrix &mat) {
    memmove(&_data, &mat._data, sizeof(_data));
  }

  // Construct from T*
  SquareMatrix<T, size> (const T *data) {
    memmove(&_data, data, sizeof(_data));
  }

  // Assignment operator.
  const SquareMatrix<T, size> &operator =(const SquareMatrix<T, size> &mat) {
    memmove(&_data, &mat._data, sizeof(_data));
    return *this;
  }

  // Zero contents.
  void clear () {
    bzero(&_data, sizeof(_data));
  }

  // Element-wise access.
  T &at (int i, int j) {
    assert(i >= 0 && i < size && j >= 0 && j < size);
    return _data[i][j];
  }

  const T &at (int i, int j) const {
    assert(i >= 0 && i < size && j >= 0 && j < size);
    return _data[i][j];
  }
    
  // Get pointer to data.
  const T* getValue () const { return (const T *)&_data; };
  T* getValue () { return (T *)&_data; };

  // Copy data to array.
  void getValue (T res[size][size]) const {
    memmove(res, _data, sizeof(_data));
  }

  // Print our contents to out.
  friend ostream& operator<< (ostream& out, const SquareMatrix<T, size> &m) {
    for (int i=0;i<size;i++) {
      for (int j=0;j<size;j ++)
	out <<m.at(i, j) <<" ";
      out <<endl;
    }
    return out;
  }

  // Multiply *this by a scalar.
  void operator *=( const T &d ) {
    for (int i=0;i<size;i++) 
      for (int j=0;j<size;j++) 
	_data[i][j] *= d;
  }

  // Set *this to the transpose of mat.
  void transpose (const SquareMatrix<T, size> &mat) {
    int i,j;
    for (i=0;i<size;i++) 
      for (j=i;j<size;j++) {
	T tmp = mat._data[i][j];
	_data[i][j] = mat._data[j][i];
	_data[j][i] = tmp;
      }
  }

  // Set *this to the product a*b.
  void mult (const SquareMatrix<T, size> &a, const SquareMatrix<T, size> &b) {
    int i,j,k;
    T sum;
    T res[size][size];
  
    for (i=0;i<size;i++) 
      for (j=0;j<size;j++) {
	sum = 0;
	for (k=0;k<size;k++)
	  sum += a._data[i][k]*b._data[k][j];
	res[i][j] = sum;
      }

    memmove(&_data, &res, sizeof(_data));
  }

  // Set *this to the outer product of V1 and V2 (a.k.a V1 * V2')
  void outer_product (const LinearVector<T, size> &V1,
		     const LinearVector<T, size> &V2) {
    const T *d1 = V1.getValue();
    const T *d2 = V2.getValue();

    for (int i=0;i<size;i++) 
      for (int j=0;j<size;j++) 
	_data[i][j] = d1[i] * d2[j];
  }

  // Set *this to the sum a+b.
  void add (const SquareMatrix<T, size> &a, const SquareMatrix<T, size> &b) {
    int i,j;
    for (i=0;i<size;i++) 
      for (j=0;j<size;j++) 
	_data[i][j] = a._data[i][j]+b._data[i][j];
  }

  // Set *this to the difference a-b.
  void sub (const SquareMatrix<T, size> &a, const SquareMatrix<T, size> &b) {
    int i,j;
    for (i=0;i<size;i++) 
      for (j=0;j<size;j++) 
	_data[i][j] = a._data[i][j]-b._data[i][j];
  }

  // Set *this to the Cholesky decomposition of a.  This is an upper
  // triangular matrix U such that U' * U = a, where U' is the
  // transpose of U.  This can only be determined for symmetric
  // positive definite matrices.  If we fail, return false.
  bool choldc (const SquareMatrix<T, size> a) {
    int i,j,k;
    T sum;
    clear();
    
    for (i=0;i<size;i++) {
      for (j=i;j<size;j++) {
	for (sum=a._data[i][j], k=i-1;k>=0;k--) 
	  sum -= _data[i][k]*_data[j][k];
	if (i == j) {
	  if (sum <= 0.0)
	    return false;
	  _data[i][i]=sqrt(sum);
	} else _data[j][i]=sum/_data[i][i];
      }
    }
    return true;
  }

  // Set *this to the inverse of the Symmetric Positive Definite matrix
  // a.  If the matrix is not S-P-D, then return false, and leave *this
  // unmodified.
  bool spd_inverse (const SquareMatrix<T, size> &a) {
    SquareMatrix<T, size> l, li, lit;
    if (!l.choldc(a)) {
      return false;
    }

    for (int i=0;i<size;i++) {
      li._data[i][i] = 1.0/l._data[i][i];
      for (int j = i+1; j<size; j++) {
	T sum=0.0;
	for (int k=i;k<j;k++) sum -= l._data[j][k]*li._data[k][i];
	li._data[j][i] = sum/l._data[j][j];
      }
    }
    
    lit.transpose(li);
    mult(lit, li);
    
    return true;
  }

  // Matrix * vector multiply, returning vector.
  LinearVector<T, size> operator *(const LinearVector<T, size> &vec) const {
    LinearVector<T, size> res;
    T *data = res.getValue();
    const T *vdata = vec.getValue();
    for (int i=0;i<size;i++) {
      data[i] = 0;
      for (int j=0;j<size;j++) 
	data[i] += _data[i][j]*vdata[j];
    }
    return res;
  }

private:

  // The data.
  T _data[size][size];
};

__UTILS_END_NAMESPACE

#endif
