// This is a simple 1D median filter that is reasonably efficient for
// low order filters.  The complexity for filtering each datum is O(n)
// in the filter order.  It would be possible to do worse than this by
// e.g. doing an O(n^2) sort of the window on each iteration.
//
// There are well known and reasonably obvious algorithms for 1D
// median with O(log(n)) complexity, e.g. using a tree so that each
// new element doesn't need to be explicitly compared to all other
// elements in the window.  However, I didn't feel like implementing
// anything that hairy.  Probably for small windows they wouldn't be
// faster either.
//
// This is an algorithm that I made up, though it's probably been done
// before.  The idea is for each element of the window, to keep track
// of the number of greater elements.  Then we can find the element
// with order/2 greater elements in our single linear pass.
//
// The num_greater count is maintained by comparing each element added
// and each element removed to all other elements.  On adding, if the
// new element is greater, increment the count.  On removing, decrement.
//
// How does this work with duplicate elements?  In that case, there
// may not be any element with order/2 greater elements.  It's a bit subtle.
// Suppose that we detected when we went to add a duplicate element,
// and instead added that element + epsilon so that it was larger than
// the older duplicate element.  Now there aren't any duplicate
// elements, so there will always be an element with order/2 greater.
//
// Instead of adding epsilon, what I actually do is make the add and
// remove tests asymmetric.  On add we use, <, on remove <=.  This
// implicitly implements a policy that when two elements are equal,
// the older element is always considered lesser.
//
// To simplify the implementation, I don't attempt to find the median
// of varying numbers of elements as the filter fills up.  Instead, on
// the first call to filter, I fill the filter with that value.  So
// the first value passed in will be returned as the result on the
// first order/2 + 1 calls.
// 
// Author: Rob MacLachlan
// Date: May 2004
// 
// This code has been placed in the public domain.  There is no warranty
// expressed or implied.

#ifndef MEDIANFILTER_H

#define MEDIANFILTER_H

template <class T, int order>
class MedianFilter {
 public:
  MedianFilter () :
    start(-1)
    {};

  // Data elements in the window.  This is a circular buffer, with the
  // oldest element at start and the newest at (start - 1) mod order.
  T data[order];

  // Number of elements in the window with greater value for each data
  // element.
  int num_greater[order];

  // The first element in the window
  int start;

  // Add element to filter and return the new value.  If filter is
  // empty, fill it with the value.
  T filter(const T &val) {
    if (start == -1) {
      for (int i = 0; i < order; i++) {
	data[i] = val;
	num_greater[i] = order - i - 1;
      }
      start = 0;
      return val;
    } else {
      int greater_than_val = 0;
      T res = val;
      for (int i = 0; i < order; i++) {
	if (i != start) {
	  if (data[i] <= val)
	    num_greater[i]++;
	  else
	    greater_than_val++;

	  if (data[i] < data[start])
	    num_greater[i]--;

	  if (num_greater[i] == order>>1)
	    res = data[i];
	}
      }
      data[start] = val;
      num_greater[start] = greater_than_val;
      start++;
      if (start == order) start = 0;
      return res;
    }
  }
};
  
#endif
