///////////////////////////////////////////////////////////////////////////////
//
//                               Input.h
//
// Defines a class that takes inputs from either a file or from a buffer.
// The class can open files by name, and can be given a set of directories
// to search for the file.  Once an input source is established, the class
// provides methods for reading in strings and numbers, while skipping white
// space and skipping over comments (started by '#')
//
// Classes defined for export:
//     Input - the input class
//
// Classes define for internal use
//     InputFile - a structure for holding an input source
//    
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_INPUT_H
#define UTILS_INPUT_H

#include <stdio.h>

#include <utils/Basic.h>
#include <utils/String.h>
#include <utils/Vector.h>

#include <sys/types.h>

__UTILS_BEGIN_NAMESPACE
    
class StringVector;

// structure for holding info about a input source
struct File {
    String		name;		// Name of file
    String		fullName;	// Name of file with full path
    FILE		*fp;		// File pointer
    void		*buffer;	// Buffer to read from (or NULL)
    char		*curBuf;	// Current location in buffer
    size_t		bufSize;	// Buffer size
    int			lineNum;	// Number of line currently reading
    bool		openedHere;	// true if opened by Input

    File();			// Too complex for inlining
};

// class for managing input sources
class Input {
 public:

    // Constructor (default Input reads from stdin)
    Input();

    // Destructor closes file if Input opened it.
    ~Input();

    // Adds a directory to list of directories to search to find named
    // files to open. Directories searched in order. By default, the list
    // is empty
    static void		addDirectoryFirst(const char *dirName);
    static void		addDirectoryLast(const char *dirName);

    // Adds directories that are named in the value of the given
    // environment variable. Directories may be separated by colons
    // or whitespace in the value.
    static void		addEnvDirectoriesFirst(const char *envVarName);
    static void		addEnvDirectoriesLast(const char *envVarName);

    // Removes given directory from list.
    static void		removeDirectory(const char *dirName);

    // given the full path of a file, push it's directory on to the front
    static void pushDirectoryOfFile(const char* full_filename);

    // remove the directory at the front of the list
    static void popDirectory();

    // Clears list of directories
    static void		clearDirectories();

    // prints the list of directories
    static void printDirectories(FILE* = stdout);

    // Returns the current list of directories.
    static const Vector<String*> &getDirectories();

    // Sets initial file pointer to read from. 
    void		setFilePointer(FILE *newFP);

    // Opens named file, sets file pointer to result. Returns false on error. 
    // If okIfNotFound is false (the default), it prints an error message if 
    // the file could not
    // be found.
    bool		openFile(const char *fileName,
				 bool okIfNotFound = false);

    // Closes the file 
    void		closeFile();

    // empties input, without closing file
    void releaseFile();

    // Returns pointer to current file, or NULL if reading from buffer
    FILE *		getCurFile() const;

    // Returns full name of current file, or NULL if reading from buffer
    const char *	getCurFileName() const;

    int getLineNum() const;
    void setLineNum(int);

    // Returns orignal name of current file, or NULL if reading from buffer
    const char*         getOrigFileName() const;

    // Sets up buffer to read from and its size
    void		setBuffer(void *bufPointer, size_t bufSize);
    void     getBuffer(void*&bufPointer, int& bufSize)
        { bufPointer = _cur_file->buffer; bufSize = _cur_file->bufSize; }

    // Returns number of bytes read from buffer. Returns 0 if not
    // reading from a buffer.
    size_t		getNumBytesRead() const;

    // Reads next character from current file/buffer. Returns false on
    // EOF or error.
    bool		get(char &c);

    // Reads next ASCII character from current buffer. Returns false on
    // EOF or error.
    bool		getASCIIBuffer(char &c);

    // Reads next ASCII character from current file. Returns false on
    // EOF or error.
    bool		getASCIIFile(char &c);

    // read the entire contents of the file into bytes, with size being the
    // number of bytes read
    bool readAll(unsigned char*& bytes, int& size);

    // Reads next ASCII format hex value from current file/buffer.
    // Returns false on EOF or error.
    bool		readHex(unsigned long &l);

    // Reads item of particular type from current file pointer/buffer. All
    // skip white space before reading and return false on EOF or if
    // item could not be read.
    bool		read(char	    &c);
    bool		read(String       &s, char delimiter=' ');
    bool		read(Name	    &n, bool validIdent = false);
    bool		read(int	    &i);
    bool		read(unsigned int   &i);
    bool		read(short	    &s);
    bool		read(unsigned short &s);
    bool                readWord(String &s, const char* word_chars="");
    // was ... C-api: name=readInt32
    //     but typedef makes this redundant.
    //bool		read(int32_t	    &l);
    // was ... C-api: name=readUInt32
    //     but typedef makes this redundant.
    //bool		read(uint32_t	    &l);
    bool		read(float	    &f);
    bool		read(double	    &d);
    int                 read(unsigned char* data, int size);
    
    // Returns true if current file/buffer is at EOF
    bool		eof() const;

    // Puts a just-read character or string back in input stream/buffer
    void		putBack(char c);
    void		putBack(const char *string);

    Input& operator= (const Input& input);

    // Skips over white space in input. Pops file if EOF is hit.
    // Returns false on error.
    bool		skipWhiteSpace();

    // Returns true if reading from memory buffer rather than file
    bool		fromBuffer() const
	{ return (_cur_file->buffer != NULL); }

    // Return the current offset into the file or buffer
    off_t              getPos(void);

    // Return the total size of the file or buffer
    off_t              getSize(void);

    // Seek the offset pointer like lseek
    int                seek(off_t offset, int whence);

  private:

    // Looks for named file and opens it. Returns NULL if not found.
    FILE *		findFile(const char *fileName, String &fullName) const;

    // Initializes reading from file
    void		initFile(FILE *newFP, const char *fileName,
                         String *fullName, bool openedHere);
        

    // Returns number of bytes left in current buffer
    size_t		freeBytesInBuf() const
	{ return (_cur_file->bufSize -
		  (_cur_file->curBuf - (char *) _cur_file->buffer)); }

    // Reads integer, unsigned integer, or floating-point number.
    // Returns false on EOF or error
    bool		readInteger(int &l);
    bool		readUnsignedInteger(unsigned int &l);
    bool		readReal(double &d);

    // Reads string of decimal or digits into num_store
    void readDigits();

    // Attempts to recover from failure reading digits
    void restore(char c = '\0');

  private:
    struct File	*_cur_file;	// Current input
    Vector<char> _back_buf;         // backup buffer
    Vector<char> _temp_store;       // temporary string storage

    static Vector<String*> *_directories;	// Directory search path.
};

__UTILS_END_NAMESPACE

#endif
