///////////////////////////////////////////////////////////////////////////////
//
//                               Generator.h
//
// Defines routines for parsing specification strings and classes for 
// managing and generating reconfigurable interfaces
//
// Classes defined for export:
//   GeneratorBase - base class for a table of generic reconfigurable 
//      interfaces that can be chosen from and generated using a 
//      specification string
//   Token - an element in a list of name/attribute pairs derived or
//      made into a specification string
//
// Templates defined for export:
//   GeneratorBase - base class for a table of specific reconfigurable 
//      interfaces that can be chosen from and generated using a 
//      specification string
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_GENERATOR_H
#define UTILS_GENERATOR_H

#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE

class SymbolTable;
class CreatorBase;
class ConfigFile;
template <class T> class Creator;

// Token - an element in a list of name/attribute pairs derived or
//    made into a specification string
struct Token {
    char* name;   // the name of the value
    char* value;  // the value
};

// GeneratorBase - base class for a table of reconfigurable interfaces
//    that can be chosen from and generated using a specification string
class GeneratorBase {
  public:
    GeneratorBase(int table_size);
    virtual ~GeneratorBase();

    void registerInterface(const char* tag,
                           void* (*creator)(GeneratorBase*,
                                            ConfigFile*, SymbolTable*),
                           Token* tokens);
    void registerInterface(const char* tag,
                           CreatorBase*, Token* tokens);
    void* interface(const char* spec_string, SymbolTable* globals);
    void* interface(ConfigFile*, SymbolTable* globals);

  private:
    SymbolTable* _interfaces;   // table of interfaces indexed by tag
};

template <class T>
class Generator : public GeneratorBase {
 public:
  Generator(int table_size = 20) : GeneratorBase(table_size) {}
  virtual ~Generator() {}
  void registerInterface(const char* tag,
                         T* (*creator)(Generator<T>*,
                                       ConfigFile*, SymbolTable*),
                         Token* tokens) {
    GeneratorBase::registerInterface(tag,
                                     (void* (*)(GeneratorBase*, ConfigFile*,
                                                SymbolTable*)) creator,
                                     tokens);
  }

  inline void registerInterface(const char* tag, Creator<T>* creator,
                                Token* tokens);
  virtual T* interface(const char* spec_string, SymbolTable* globals)
    { return (T*) GeneratorBase::interface(spec_string, globals); }
  virtual T* interface(ConfigFile* config_spec, SymbolTable* globals)
    { return (T*) GeneratorBase::interface(config_spec, globals); }
};

template <class T>
inline void Generator<T>::registerInterface(const char* tag,
                                                  Creator<T>* creator,
                                                  Token* tokens)
{ registerInterface(tag, (CreatorBase*) creator, tokens);  };

__UTILS_END_NAMESPACE

#endif
