/***************************************************************************

                                ConfigFile.h

  Defines a class which reads a hierarchical, typed parameter file and
  can be queried for the values set in that parameter file.  In addition,
  the user can set the values after the parameter file has been read.

  Vastly more documentation can be found in ConfigFile.cc

  Classes defined for export:
    ConfigFile - the configuration file class

  Typedef defined for export:
    ConfigElemFunc - creation functions for config elements, used by
                     ConfigFile::registerElem
    UTL_ConfigFile - C/namespace braindead way of accessing utils::ConfigFile
      type

  C Cover functions defined for the equivalent ConfigFile methods:
    CF_new, CF_delete, CF_open, CF_parse, CF_close, CF_get, CF_getInt
    CF_getFloat, CF_getDouble, CF_getChar, CF_getString, CF_getBool
    CF_set, CF_setValues, CF_getInts, CF_getFloats, CF_getDoubles, CF_getChars
    CF_getStrings, CF_getBools, CF_intType, CF_floatType, CF_doubleType
    CF_charType, CF_stringType, CF_boolType

***************************************************************************/
#ifndef UTILS_CONFIG_FILE_H
#define UTILS_CONFIG_FILE_H

#include <utils/Basic.h>
#include <utils/Type.h>

#ifdef __cplusplus

__UTILS_BEGIN_NAMESPACE

class StructElem;
class ConfigElem;
class Input;
class Output;

typedef ConfigElem* (*ConfigElemFunc)();

class ConfigFile {
 public:
  ConfigFile(StructElem* = (StructElem*) 0L);
  ConfigFile(ConfigElem*);
  ~ConfigFile();

  bool open(const char* name);
  bool parse(const char* source);
  bool parse(Input& input) { return read(input); }
  void close();
  void initRoot(StructElem*);
  void write(Output& output);
  void clear();

  int get(Type type, const char* name, void* data, int max_num=1,
          void* default_data=0L, int num_defaults=1);

  int getInt(const char* name, int def=0) {
    int data;
    if (get(intType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  float getFloat(const char* name, float def=0.0) {
    float data;
    if (get(floatType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  double getDouble(const char* name, double def=0.0) {
    double data;
    if (get(doubleType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  char getChar(const char* name, char def='\0') {
    char data;
    if (get(charType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  const char* getString(const char* name, const char* def = "") {
    const char* data;
    if (get(stringType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  bool getString(const char* name, const char* def, char* dest, int max_len);

  bool getBool(const char* name, bool def=false) {
    bool data;
    if (get(boolType(), name, &data, 1, &def) > 0)
      return data;
    else
      return def;
  }

  ConfigElem* getStruct(const char* name) {
    ConfigElem* elem;
    if (getStructs(name, &elem, 1) < 1)
      return NULL;
    return elem;
  }
  bool getStruct(const char* name, utils::ConfigFile& params);

  int getInts(const char* name, int* data, int max_num=1,
              int* default_data=(int*) 0L, int num_defaults=1) {
    return get(intType(), name, data, max_num,
               default_data, num_defaults);
  }

  int getFloats(const char* name, float* data, int max_num=1,
                float* default_data=(float*) 0L, int num_defaults=1) {
    return get(floatType(), name, data, max_num,
               default_data, num_defaults);
  }

  int getDoubles(const char* name, double* data, int max_num=1,
                 double* default_data=(double*) 0L, int num_defaults=1) {
    return get(doubleType(), name, data, max_num,
               default_data, num_defaults);
  }

  int getChars(const char* name, char* data, int max_num=1,
               char* default_data=(char*) 0L, int num_defaults=1) {
    return get(charType(), name, data, max_num,
               default_data, num_defaults);
  }

  int getStrings(const char* name, const char** data, int max_num=1,
                 const char** default_data=(const char**) 0L,
                 int num_defaults=1) {
    return get(stringType(), name, data, max_num,
               default_data, num_defaults);
  }

  int getBools(const char* name, bool* data, int max_num=1,
               bool* default_data=(bool*) 0L, int num_defaults=1) {
    return get(boolType(), name, data, max_num, default_data, num_defaults);
  }
  int getStructs(const char* name, ConfigElem** data, int max_num=1);

  ConfigElem* lookup(const char* name);
  ConfigElem* set(const char* name, const char* value,
                  bool ignore_duplicates=true);
  ConfigElem* set(Type type, const char* name, void* data, int num_values=1,
                  bool ignore_duplicates=true);

  bool setInt(const char* name, int value) {
    return set(intType(), name, &value);
  }
  bool setFloat(const char* name, float value) {
    return set(floatType(), name, &value);
  }
  bool setDouble(const char* name, double value) {
    return set(doubleType(), name, &value);
  }
  bool setChar(const char* name, double value) {
    return set(charType(), name, &value);
  }
  bool setString(const char* name, const char* value) {
    return set(stringType(), name, &value);
  }
  bool setBool(const char* name, bool value) {
    return set(boolType(), name, &value);
  }
  bool setInts(const char* name, int* values, int num_values) {
    return set(intType(), name, values, num_values);
  }
  bool setFloats(const char* name, float* values, int num_values) {
    return set(floatType(), name, values, num_values);
  }
  bool setDoubles(const char* name, double* values, int num_values) {
    return set(doubleType(), name, values, num_values);
  }
  bool setChars(const char* name, char* values, int num_values) {
    return set(charType(), name, values, num_values);
  }
  bool setStrings(const char* name, const char** values, int num_values) {
    return set(stringType(), name, values, num_values);
  }
  bool setBools(const char* name, bool* values, int num_values) {
    return set(boolType(), name, values, num_values);
  }

  int numValues(const char* name);

  ConfigElem* setStruct(const char* name, utils::ConfigFile& params);
  ConfigElem* mergeStruct(const char* name, utils::ConfigFile& params);

  ConfigElem* getRoot() const;
  ConfigElem* makeElem(const char*, bool ignore_duplicates=false,
                       bool create_new=true);

  ConfigFile& operator=(const ConfigFile& in);

  static Type intType();
  static Type floatType();
  static Type doubleType();
  static Type charType();
  static Type stringType();
  static Type boolType();

  static void init();
  static ConfigFile* copy(ConfigFile*);
  static void copy(const ConfigFile& src, ConfigFile& dest);
  static int verbosity();

 private:
  bool read(Input&, StructElem* = (StructElem*) 0L);

 private:
  StructElem* _root;  /* the root element */
};

__UTILS_END_NAMESPACE

/* Used for convenience by namespace braindead compilers */
typedef __UTILS::ConfigFile *UTL_ConfigFile;

extern "C" {

#else

  /* used to represent utils::ConfigFile to C compilers */
  typedef void* UTL_ConfigFile;

#endif

  /* the C cover function prototypes */

  extern UTL_ConfigFile CF_new();
  extern void CF_delete(UTL_ConfigFile cf);
  extern bool CF_open(UTL_ConfigFile cf, const char* name);
  extern bool CF_parse(UTL_ConfigFile cf, const char* source);
  extern void CF_close(UTL_ConfigFile cf);

  extern int CF_get(UTL_ConfigFile, int type, const char* name,
                    void* data, int max_num,
                    void* default_data, int num_defaults);
  extern int CF_getInt(UTL_ConfigFile cf,const char* name, int def);
  extern float CF_getFloat(UTL_ConfigFile cf, const char* name, float def);
  extern double CF_getDouble(UTL_ConfigFile cf, const char* name, double def);
  extern char CF_getChar(UTL_ConfigFile cf, const char* name, char def);
  extern const char* CF_getString(UTL_ConfigFile cf, const char* name,
                                  const char* def);
  extern bool CF_getBool(UTL_ConfigFile cf, const char* name, bool def);

  extern bool CF_set(UTL_ConfigFile cf,
                     const char* name, const char* value);

  extern int CF_getInts(UTL_ConfigFile cf, const char* name,
                        int* data, int max_num,
                        int* default_data, int num_defaults);
  extern int CF_getFloats(UTL_ConfigFile cf, const char* name,
                          float* data, int max_num,
                          float* default_data, int num_defaults);
  extern int CF_getDoubles(UTL_ConfigFile cf, const char* name,
                           double* data, int max_num,
                           double* default_data, int num_defaults);
  extern int CF_getChars(UTL_ConfigFile cf, const char* name,
                         char* data, int max_num,
                         char* default_data, int num_defaults);
  extern int CF_getStrings(UTL_ConfigFile cf, const char* name,
                           const char** data, int max_num,
                           const char** default_data, int num_defaults);
  extern int CF_getBools(UTL_ConfigFile cf, const char* name,
                         bool* data, int max_num,
                         bool* default_data, int num_defaults);

  extern int CF_intType();
  extern int CF_floatType();
  extern int CF_doubleType();
  extern int CF_charType();
  extern int CF_stringType();
  extern int CF_boolType();

#ifdef __cplusplus
}
#endif

#endif
