#ifndef _CANNEDDATAWRITE_H
#define _CANNEDDATAWRITE_H

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
//#include <stdint.h>

#include <utils/String.h>
#include <utils/List.h>

__UTILS_BEGIN_NAMESPACE

struct CDWThrottleQueueElt
{
	size_t size;
	struct timeval tv;
};

class CannedDataWriter {
 public:
	CannedDataWriter();
	~CannedDataWriter();
	
	/*
	 * Open a file for writing.  Create the
	 * file if it doesn't exist.  If it does exist, 
	 * clobber it iff mode doesn't include O_EXCL.  Set 
	 * file type and version data accordingly Ignore
	 * the other mode bits for now.  
 	 */
	void open(const char *filename, mode_t mode = 0);


	/* All of the writeXXX functions must be called after a successful
	 * open, and may not be called after a close.  With the exception
	 * of writeRecord, all of these functions can only be called once
	 * for a given file.  It is not strictly necessary to call any
	 * of these functions to have a valid data file.
	 */

	/*
	 * Set the file type. 
	 */
	void writeType(CannedDataType type);
	
	/* 
	 * Set the version information for this file.  
	 */
	
	void writeVersion(uint32_t major, uint32_t minor);

	/*
	 * Add a description of this data.  The string should be null
	 * terminated.
	 */
	void writeDescription(const char *desc);
	
	/*
	 * Write some custom header information for this file.
	 */
	void writeHeader(void *buf, size_t size);
	
	/*
	 * Write out a data record to the file with the given time stamp 
	 */
	void writeData(void *buf, size_t size, uint64_t secs, uint64_t usecs);

	/*
	 * Set the rate, at which we throttle data output, and the
	 * window over which this is enforced. 
	 *
	 * For example, setThrottle(1, 1) would make sure that no more than
	 * 1 MB of data was written in any given 1 second window.  This is
	 * not the same as setThrottle(5, 5), which would make sure that
	 * no more than any 5 MB was written in any 5 second window.
	 *
	 * To prevent throttle bandwidth from being exceeded, write requests
	 * may be dropped on the floor.
	 * 
	 * The throttle defaults to unlimited.  If it is set too high and we
	 * get too much data, you run the risk of making this process go
	 * to sleep while buffers clear out to disk.  Pick your poison...
	 */
	void setThrottle(size_t size, long window);
	


	/* 
	 * Write all metadata to disk, and finish out the file, closing it.
	 * No further write* calls will be valid after close() is invoked
	 */
	void close();
 private:
	/*
	 * Write out an index record with this ID and length
	 */
	void             _writeIdxRecord(CannedDataFieldID id, const void *buf, uint16_t size);

	/*
	 * Fwrite this stuff out to the _file pointer, checking for errors.
	 */
	void             _doFwrite(const void *buf, size_t size, FILE *file);

	/*
	 * Create the next data file to be used 
	 */
        void             _openNextDataFile();

	/*
	 * If the file exists and we don't want to clobber it, toss an error 
	 */
	void             _checkClobber(const String &filename);

	/*
	 * Write a data record to the data file
	 */
	void             _writeDataRecord(void *buf, size_t size);

	/* 
	 * Reset object state so we're ready for a new open.  Do it carefully,
	 * as we may end up doing this in an error condition 
	 */
	void             _cleanup();

	/*
	 * Initialize data elements 
	 */
	void             _init();

	/*
	 * Start a new data file
	 */
	void             _newDataFile();
	
	/*
	 * Check whether a write of this size would violate the throttle.  
	 * If not, then add size to the throttle queue and increase the
	 * count of data in flight by size, and return 0.  Otherwise
	 * don't record this one and return 1;
	 */
	int _checkThrottle(size_t size);

	/* Private data */
	int              _state;
	uint32_t         _headers_written;
	uint32_t         _set_id;
        uint64_t         _last_secs_stamp;
        uint64_t         _last_usecs_stamp;
	String           _base_name;
	int              _data_file_num;
	size_t           _data_file_size;
	mode_t           _mode;
	FILE            *_idx_file, *_data_file;
	size_t           _throttle_size;
	long             _throttle_window;
	size_t           _throttle_data_in_flight;
	List<CDWThrottleQueueElt> _throttle_queue;
};

__UTILS_END_NAMESPACE

#endif
	

