#ifndef _CANNED_BASE_H
#define _CANNED_BASE_H

#include <utils/Basic.h>
#include <utils/Time.h>
#include <stdint.h>
#include <utils/CannedDataAccess.h>


__UTILS_BEGIN_NAMESPACE

/*
 * Base class for Canning class readers.  
 */

class CannedReader
{
 public:
        /*
         * Factory function; opens a file and returns a reader of the appropriate
         * type
         */
        static CannedReader   *create(const char *filename);
        
        virtual ~CannedReader() {};
        /*
         * Open a file for reading.  Load an index of the contents,
         * so this could take a non-trivial amount of time.
         */
        virtual void           open(const char *filename)                   = 0;
        
        /*
         * Close a file, prep the class for a new open()
         */
        virtual void           close()                                      = 0;

	/*
	 * Return the creation time of the dataset, in seconds
	 * from the epoch.  0 if no creation time was stored.
	 */
        virtual uint64_t       getCreationTime()                            = 0;

        
        /*
         * Return the version of this data format being used
         */
        virtual void           getVersion(uint32_t *major, uint32_t *minor) = 0;

        /*
         * Get a file pointer to the beginning of the header information.
         * The file pointer is only valid until the next member function
         * call
         */
        virtual FILE          *getHeader()                                  = 0;

	/*
	 * Return the total size of the header
	 */
        virtual size_t         getHeaderSize()                              = 0;

	/* 
	 * Adjust reading position to be the record
	 * closest to the passed time.  
	 */
        virtual void           seek(const Time &when)                       = 0;

	/*
	 * Adjust reading position to be the record
	 * _in the past_ closest to the passed time.
	 * 
	 * Returns -1 and doesn't move the file
	 * pointer if there is no record earlier
	 * than this one.  Returns 0 on success
	 */
        virtual int            seekBefore(const Time &when)                 = 0;

	/* 
	 * Returns 1 if the last time requested was
	 * beyond the end of the file.  Note the
	 * seek will still point at the last 
	 * record of the file and return valid
	 * results in a read, though.
	 */
        virtual int            eof()                                        = 0;

        /*
         * Rewind to the start of the video
         */
        virtual void           first()                                      = 0;

        /*
         * Load the next frame of the video.
         */
        virtual void           next()                                       = 0;

        /*
         * Load the previous frame of the video.
         */
        virtual void           prev()                                       = 0;

        /*
         * Load the last frame of the video. 
         */
        virtual void           last()                                       = 0;

        /*
         * Get the timestamp associated with the current record
         */
        virtual Time           getCurTimeStamp()                            = 0;

        /*
         * Get the timestamp associated with the first record of the
         * file
         */
        virtual Time           getFirstTimeStamp()                          = 0;

	/*
	 * Get the timestamp associated with the last
	 * record of the file 
	 */
        virtual Time           getLastTimeStamp()                           = 0;

	/*
	 * Get the timestamp of the record before this
	 * one.  Returns 0 on success, -1 if this is the
	 * first record in the file
	 */
        virtual int            getPrevTimeStamp(Time *time)                 = 0;

	/*
	 * Get the timestamp of the record after this
	 * one.  Returns 0 on success, -1 if this is the
	 * last record in the file
	 */
        virtual int            getNextTimeStamp(Time *time)                 = 0;

        /*
         * Get the number of frames in the file
         */
        virtual unsigned int   getNumRecords()                              = 0;
};


__UTILS_END_NAMESPACE

#endif
