/***************************************************************************

                                cSymbolTable.cc

  Implements C cover functions to give C programmers access to the SymbolTable
  functionality.  See StringDict.cc for more documentation on each
  function

  C Cover functions implemented for the equivalent StringDict methods:
    SYM_new, SYM_delete, SYM_set, SYM_remove, SYM_clear, SYM_get, SYM_probe
    SYM_getInt, SYM_getFloat, SYM_getDouble, SYM_getString, SYM_applyToAll

***************************************************************************/

#include <utils/SymbolTable.h>

__UTILS_BEGIN_NAMESPACE

// helper class to allow C programmers to define functions which will
// be symbol managers
class cSymbolManager : public SymbolManager {
public:
    cSymbolManager(SYM_ManageFunc linker, SYM_ManageFunc unlinker,
                   void* cb_data) {
        _linker = linker;
        _unlinker = unlinker;
        _cb_data = cb_data;
    }

    virtual void link(void* data) {
        if (_linker)
            (*_linker)(data, _cb_data);
    }
    virtual void unlink(void* data) {
        if (_unlinker)
            (*_unlinker)(data, _cb_data);
    }
private:
    SYM_ManageFunc _linker;
    SYM_ManageFunc _unlinker;
    void* _cb_data;
};

__UTILS_END_NAMESPACE

extern "C" {

UTL_SymbolTable SYM_new(int num_buckets)
{
    return new __UTILS::SymbolTable(num_buckets);
}

void SYM_delete(UTL_SymbolTable table)
{
    delete table;
}

bool SYM_set(UTL_SymbolTable table, const char* name, void* data,
             SYM_ManageFunc linker, SYM_ManageFunc unlinker, void* cb_data,
             bool overwrite)
{
    __UTILS::SymbolManager* mgr = (__UTILS::SymbolManager*) NULL;
    if (linker || unlinker) 
        mgr = new __UTILS::cSymbolManager(linker, unlinker, cb_data);
    return table->set(name, data, mgr, overwrite);
}

bool SYM_remove(UTL_SymbolTable table, const char* name)
{
    return table->remove(name);
}

void SYM_clear(UTL_SymbolTable table)
{
    table->clear();
}

void* SYM_get(UTL_SymbolTable table, const char* name)
{
    return table->get(name);
}

bool SYM_probe(UTL_SymbolTable table, const char* name, void** res)
{
    return table->get(name, *res);
}
    
int SYM_getInt(UTL_SymbolTable table, const char* name)
{
    return table->getInt(name);
}

float SYM_getFloat(UTL_SymbolTable table, const char* name)
{
    return table->getFloat(name);
}

double SYM_getDouble(UTL_SymbolTable table, const char* name)
{
    return table->getDouble(name);
}

const char* SYM_getString(UTL_SymbolTable table, const char* name)
{
    return table->getString(name);
}

struct SymCallbackPackage {
    SYM_CallbackFunc func;
    void* cb_data;
};

void c_func(const char* key, __UTILS::Symbol* sym, void* data)
{
    SymCallbackPackage* pkg = (SymCallbackPackage*) data;
    (*pkg->func)(key, sym->data, pkg->cb_data);
}

void SYM_applyToAll(UTL_SymbolTable table, SYM_CallbackFunc func, void* data)
{
    SymCallbackPackage pkg;
    pkg.func = func;
    pkg.cb_data = data;
    table->applyToAll(c_func, (void*) &pkg);
}

}
