///////////////////////////////////////////////////////////////////////////////
//
//                               Type.cc
//
// Implements the class for defining a type.  This class lets types be checked
// at run-time, not just at compile time.  Type names are stored in a static
// dictionary so that types can be looked up by name.
//
// Classes implemented for export:
//   Type - the hierarchical type class
//
// Classes defined for internal use
//   TypeData - the data stored in a type entry
//   TypeInitializer - initializer for the type class
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>

#include <utils/Basic.h>

#include <string.h>
#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#include <utils/Mutex.h>
#include <utils/Type.h>
#include <utils/String.h>
#include <utils/StringDict.h>

__UTILS_BEGIN_NAMESPACE

// the data stored in a type entry
struct TypeData {
    Name name;             // the name of the type
    Type parent;           // the parent type of this type
    void* (*createMethod)(Name, void*); // how to create an instance 
    void* clientData;        // closure
    int depth;               // the depth into the type hierarchy this type is
};

StringDict<void*>* Type::nameDict = NULL;
int Type::nextIndex;
int Type::arraySize;
TypeData* Type::typeData = NULL;

// semaphore for making type creation thread safe - called TypeMutex in
// the global symbol table
static Mutex* _Type_Mutex = (Mutex*) -1;

// static member to lookup name in the type dictionary.  Returns a badType
// if the name is not a type
Type Type::fromName(Name name)
{
    void* storage;
    if (nameDict->find(name.getString(), storage)) {
        Type res;
        memcpy((char*) &res, (char*) &storage, sizeof(Type));

        return res;
    } else
        return Type::badType();
}

// get a type's name
Name Type::getName() const 
{
    return typeData[getKey()].name;
}

// get a type's parent
Type Type::getParent() const
{
    return typeData[getKey()].parent;
}

// return the empty, or archetypal "bad", type
Type Type::badType()
{
    return Type();
}

// return true is this type is derived from t
bool Type::isDerivedFrom(Type t) const
{
    if (t == *this)
        return true;

    TypeData& elem = typeData[getKey()];
    TypeData& other = typeData[t.getKey()];
    if (elem.depth <= 1 || other.depth >= elem.depth)
        return false;
    else
        return elem.parent.isDerivedFrom(t);
}

// returns true if there is a creation method for the type
bool Type::canCreateInstance() const
{
    return typeData[getKey()].createMethod != NULL;
}

// create a instance of this type
void* Type::createInstance() const
{
    if (canCreateInstance()) {
        TypeData& elem = typeData[getKey()];
        return elem.createMethod(elem.name, elem.clientData);
    } else
        return NULL;
}

// make a type descended from parent named name with creation method 
// createMethod
Type Type::createType(Type parent, Name name,
                          void* (*createMethod)(Name, void*),
                          void* clientData)
{
    if (_Type_Mutex) {
        if (_Type_Mutex == (Mutex*) -1) {
            if (Mutex_Factory) {
                _Type_Mutex = (*Mutex_Factory)();
                if (_Type_Mutex) 
                    _Type_Mutex->lock();
            }
        } else
            _Type_Mutex->lock();
    }
        
    Type res;

    if (nextIndex >= arraySize)
        expandTypeData();
    
    TypeData& elem = typeData[nextIndex];
    elem.name = name;
    elem.parent = parent;
    elem.createMethod = createMethod;
    elem.clientData = clientData;
    if (parent.getKey()) 
        elem.depth = typeData[parent.getKey()].depth+1;
    else
        elem.depth = 1;

    res.index = nextIndex++;

    void* storage;
    memcpy((char*) &storage, (char*) &res, sizeof(Type));
    if (!nameDict->enter(name.getString(), storage))
        printf("Warning: Redefining type %s from %s (%d)\n",
               name.getString(), parent.getName().getString(), res.getKey());

    if (_Type_Mutex && (_Type_Mutex != (Mutex*) -1))
        _Type_Mutex->unlock();

    //    printf("Creating type %s from %s (%d)\n", name.getString(),
    //           parent.getName().getString(), res.getKey());

    return res;
}

#define TYPE_CHUNK_SIZE 100

// initialize the type system
void Type::init()
{
    if (nameDict)
        return;

    nameDict = new StringDict<void*>(500);

    nextIndex = 1;
    arraySize = TYPE_CHUNK_SIZE;
    typeData = new TypeData[TYPE_CHUNK_SIZE];
    memset((char*) typeData, 0, TYPE_CHUNK_SIZE*sizeof(TypeData));
    typeData[0].name = Name("");
    typeData[0].parent = badType();
    typeData[0].createMethod = NULL;
    typeData[0].depth = 0;
}

// make room for more types
void Type::expandTypeData()
{
    int new_size = arraySize + TYPE_CHUNK_SIZE;
    TypeData* new_data = new TypeData[new_size];
    memcpy((char*) new_data, (char*) typeData, arraySize*sizeof(TypeData));
    delete [] typeData;
    arraySize = new_size;
    typeData = new_data;
}

__UTILS_END_NAMESPACE
