///////////////////////////////////////////////////////////////////////////////
//
//                               StringDict.cc
//
// Implements a string dictionary i.e., it associates strings with void*'s
//
// Classes implemented for export:
//     BasicStringDict - a dictionary of strings and void*
// 
// Structures for internal use
//     StringDictEntry - an entry in the dictionary
//    
///////////////////////////////////////////////////////////////////////////////

#include <utils/Basic.h>

#include <stdlib.h>
#include <string.h>

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#include <utils/StringDict.h>
#include <utils/String.h>

__UTILS_BEGIN_NAMESPACE

struct StringDictEntry {
    void* data;          // the entry data
    unsigned long key;   // the hash value of the string tag
    char* str;           // the string tag 
    StringDictEntry* next;   // the next element in the dictionary
};

BasicStringDict::BasicStringDict(int entries)
{
    _table_size = entries;
    _buckets = new StringDictEntry*[entries];
    memset((char*) _buckets, 0, sizeof(StringDictEntry*)*entries);
}

BasicStringDict::~BasicStringDict()
{
    clear();
    delete [] _buckets;
}

void BasicStringDict::applyToAll(void (*rtn)(const char* key,
                                                      void *value) )
{
    for (int i=0;i<_table_size;i++)
        for (StringDictEntry* elem=_buckets[i]; elem; elem=elem->next)
            (*rtn)(elem->str, elem->data);
}
    
void BasicStringDict::applyToAll(void (*rtn)(const char*,void*,void*),
                                          void* cb_data)
{
    for (int i=0;i<_table_size;i++)
        for (StringDictEntry* elem=_buckets[i]; elem; elem=elem->next)
            (*rtn)(elem->str, elem->data, cb_data);
}
    
void BasicStringDict::clear()
{
    StringDictEntry* doomed;
    for (int i=0;i<_table_size;i++) {
        for (StringDictEntry* elem = _buckets[i]; elem; ) {
            delete [] elem->str;
            doomed = elem;
            elem=elem->next;
            delete doomed;
        }
        _buckets[i] = 0L;
    }
}    
            
bool BasicStringDict::enter(const char* str, void *value)
{
    unsigned long key = String::hash(str);
    StringDictEntry* elem = findEntry(key, str);

    if (elem) {
        elem->data = value;
        return false;
    }

    elem = new StringDictEntry;
    elem->data = value;
    elem->key = key;
    elem->str = String::copy(str);

    int index = key % _table_size;
    elem->next = _buckets[index];
    _buckets[index] = elem;

    return true;
}

bool BasicStringDict::find(const char* str, void*& value) const
{
    StringDictEntry* elem = findEntry(String::hash(str), str);
    if (!elem)
        return false;
    value = elem->data;
    return true;
}

bool BasicStringDict::remove(const char* str)
{
    unsigned long key = String::hash(str);
    int index = key % _table_size;
    StringDictEntry* prev = 0L;
    for (StringDictEntry* elem=_buckets[index];elem;elem=elem->next) {
        if (key == elem->key && !strcmp(str, elem->str)) {
            if (prev) 
                prev->next = elem->next;
            else
                _buckets[index] = elem->next;
            delete [] elem->str;
            delete elem;
            return true;
        }
        prev = elem;
    }
    return false;
}

StringDictEntry*
BasicStringDict::findEntry(unsigned long key,
                                    const char* str) const
{
    int index = key % _table_size;
    for (StringDictEntry* elem=_buckets[index]; elem; elem=elem->next)
        if (key == elem->key && !strcmp(str, elem->str))
            return elem;
    return (StringDictEntry*) NULL;
}
            
StringDictIteratorBase::StringDictIteratorBase(BasicStringDict& dict)
  : _dict(dict)
{
  _index = -1;
  _next = NULL;
}

void* StringDictIteratorBase::first()
{
  _index = -1;
  _next = NULL;
  return next();
}

void* StringDictIteratorBase::next()
{
  while (1) {
    if (!_next) {
      while (1) {
        _index++;
        if (_index >= _dict._table_size)
          return NULL;
        _next = _dict._buckets[_index];
        if (_next)
          return _next->data;
      }
    }
    _next = _next->next;
    if (_next)
      return _next->data;
  }
}

const char* StringDictIteratorBase::key()
{
  if (_next) {
    return _next->str;
  } else
    return NULL;
}

__UTILS_END_NAMESPACE
