///////////////////////////////////////////////////////////////////////////////
//
//                               String.cc
//
// Implements classes for creating and using smart strings and unique strings
//
// Classes implemented for export:
//     String - a smart string class
//     Name - a unique string class
//     NameEntry - representation of a unique string
//
// Structures implemented for internal usage
//     NameChunk - Allocates blocks or chunks for name strings.
//
///////////////////////////////////////////////////////////////////////////////

#include <utils/Basic.h>

#include <ctype.h>
#include <string.h>
#include <stdio.h>

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#include <utils/String.h>
#include <utils/Mutex.h>

///////////////////
// struct NameChunk
// Allocates blocks or chunks for name strings.

#define CHUNK_SIZE	4000

__UTILS_BEGIN_NAMESPACE

// semaphore for making name entries thread safe - called StringMutex in
// the global symbol table
static Mutex* _String_Mutex = (Mutex*) -1;

// acquire a lock on the string mutex semaphore, if necessary
static void acquire_lock() {
    if (_String_Mutex) {
        if (_String_Mutex == (Mutex*) -1) {
            if (Mutex_Factory)
                _String_Mutex = (*Mutex_Factory)();
            else
                _String_Mutex = 0L;
            if (_String_Mutex) 
                _String_Mutex->lock();
        } else
            _String_Mutex->lock();
    }
}

// release a lock on the string mutex semaphore, if necessary
static void release_lock() {
    if (_String_Mutex && (_String_Mutex != (Mutex*) -1))
        _String_Mutex->unlock();
}

struct NameChunk
{
    char				mem[ CHUNK_SIZE ];
    char				*curByte;
    int					bytesLeft;
    struct NameChunk	*next;
};

////////////////////////
// class NameEntry

int NameEntry::_name_table_size = 0;
NameEntry **NameEntry::_name_table = (NameEntry**) NULL;
NameChunk *NameEntry::_chunk = (NameChunk*) NULL;

NameEntry::NameEntry( const char *s, unsigned long h, NameEntry *n )
{
    _string = s;
    _hash_value = h;
    _next = n;
}

void NameEntry::initClass()
{
    // note no mutex locking is required, since this is _guaranteed_ to 
    // happen during the C++ global constructor initialization

    if (_name_table_size)
        return;
    
    _name_table_size = 1999;
    _name_table	  = new NameEntry*[ _name_table_size ];

    for ( int i = 0; i < _name_table_size; i++ )
        _name_table[ i ] = 0L;

    _chunk = 0L;
}

void NameEntry::_cleanClass()
{
    acquire_lock();
    for ( int i = 0; i < _name_table_size; i++ ) {
        NameEntry *entry = _name_table[i];
        NameEntry *nextentry;

        while ( entry != NULL ) {
            nextentry = entry->_next;
            delete entry;
            entry = nextentry;
        }
    }

    NameChunk* nextchunk;

    while ( _chunk != NULL ) {
        nextchunk = _chunk->next;
        delete _chunk;
        _chunk = nextchunk;
    }

    delete [] _name_table;
    release_lock();
}

bool NameEntry::isEmpty() const
{
    return (_string[0] == '\0');
}

bool NameEntry::isEqual( const char *s ) const
{
    return (_string[0] == s[0] && ! strcmp(_string, s));
}

const NameEntry * NameEntry::insert( const char *s )
{
    unsigned long		h = String::hash(s);
    unsigned long		i;
    NameEntry		*entry;

    acquire_lock();

    // Hash index
    i = h % _name_table_size;
    entry = _name_table[ i ];

    while ( entry != NULL ) {
        if ( entry->_hash_value == h && entry->isEqual(s) )
            break;		// Stop looking if the hash value is the same
        // and the strings are equal.

        entry = entry->_next;
    }

    if ( entry == NULL ) { // No entries, make one.
        int len = strlen(s) + 1;

        if ( _chunk == NULL || _chunk->bytesLeft < len ) {
            // Make a new chunk if there wasn't enough
            // space in the last one.
            struct NameChunk	*newChunk = new NameChunk;

            newChunk->curByte   = newChunk->mem;
            newChunk->bytesLeft = CHUNK_SIZE;
            newChunk->next      = _chunk;

            _chunk = newChunk;
        }

        strcpy(_chunk->curByte, s);
        s = _chunk->curByte;

        _chunk->curByte   += len;
        _chunk->bytesLeft -= len;

        entry = new NameEntry( s, h, _name_table[ i ] );
        _name_table[ i ] = entry;
    }

    release_lock();

    return entry;
}

const char*	NameEntry::getString() const
{
    return _string;
}


///////////////////
// class Name

const char Name::_bad_characters[] = "+\'\"\\{}";
const NameEntry* Name::_empty_string = (NameEntry*) NULL;

Name::Name()
{
    _entry = _empty_string;
}

Name::Name( const char *s )
{
    _entry = NameEntry::insert( s );
}

Name::Name( const String &s )
{
    _entry = NameEntry::insert( s.getString() );
}

Name::Name( const Name &n )
{
    _entry = n._entry;
}

Name::~Name()
{
}

void Name::initClass()
{
    if (_empty_string != NULL)
        return;
                               
    NameEntry::initClass();
    _empty_string = NameEntry::insert("");
}

void Name::_cleanClass()
{
    NameEntry::_cleanClass();
}

const char* Name::getString() const
{
    return _entry->getString();
}

int Name::getLength() const
{
    return strlen( _entry->getString() );
}

int	Name::operator !() const
{
    return (_entry == NULL || _entry->isEmpty());
}

int operator ==( const Name &n, const char *s )
{
    return (n._entry != NULL && n._entry->isEqual( s ));
}

int operator ==( const char *s, const Name &n )
{
    return (n._entry != NULL && n._entry->isEqual( s ));
}

int operator ==( const Name &n1, const Name &n2 )
{
    return (n1._entry == n2._entry);
}

int operator !=( const Name &n, const char *s )
{
    return ! (n == s);
}

int operator !=( const char *s, const Name &n )
{
    return ! (n == s);
}

int operator !=( const Name &n1, const Name &n2 )
{
    return ! (n1 == n2);
}

bool Name::isIdentStartChar( char c )
{
    if ( isdigit(c) )
        return false;

    return isIdentChar(c);
}

bool Name::isIdentChar( char c )
{
    if ( isalnum(c) || c == '_' )
        return true;

    return false;
}

bool Name::isNodeNameStartChar(char c)
{
    if ( isdigit(c) )
        return false;

    return isIdentChar(c);
}

bool Name::isNodeNameChar(char c)
{
    if ( isalnum(c) )
        return true;

    if ( (strchr(_bad_characters, c) != NULL) || isspace(c) || iscntrl(c) )
        return false;

    return true;
}


//////////////////////////////////////////////
// class String

const char String::_null_string[] = "";

String::String()
{
    _string = (char*)_null_string;
    _storage_size = 0;
}

String::String( const char *str )
{
    _string = (char*)_null_string;
    _storage_size = 0;
    if (str)
      this->operator= ( str );
}

String::String( const char *str, int start, int end )
{
    _string = (char*)_null_string;
    _storage_size = 0;

    expand( end + 2 - start );
    strncpy( _string, str + start, end + 1 - start );
    _string[end+1-start]='\0';
}

String::String( const String &str )
{
    _string = (char*)_null_string;
    _storage_size = 0;

    this->operator= ( str );
}

String::String( int digitString )
{
    _string = (char*)_null_string;
    _storage_size = 0;

    char buffer[20];
    sprintf(buffer, "%d", digitString);

    this->operator= (buffer);
}

String::~String()
{
  if (_string == NULL) {
    fprintf(stderr, "Warning: Double deleting string %p\n",
            this);
  } else
    if ( _string != _null_string ) {
      delete [] _string;
      _string = NULL;
    }
}

unsigned long String::hash( const char *s )
{
    unsigned long	total, shift;

    total = shift = 0;

    while ( *s )
	{
            total = total ^ ((*s) << shift);
            shift += 5;

            if ( shift > 24 )
                shift -= 24;
            s++;
	}

    return( total );
}

int String::getLength() const
{
    return strlen( _string );
}

void String::makeEmpty( bool freeOld )
{
    if ( _string != _null_string )
	{
            if ( freeOld == true )
		{
                    delete [] _string;
                    _string = (char*)_null_string;
                    _storage_size = 0;
		}
            else
                _string[0] = '\0';
	}
}

const char* String::getString() const
{
    return _string;
}

String String::getSubString( int startChar, int endChar ) const
{
    if ( endChar == -1 )
        endChar = strlen( _string ) - 1;

    return String( _string, startChar, endChar );
}

void String::deleteSubString( int startChar, int endChar )
{
    int nlen = strlen( _string );

    if ( endChar == -1 ) {
        *(_string+startChar)='\0';
        return;
    }

    if (endChar < startChar)
        return;

    memmove( _string + startChar, _string + endChar + 1, nlen - endChar );
}

String& String::operator =( const char *str )
{
    if (!str || !*str) {
      if (_string == _null_string)
        return *this;
      str = "";
    }

    expand( strlen( str ) );
    strcpy( _string, str );
    return *this;
}

String& String::operator =( const String &str )
{
    return operator =( str._string );
}

String& String::operator +=( const char *str )
{
    expand( strlen(str) + strlen(_string) );
    strcat( _string, str );

    return *this;
}

String& String::operator +=( const String &str )
{
    return operator +=( str._string );
}

int String::operator !() const
{
    return (_string == _null_string || ! strlen( _string ));
}

int operator ==( const char *s, const String &str )
{
    return ! strcmp( s, str._string );
}

int operator ==( const String &str, const char *s )
{
    return operator ==( s, str );
}

int operator ==( const String &str1, const String &str2 )
{
    return operator ==( str1._string, str2 );
}

int operator !=( const char *s, const String &str )
{
    return ! operator ==( s, str );
}

int operator !=( const String &str, const char *s )
{
    return ! operator ==( s, str );
}

int operator !=( const String &str1, const String &str2 )
{
    return ! operator ==( str1._string, str2 );
}

void String::expand( int size )
{
    if ( size + 1 > _storage_size )
	{
            char *newstring = new char[ size + 1 ];

            strcpy( newstring, _string );

            if ( _string != _null_string )
                delete [] _string;

            _string = newstring;
            _storage_size = size + 1;
	}
}

char* String::copy(const char* str)
{
    if (!str)
        return (char*) NULL;

    int len = strlen(str) + 1;
    char* res = new char[len];
    memcpy(res, str, len);

    return res;
}
    
__UTILS_END_NAMESPACE

