///////////////////////////////////////////////////////////////////////////////
//
//                               Output.cc
//
// Implements a class that outputs to either a file or to a buffer.
// The class can open files by name, and ance an output destination is 
// established, the class provides methods for writing out strings and 
// numbers, as well as conveniently indenting the output file
//
// Classes defined for export:
//     Output - the output class
//
///////////////////////////////////////////////////////////////////////////////

#include <utils/Basic.h>

#include <string.h>
#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#include <utils/String.h>
#include <utils/Output.h>

#define TAB_WIDTH 4

__UTILS_BEGIN_NAMESPACE

Output::Output()
{
    reset();
    _fp = stdout;
}

Output::Output(const Output& output)
{
    _fp = output._fp;
    _to_buffer = output._to_buffer;
    _buffer = output._buffer;
    _cur_buf = output._cur_buf;
    _indent_level = output._indent_level;
    _compact = output._compact;
    _opened_here = output._opened_here;
    _resize_proc = output._resize_proc;
    _callback_data = output._callback_data;
}

Output::~Output()
{
    if (_fp && _opened_here)
        fclose(_fp);
}


void Output::reset()
{
    _fp = (FILE*) NULL;
    _to_buffer = false;
    _buffer = NULL;
    _cur_buf = (char*) NULL;
    _indent_level = 0;
    _compact = false;
    _opened_here = false;
    _resize_proc = 0L;
    _callback_data = NULL;
}

void Output::setFilePointer(FILE* newFP)
{
    reset();
    _fp = newFP;
}

FILE* Output::getFilePointer() const
{
    if (isToBuffer())
        return (FILE*) NULL;
    else
        return _fp;
}

bool Output::openFile(const char* file_name, bool append)
{
    reset();

    if (append) {
      _fp = fopen(file_name, "ab");
    } else {
      _fp = fopen(file_name, "wb");
    }
    if (_fp) {
        _opened_here = true;
        return true;
    } else
        return false;
}

void Output::closeFile()
{
    if (_fp && _opened_here)
        fclose(_fp);
    reset();
}

void Output::setBuffer(void* buffer, int size,
                         bool (*resize_proc)(void*&,int&,char*&,void*),
                         void* callback_data, int offset)
{
    reset();
    _buffer = buffer;
    _buf_size = size;
    _cur_buf = ((char*) buffer) + offset;
    _to_buffer = true;
    _resize_proc = resize_proc;
    _callback_data = callback_data;
}

bool Output::getBuffer(void*& buffer, int& size) const
{
    if (!isToBuffer())
        return false;

    buffer = _buffer;
    size = bytesInBuf();

    return true;
}

void Output::resetBuffer()
{
    _cur_buf = ((char*) _buffer);
}

void Output::write(char c)
{
    if (isToBuffer()) {
        if (_cur_buf - ((char*) _buffer) >= _buf_size &&
            !resizeBuffer())
            return;
            
        *_cur_buf++ = c;
    } else if (_fp)
        fputc(c, _fp);
}

void Output::write(const char* s)
{
    if (!s) 
        return;
    while (*s) { write(*s++); }
}

void Output::write(const String& s)
{
    write('"');
    write(s.getString());
    write('"');
}
    
void Output::write(const Name& n)
{
    write(n.getString());
}
    
void Output::write(int i)
{
    String str(i);
    write(str.getString());
}

void Output::write(unsigned int i)
{
    char output[10];
    sprintf(output, "%04x", i);
    write(output);
}

void Output::write(short s)
{
    String str(s);
    write(str.getString());
}

void Output::write(unsigned short s)
{
    char output[10];
    sprintf(output, "%02x", s);
    write(output);
}

void Output::write(float f)
{
    char output[30];
    sprintf(output, "%7.7f", f);
    write(output);
}

void Output::write(double d)
{
    char output[30];
    sprintf(output, "%12.12f", d);
    write(output);
}

void Output::write(unsigned char* data, int size)
{
    if (isToBuffer()) {
        while (_cur_buf + size - ((char*) _buffer) >= _buf_size) 
            if (!resizeBuffer())
                return;
        memcpy((char*) _cur_buf, (char*) data, size);
        _cur_buf += size;
    } else 
        fwrite(data, 1, size, _fp);
}

void Output::flush()
{
    if (_fp)
        fflush(_fp);
}

void Output::indent()
{
    if (!_compact)
        for (int i=0;i<_indent_level*TAB_WIDTH;i++) 
            write(' ');
}

bool Output::resizeBuffer()
{
    if (!_resize_proc)
        return false;
    
    return (*_resize_proc)(_buffer, _buf_size, _cur_buf, _callback_data);
}
    
bool Output::standardResize(void*& buffer, int& buf_size, char*& cur_buf,
                            void*)
{
    int index = cur_buf - (char*) buffer;
    int new_size = 2*buf_size+1;
    char* new_buf = new char[new_size];
    memcpy((char*) new_buf, (char*) buffer, buf_size);
    if (buffer)
        delete [] (unsigned char*) buffer;
    buffer = new_buf;
    buf_size = new_size;
    cur_buf = (char*) buffer + index;
    
    return true;
}

__UTILS_END_NAMESPACE
