///////////////////////////////////////////////////////////////////////////////
//
//                               Managed.cc
//
// Implements the Managed class.  This class provides a reference counting 
// system for memory management for all classes descended from it.  If the 
// environment variable TRACE_REFERENCES is set, Managed prints out copious 
// information about what is getting referenced and unreferenced
//
// Managed maintains a static list of deferred deletions for all managed
// pointers.  If you want it to be thread-safe you must define a MutexFactory
// which will be used to produce a mutex for managed access to this list.
// (See Mutex.cc)
//
// Classes implemented for export:
//     Managed
//
// Classes defined for internal use
//     ManagedInitializer - helper class used to initialize Managed
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <utils/Vector.h>
#include <utils/Managed.h>
#include <utils/Mutex.h>

__UTILS_BEGIN_NAMESPACE

bool Managed::traceReferences = false;
Vector<Managed*>* Managed::_deferred_deletes = NULL;
int Managed::_deferral_status = 0;

// semaphore for making name entries thread safe

static Mutex* _Deferred_Mutex = (Mutex*) -1;

// acquire a lock on the managed mutex semaphore, if necessary
static void acquire_lock() {
    if (_Deferred_Mutex) {
        if (_Deferred_Mutex == (Mutex*) -1) {
            if (Mutex_Factory)
                _Deferred_Mutex = (*Mutex_Factory)();
            else
                _Deferred_Mutex = NULL;
            if (_Deferred_Mutex) 
                _Deferred_Mutex->lock();
        } else
            _Deferred_Mutex->lock();
    }
}

// release a lock on the managed mutex semaphore, if necessary
static void release_lock() {
    if (_Deferred_Mutex && (_Deferred_Mutex != (Mutex*) -1))
        _Deferred_Mutex->unlock();
}

// Managed's initializion class
class ManagedInitializer { 
  public: 
    ManagedInitializer() { Managed::initClass(); } 
};

Managed::~Managed()
{
}

// increment the reference count
void Managed::ref() const
{
    if (_ref_count == STATIC_REF_COUNT)
        return;
    ((Managed*) this)->_ref_count++;
    if (traceReferences) {
        printf("%p (%s %s): Ref %d\n", this,
               getTypeName(), getName(), _ref_count);
    }
}

// Decrement the reference count.  If it goes to 0 or below, commit hari-kari
void Managed::unref() const
{
    if (_ref_count == STATIC_REF_COUNT)
        return;
    Managed* obj = (Managed*) this;

    obj->_ref_count--;
    if (_ref_count < 0) {
        printf("Warning: negative reference count %p (%s %s): Unref(nd) %d\n",
               this, getTypeName(), getName(), _ref_count);
        return;
    }
    if (traceReferences)
        printf("%p (%s %s): Unref %d\n",
               this, getTypeName(), getName(), _ref_count);
    if (obj->_ref_count <= 0) {
        if (_deferral_status > 0) {
            acquire_lock();
            _deferred_deletes->append(obj);
            release_lock();
            if (traceReferences)
                printf("\tDeferring Deletion (%d)\n", _deferral_status);
        } else {
            if (traceReferences)
                printf("\tDeleting\n");
            delete obj;
        }
    }
}

// decrement the reference count, but do not delete
void Managed::unrefNoDelete() const 
{
    if (_ref_count == STATIC_REF_COUNT)
        return;
    ((Managed*) this)->_ref_count--;
    if (_ref_count < 0) {
        printf("Warning: negative reference count %p (%s %s): Unref(nd) %d\n",
               this, getTypeName(), getName(), _ref_count); 
       return;
    }
    if (traceReferences)
        printf("%p (%s %s): Unref(nd) %d\n",
               this, getTypeName(), getName(), _ref_count);
}

// create an Managed
Managed::Managed()
{
    // the static initializer is a C++ trick to ensure that the initClass
    // routine for Managed is called before the first Managed is created
    static ManagedInitializer initializer;

    _ref_count = 0;
}

// initialize the Managed class
void Managed::initClass()
{
    if (_deferred_deletes)
        return;

    // check if we should debug
    if (getenv("TRACE_REFERENCES"))
        traceReferences = true;

    _deferral_status = 0;
    _deferred_deletes = new Vector<Managed*>(30);
}

// increment deferral status to indicate we should defer deletions
void Managed::deferDeletions()
{
    _deferral_status++;
}

// decrement deferral status, if it has gone to 0, flush the deferred deletions
void Managed::undeferDeletions()
{
    _deferral_status--;
               
    if (_deferral_status < 0)
        _deferral_status = 0;

    if (!_deferral_status)
        flushReferences();
}

// reset to no deferring deletions
void Managed::resetDeletions()
{
    _deferral_status = 0;
    if (traceReferences)
        printf("Resetting deletion deferral\n");
}

// flush deferred references
void Managed::flushReferences()
{
    static bool flushing = false;
    if (flushing)
        return;

    flushing = true;
    int i;
    Managed* elem;
    acquire_lock();
    Vector<Managed*> safe_list(_deferred_deletes->numElems());
    for (i=0;i<_deferred_deletes->numElems();i++) 
        safe_list.append((*_deferred_deletes)[i]);
    _deferred_deletes->clear();
    release_lock();

    for (i=0; i<safe_list.numElems();i++) {
        elem = safe_list[i];
        if (elem->_ref_count <= 0) {
            if (traceReferences) 
                printf("%p: Deferred deletion\n", elem);
            delete elem;
        } else if (traceReferences) 
            printf("%p: Deferred deletion canceled\n", elem);
    }
    flushing = false;
}

// makes this static, i.e., ref'ing and unref'ing have no effect
void Managed::makeStatic()
{
    _ref_count = STATIC_REF_COUNT;
}

__UTILS_END_NAMESPACE
