/***************************************************************************

                                Logger.cc

  Implements a class to output values tagged with times using the
  CannedDataWrite class

  Classes implemented for export:
    Logger - the logging class

***************************************************************************/

#include <utils/Logger.h>
#include <utils/ConfigFile.h>
#include <utils/Output.h>
#include <utils/CannedDataAccess.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/FormConfig.h>

__UTILS_BEGIN_NAMESPACE

class LogElem {
public:
  LogElem(const char* name, const char* format,
          FormatSpec* spec, void* address);
  ~LogElem();

  const char* getName() const { return _name.getString(); }
  const char* getFormat() const { return _format.getString(); }
  int getDataSize();
  int packData(unsigned char* dest, int dest_size);

protected:
  FormatSpec* _spec;
  void* _address;
  String _name;
  String _format;
  int _data_size;
};

LogElem::LogElem(const char* name, const char* format,
                 FormatSpec* spec, void* address)
{
  _data_size = 0;
  _name = name;
  _format = format;
  spec->ref();
  _spec = spec;
  _address = address;
}

LogElem::~LogElem()
{
  _spec->unref();
}

int LogElem::getDataSize()
{
  _data_size = BufferSizeAction::size(_spec, _address);
  return _data_size;
}

int LogElem::packData(unsigned char* dest, int dest_size)
{
  if (!PackAction::pack(_spec, dest, dest_size, _address))
    return -1;
  return _data_size;
}

Logger::Logger()
{
  FormatAction::initClasses();
  _parser = new FormatParser;
  _buffer = new unsigned char[200];
  _buffer_size = 200;
}

Logger::~Logger()
{
  LogElem* current;
  while ((current = _elems.pop()))
    delete current;
  delete [] _buffer;
  delete _parser;
}

void Logger::close()
{
  _writer.close();
}

bool Logger::declare(const char* name, const char* format, void* address)
{
  FormatSpec* spec = _parser->parseString(format);
  if (!spec) {
    fprintf(stderr, "Invalid format string '%s'\n", format);
    return false;
  }

  LogElem* elem = new LogElem(name, format, spec, address);
  _elems.append(elem);
  return true;
}

bool Logger::open(const char* base_name, ConfigFile& params)
{
  bool allow_clobber = params.getBool("allow_clobber", false);
  
  Output output;
  output.setBuffer(_buffer, 200, Output::standardResize);
  ListIterator<LogElem> iter(_elems);
  LogElem* current;
  output.indent();
  output.write("string names =");
  for (current = iter.first(); current; current = iter.next()) {
    output.write(" '");
    output.write(current->getName());
    output.write("'");
  }
  output.write(";\n");

  output.indent();
  output.write("string formats =");
  for (current = iter.first(); current; current = iter.next()) {
    output.write(" '");
    output.write(current->getFormat());
    output.write("'");
  }
  output.write(";\n");

  output.indent();
  output.write("int byte_order = ");
  output.write((int) BYTE_ORDER);
  output.write(";\n");

  output.indent();
  output.write("int alignment = ");
  output.write((int) ALIGN);
  output.write(";\n");
  output.write('\0');

  void* out_buf;
  int size;
  output.getBuffer(out_buf, size);
  _buffer = (unsigned char*) out_buf;

  utils::ConfigFile header;
  header.parse((char*) _buffer);
  params.mergeStruct("struct DataFormat", header);

  output.setBuffer(_buffer, _buffer_size, Output::standardResize);
  params.write(output);
  output.write('\0');  // make sure it is null terminated
  output.getBuffer(out_buf, size);
  _buffer = (unsigned char*) out_buf;

  try {
    if (!allow_clobber)
      _writer.open(base_name, O_EXCL);
    else
      _writer.open(base_name);

    _writer.writeType((CannedDataType)
                      params.getInt("DataFormat.type", CDT_UNKNOWN));
    _writer.writeVersion(params.getInt("DataFormat.version_major", 0),
                         params.getInt("DataFormat.version_minor", 0));
    _writer.writeDescription(params.getString("DataFormat.description",
                                              "DataFile"));
    _writer.writeHeader(out_buf, size);
    _writer.setThrottle(params.getInt("DataFormat.throttle_size", 0),
                        params.getInt("DataFormat.throttle_window", 0));
  
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    throw err;
    return false;
  }
  return true;
}

bool Logger::log(Time t)
{
  long secs, usecs;
  t.getValue(secs, usecs);

  int total_size = 0;
  ListIterator<LogElem> iter(_elems);
  for (LogElem* current = iter.first(); current; current = iter.next()) {
    total_size += current->getDataSize();
  }

  if (total_size > _buffer_size) {
    delete [] _buffer;
    _buffer = new unsigned char[total_size];
    _buffer_size = total_size;
  }

  int index = 0;
  for (LogElem* current = iter.first(); current; current = iter.next()) {
    int size = current->packData(_buffer + index, _buffer_size-index);
    if (size < 0)
      return false;
    index += size;
  }

  try {
    _writer.writeData(_buffer, total_size, secs, usecs);
  } catch (CannedDataError err) {
    String errMsg;
    errMsg = err.getMsg();
    fprintf(stderr, "Ack!: %s\n", errMsg.getString());
    return false;
  }
  return true;
}
  
__UTILS_END_NAMESPACE
