///////////////////////////////////////////////////////////////////////////////
//
//                                 List.cc
//
// This file implements the class to implement a simple linked list 
// abstract data type.  
//
// This code began its life as C code gotten from Chris Fedor for TCA, it
// mutated into C++ code and from there to the basis for a templated list.
// Its varied pedigree is what results in some of the redundancy in how
// you can iterate through a list, either through passing in a function
// to be called for each element or through using a ListIteratorBase
//
// One of its quirks is that it tries to do  efficient memory management of 
// list elements, which means it has a local static global variable 
// _cellFreeList which is used by all ListBase's to manage free list cells.
// If MutexFactory is defined, this code becomes threadsafe as access to
// _cellFreeList will be protected by mutex's.
//
// Classes defined for export:
//    ListBase - a linked list of void*
//    ListIteratorBase - iterator for a linked list of void*
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>

#include <utils/List.h>
#include <utils/Mutex.h>

__UTILS_BEGIN_NAMESPACE

// semaphore for making cell list manipulation thread safe
static Mutex* _Cell_Mutex = (Mutex*) -1;

// acquire a lock on the cell mutex semaphore, if necessary
static void acquire_lock() {
    if (_Cell_Mutex) {
        if (_Cell_Mutex == (Mutex*) -1) {
            if (Mutex_Factory)
                _Cell_Mutex = (*Mutex_Factory)();
            else
                _Cell_Mutex = NULL;
            if (_Cell_Mutex) 
                _Cell_Mutex->lock();
        } else
            _Cell_Mutex->lock();
    }
}

// release lock on the cell mutex semaphore, if necessary
static void release_lock() {
    if (_Cell_Mutex && (_Cell_Mutex != (Mutex*) -1))
        _Cell_Mutex->unlock();
}

/* a list element */
struct ListElem {
    ListElem(ListElem*);
    
    void *item;            // the data 
    ListElem* next;      // the next element
    ListElem* previous;  // the previous element
};

/* list of free cells */
static ListElem* _cellFreeList = NULL;

/* how much to increase the list of free cells by when we run out */
#define LIST_CELL_INC_AMOUNT 10

// make the free cell list LIST_CELL_INC_AMOUNT bigger
static void listIncCellFreeList()
{
    int i;
    ListElem* newCell;

    for(i=1;i<LIST_CELL_INC_AMOUNT;i++) {
        newCell = new ListElem(_cellFreeList);

        _cellFreeList = newCell;
    }
}

// get the next free cell
inline ListElem* listGetFreeCell()
{
    acquire_lock();

    if (!_cellFreeList)
        listIncCellFreeList();
    ListElem* elem = _cellFreeList;
    _cellFreeList = _cellFreeList->next;

    release_lock();

    return elem;
}

// mark listCell as free on the free list
static void listFreeCell(ListElem* listCell)
{
    listCell->item = NULL;
    listCell->previous = NULL;

    if (!listCell)
      return;

    acquire_lock();

    listCell->next = _cellFreeList;
    _cellFreeList = listCell;

    release_lock();
}

/* create a list element, and put it on the list of free cells "freecell" */
ListElem::ListElem(ListElem* freecell)
{
    item = NULL;
    previous = NULL;
    next = freecell;
}

// common constructor stuff
void ListBase::initialize()
{
    _length = 0;
    _first = NULL;
    _last = NULL;
}

/* Create a new list and but it on list of free list "freelist" */
ListBase::ListBase()
{
    initialize();
}

// create a list with one item
ListBase::ListBase(void* item)
{
    initialize();

    prepend(item);
}

// create a list with two items
ListBase::ListBase(void* item1, void* item2)
{
    initialize();

    prepend(item2);
    prepend(item1);
}

// delete a list
ListBase::~ListBase()
{
    ListElem* tmpA;
    ListElem* tmpB;

    tmpA = _first;

    while (tmpA) {
        tmpB = tmpA;
        tmpA = tmpA->next;

        listFreeCell(tmpB);
    }
}

/**************************************************************************
 *
 * FUNCTION: prepend(item)
 *
 * DESCRIPTION: Adds item as the first item in the list.
 *
 * INPUTS: 
 * void *item - a pointer to an item of data.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a list element to store item and insert the element as the first item
 * in the list.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void ListBase::prepend(void* item)
{
    ListElem* element;

    element = listGetFreeCell();
    element->item = item;
    element->next = _first;
    element->previous = NULL;

    if (!_first) {
        _first = element;
    }
    else {
        _first->previous = element;
    }

    if (!_last) {
        _last = element;
    }

    _length++;
    _first = element;
}


/**************************************************************************
 *
 * FUNCTION: void append(item)
 *
 * DESCRIPTION: Adds item as the last item in the list.
 *
 * INPUTS: 
 * void *item - a pointer to an item of data.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 *
 * DESIGN: 
 * Create a list element to store item and insert the element as the first item
 * in the list.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void ListBase::append(void* item)
{
    ListElem* element;

    if (!item || !this)
        return;

    element = listGetFreeCell();
    element->item = item;
    element->next = NULL;
    element->previous = _last;

    if (!_first) {
        _first = element;
    }

    if (!_last) {
        _last = element;
    }
    else {
        _last->next = element;
    }

    _length++;
    _last = element;
}



/**************************************************************************
 *
 * FUNCTION: void insertAfter(item, after)
 *
 * DESCRIPTION: Splices item into the list after, after.
 *
 * INPUTS: 
 * void *item - a pointer to an item of data.
 * void *after - the item to be inserted after.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  
 * If the list is NULL, return.
 * If the item is NULL, return.
 * If after is NULL call listInsertItemFirst.
 * If after is not found call listInsertItemLast.
 *
 * NOTES: If malloc returns NULL simply return <- this is a major problem.
 *
 **************************************************************************/

void ListBase::insertAfter(void* item, void* after)
{
    ListElem* element;
    ListElem* tmp;

    if (!item || !this)
        return;

    if (!after) {
        prepend(item);
        return;
    }

    tmp = _first;

    while (tmp && tmp->item != after) 
        tmp = tmp->next;

    if (!tmp) {
        append(item);
        return;
    }

    element = listGetFreeCell();
    element->item = item;
    element->next = tmp->next;
    element->previous = tmp;
    if (tmp->next)
        tmp->next->previous = element;

    tmp->next = element;

    _length++;
}


/**************************************************************************
 *
 * FUNCTION: void *pop()
 *
 * DESCRIPTION: Removes and returns first item from list.
 *
 * OUTPUTS:
 * The first item or NULL
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return NULL.
 *
 * DESIGN: 
 * Remove the first list element, return the item and free the element.
 *
 * NOTES:
 * might want to start a free list of list elements.
 *
 **************************************************************************/

void *ListBase::pop()
{
    void *item;
    ListElem* oldElement;

    item = NULL;
    
    if (this && _first) {
        item = _first->item;
        oldElement = _first;
        _first = _first->next;
        if (_first) {
            _first->previous = NULL;
        }
        if (_last == oldElement) {
            _last = NULL;
        }
        _length--;
        listFreeCell(oldElement);
    }

    return item;
}


/* remove the list element that matches "param" according to the match function
   "func".  Return the data associated with this list element, or NULL if there
   was no such matching element */
void* ListBase::remove(int (*func)(void*, void*), void *param)
{
    ListElem* current;
    ListElem* previous;

    if (!this || !_first)
        return NULL;

    current = previous = _first;

    if ((*func)(param, current->item)) {
        /* item is the first element of the list */
        if (_last == current) {
            _last = NULL;
        }
        _first = current->next;
        if (current->next) {
            current->next->previous = NULL;
        }
        _length--;
        void* res = current->item;
        listFreeCell(current);
        return res;
    }

    current = current->next;

    while (current) {
        if ((*func)(param, current->item)) {
            if (_last == current) {
                _last = previous;
            }
            current->previous = previous;
            previous->next = current->next;

            if (current->next) {
                current->next->previous = previous;
            }

            _length--;
            void* res = current->item;
            listFreeCell(current);
            return res;
        }
        previous = current;
        current = current->next;
    }

    return NULL;
}

/**************************************************************************
 *
 * FUNCTION: void removeAll(func, param)
 *
 * DESCRIPTION: 
 * removes all items in the list found such that func(param, item)
 * returns 1 (TRUE).
 *
 * INPUTS: 
 * int (*func)() - pointer to a test function of the form func(param, item).
 * void *param - a pointer to a parameter for func.
 *
 * OUTPUTS: none.
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return;
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item,
 * such that func(param, item) returns 1 (TRUE). If found
 * the list element is removed and freed. All the items of the list
 * are tested. Reset element's previous pointer and _last if needed.
 *
 *
 **************************************************************************/

void ListBase::removeAll(int (*func)(void*, void*), void* param)
{
    ListElem* current;
    ListElem* previous;

    if (!this || !_first)
        return;

    int res;

    while (_first && (res=(*func)(param, _first->item))) {
        if (res == -1)
            return;
        _length--;
        current = _first;
        _first = current->next;

        if (_first) {
            _first->previous = NULL;
        }
        if (_last == current) {
            _last = NULL;
        }
        listFreeCell(current);
    }

    if (!_first)
        return;

    previous = _first;
    current  = _first->next;

    while (current) {
        if ((res = (*func)(param, current->item))) {
            if (res == -1)
                return;
            if (_last == current) {
                _last = previous;
            }

            previous->next = current->next;

            if (current->next) {
                current->next->previous = previous;
            }

            _length--;
            listFreeCell(current);
            
            current = previous->next;
        }
        else {
            previous = current;
            current = current->next;
        }
    }
}


/**************************************************************************
 *
 * FUNCTION: bool listItemEq(a, b)
 *
 * DESCRIPTION: Simple Equal Test for remove.
 *
 * INPUTS:
 * void *a, *b;
 *
 * OUTPUTS: Returns 1 TRUE or 0 FALSE.
 *
 * DESIGN: return(a == b);
 *
 * NOTES:
 *
 **************************************************************************/

int listItemEq(void* a, void* b)
{
    return(a == b);
}


/**************************************************************************
 *
 * FUNCTION: remove(item)
 *
 * DESCRIPTION: removes an item from list.
 *
 * INPUTS:
 * void *item; 
 *
 * OUTPUTS: none.
 *
 * DESIGN: call remove with listItemEq test.
 *
 * NOTES: list is modified.
 *
 **************************************************************************/

bool ListBase::remove(void* item)
{
    return remove((int (*)(void*, void*)) listItemEq, item) != NULL;
}


/**************************************************************************
 *
 * FUNCTION: removeAll(item)
 *
 * DESCRIPTION: removes an all such item from list.
 *
 * INPUTS:
 * void *item; 
 *
 * OUTPUTS: none.
 *
 * DESIGN: call removeAll with listItemEq test.
 *
 * NOTES: list is modified.
 *
 **************************************************************************/

void ListBase::removeAll(void* item)
{
    removeAll(listItemEq, item);
}

// helper equality test function used in removeAll
static int listTrue(void*, void*)
{
    return 1;
}

// remove all elements from the list
void ListBase::clear()
{
    removeAll(listTrue, NULL);
}

/**************************************************************************
 *
 * FUNCTION: bool member(item)
 *
 * DESCRIPTION: Tests if item is an element of list.
 *
 * INPUTS: 
 * void *item - a pointer to an item of data.
 *
 * OUTPUTS: 
 * 0 - FALSE 
 * 1 - TRUE
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, return;
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item. 
 * If found the value 1 is returned, else the value 0 is returned.
 *
 * NOTES:
 *
 **************************************************************************/

bool ListBase::member(void* item)
{
    ListElem* tmp;

    if (!this)
        return false; /* False */

    tmp = _first;

    while (tmp) {
        if (tmp->item == item)
            return true; /* TRUE */
        tmp = tmp->next;
    }

    return false; /* False */
}


/**************************************************************************
 *
 * FUNCTION: void *find(func, param)
 *
 * DESCRIPTION:
 * find is a more general form of member.
 * find will return the item (or one of the items) in list
 * for which func(param, item) is non-zero, i.e. is TRUE.
 * The function takes two arguments, the first is the param and the second is 
 * an item of the list and returns an integer value. bool func(param, item).
 * If the functions does not satisfy any of the items in the list NULL
 * is returned.
 *
 * INPUTS: 
 * int (*func)();
 * void *param - a pointer to a parameter that is passed to func.
 *
 * OUTPUTS: 
 * A pointer to an item in the list that satisfies func(param, item) 
 * or NULL if no such item exists. 
 *
 * EXCEPTIONS:  If the list is NULL or the list is empty, NULL is returned.
 *
 * DESIGN: 
 * Linearly search the list for a list element containing item, such that
 * func(param, item) is non-zero. Then return the item.
 *
 * NOTES:
 *
 **************************************************************************/

void *ListBase::find(int (*func)(void*, void*), void *param)
{
    ListElem* tmp;
    ListElem* nextTmp;

    if (!this)
        return NULL;
    else {
        tmp = _first;
        while (tmp) {
            nextTmp = tmp->next;
            if ((*func)(param, tmp->item))
                return tmp->item;
            else
                tmp = nextTmp;
        }
        return NULL;
    }
}


/**************************************************************************
 *
 * FUNCTION: bool iterate(func, param)
 *
 * DESCRIPTION:
 * iterate will call the function func with param on all of its
 * elements stoping when the list is finished or when func returns 0 (ie
 * FALSE). The function func will take two arguments the first is 
 * param the second is an item of the set. func(param, item).
 *
 * iterate starts from the first item in the list and iterates
 * forward through the items in the list.
 *
 * INPUTS: 
 * int (*func)();
 * void *param - a pointer to a parameter that is passed to func.
 *
 * OUTPUTS: 
 * iterate returns 0 (FALSE) if the function func returns 0 (FALSE).
 * Otherwise iterate returns 1 (TRUE).
 *
 * EXCEPTIONS: 
 *
 * iterate will return false (FASLE) if the list is NULL.
 *
 * DESIGN: 
 * iterate through the list of elements calling func on each item.
 * return when the list is finished or func has returned 0 (FALSE).
 *
 **************************************************************************/

bool ListBase::iterate(int (*func)(void*, void*), void *param)
{
    void *item;
    ListElem* tmp;
    ListElem* nextTmp;

    if (!this)
        return false;

    tmp = _first;

    while (tmp) {
        item = tmp->item;
        nextTmp = tmp->next;
        if (item && !(*func)(param, item))
            return false;
        tmp = nextTmp;
    }

    return true;
}


/**************************************************************************
 *
 * FUNCTION: bool iterateFromLast(func, param)
 *
 * DESCRIPTION:
 * iterateFromLast will call the function func with param on all of its
 * elements stoping when the list is finished or when func returns 0 (ie
 * FALSE). The function func will take two arguments the first is 
 * param the second is an item of the set. func(param, item).
 *
 * iterateFromLast starts with thelast item in the list and iterates
 * backwards through the list.
 *
 * INPUTS: 
 * int (*func)();
 * void *param - a pointer to a parameter that is passed to func.
 *
 * OUTPUTS: 
 * iterateFromLast returns 0 (FALSE) if the function func returns 0 (FALSE).
 * Otherwise iterateFromLast returns 1 (TRUE).
 *
 * EXCEPTIONS: 
 *
 * iterateFromLast will return false (FASLE) if the list is NULL.
 *
 * DESIGN: 
 * iterate through the list of elements calling func on each item.
 * return when the list is finished or func has returned 0 (FALSE).
 *
 **************************************************************************/

bool ListBase::iterateFromLast(int (*func)(void*, void*), void *param)
{
    void *item;
    ListElem* tmp;
    ListElem* previousTmp;

    if (!this)
        return false;

    tmp = _last;

    while (tmp) {
        item = tmp->item;
        previousTmp = tmp->previous;
        if (item && !(*func)(param, item))
            return false;
        tmp = previousTmp;
    }

    return true;
}


/* insert an item into list.  Used in Copy iteration */
static int listCopyInsert(void* param, void* item)
{
    ListBase* list = (ListBase*) param;
    list->prepend(item);

    return true;
}

/**************************************************************************
 *
 * FUNCTION: ListBase* Copy()
 *
 * DESCRIPTION: Copies the given list.
 *
 * OUTPUTS: A pointer to the newly created list of type ListBase*.
 * If there is an error NULL is returned.
 *
 * EXCEPTIONS: none.
 *
 * DESIGN: iterate through the original list inserting the items into the
 * new list.
 *
 * NOTES:
 *
 **************************************************************************/

ListBase* ListBase::copy()
{
    ListBase* newList;

    newList = new ListBase();

    iterateFromLast(listCopyInsert, newList);

    return newList;
}

/* find true if list2 eq's this list */
bool ListBase::equal(ListBase* list2)
{
    bool good;
    ListElem* a;
    ListElem* b;

    if (this == list2)
        return true;

    /* this is the same style test used in tms.c but it is not general ! */

    a = _first;
    b = list2->_first;

    good = 1;
    while (good && a && b) {
        good = (a->item == b->item);
        a = a->next;
        b = b->next;
    }

    return(good && (a == NULL) && (b == NULL));
}

/* insert "item" if it is not already a member of the list */
void ListBase::insertUnique(void* item)
{
    if (!member(item)) {
        prepend(item);
    }
}

void* ListBase::first() const
{
  if (!_first)
    return NULL;
  else
    return _first->item;
}

void* ListBase::last() const
{
  if (!_last)
    return NULL;
  else
    return _last->item;
}

void ListBase::clearCellList()
{
    ListElem* tmpA;

    acquire_lock();

    while(_cellFreeList != NULL) {
        tmpA = _cellFreeList;
        _cellFreeList = _cellFreeList->next;
        delete tmpA;
    }

    release_lock();
}


///////////////////////////////////////////////////////////////////////////
// ListIteratorBase - the list iterator

ListIteratorBase::ListIteratorBase(ListBase& list) : _list(list)
{
  _next = NULL;
}

/* find the first item of the list.  Also set up _next so that next() will
   work in a simple for (i=first(); i; i=next()) loop through the list */
void *ListIteratorBase::first()
{
    if (this && _list._first) {
        _next = _list._first;
        return(_list._first->item);
    }
    else
        return NULL;
}

/* Get the next item in the list.  first() must have been called first */
void *ListIteratorBase::next()
{
    if (this && _next) {
      if (_next == (ListElem*) -1) {
        if (!_list._first) {
          _next = NULL;
          return NULL;
        }
        _next = _list._first;
        return _list._first->item;
      } else {
        _next = _next->next;

        if (_next)
            return(_next->item);
      }
    }

    return NULL;
}

// remove the current element in the list
void ListIteratorBase::removeCurrent()
{
  if (!_next)
    return;

  ListElem* current;
  if (_next == (ListElem*) -1) {
    current = _list._first;
    if (_list._first) {
      _list._first = _list._first->next;
      _list._first->next->previous = NULL;
    }
  } else {
    current = _next;
    _next = _next->previous;
    if (!_next)
      _next = (ListElem*) -1;

    /* item is the first element of the list */
    if (current == _list._first) {
      if (_list._last == current) {
        _list._last = NULL;
      }
      _list._first = current->next;
      if (current->next) {
        current->next->previous = NULL;
      }
    } else {
      if (_list._last == current) {
        _list._last = current->previous;
      }
      current->previous->next = current->next;

      if (current->next) {
        current->next->previous = current->previous;
      }
    }
  }

  _list._length--;
  listFreeCell(current);
}

// return current element in the list
ListElem* ListIteratorBase::getCurrent() const { 
  return _next;
}

// set the current element in the list
void ListIteratorBase::setCurrent(ListElem* elem)
{
  _next = elem;
}

// insert after the current element in the list
void ListIteratorBase::insertAfterCurrent(void* item)
{
  if (!_next) {
    _list.append(item);
    _next = _list._last;
    return;
  }
  if (_next == (ListElem*) -1) {
    _list.prepend(item);
    return;
  }

  ListElem* current = _next;
  if (!current->next) {
    _list.append(item);
    return;
  }

  ListElem* element = listGetFreeCell();
  element->item = item;
  element->next = current->next;
  element->previous = current;
  current->next->previous = element;
  current->next = element;
    
  _list._length++;
}

// insert before the current element in the list
void ListIteratorBase::insertBeforeCurrent(void* item)
{
  if (!_next || _next == (ListElem*) -1) {
    _list.prepend(item);
    _next = _list._first;
    return;
  }

  ListElem* current = _next;
  if (!current->previous) {
    _list.prepend(item);
    return;
  }

  ListElem* element = listGetFreeCell();
  element->item = item;
  element->next = current;
  element->previous = current->previous;
  current->previous->next = element;
  current->previous = element;

  _list._length++;
}
 


__UTILS_END_NAMESPACE
