///////////////////////////////////////////////////////////////////////////////
//
//                               Linear.cc
//
// Implements classes for creating and using simple linear entities
//
// Classes implemented for export:
//     Rotation - a 3D rotation class
//     Line - a 3D line class
//     Transform - a 6D homogenous transform matrix class
//
///////////////////////////////////////////////////////////////////////////////

#include <math.h>
#include <string.h>

#include <utils/Linear.h>
#include <utils/ConfigFile.h>

__UTILS_BEGIN_NAMESPACE

// constants for quaternion elements of Rotation
#define X 0
#define Y 1
#define Z 2
#define W 3
#define	Q_EPSILON   (1e-10)

///////////////////////////////////////////////////////////////////////////////
// class Rotation
///////////////////////////////////////////////////////////////////////////////

Rotation::Rotation()
{
  // equivalent, and more direct. -- ram
  quaternion[0] = quaternion[1] = quaternion[2] = 0;
  quaternion[3] = 1;
}

Rotation::Rotation( const double v[4] )
{
  setValue( v );
}

Rotation::Rotation( double q0, double q1, double q2, double q3 )
{
  setValue( q0, q1, q2, q3 );
}

Rotation::Rotation( const Transform &m )
{
  setValue( m );
}

Rotation::Rotation( const Vec3d &axis, double radians )
{
  setValue( axis, radians );
}

Rotation::Rotation( const Vec3d &rotateFrom, const Vec3d &rotateTo )
{
  setValue( rotateFrom, rotateTo );
}


Rotation::Rotation (double roll, double pitch, double yaw)
{
  Vec3d tmp;
  tmp[0] = 0; tmp[1] = 0; tmp[2] = 1;
  Rotation rot_yaw(tmp, yaw);
  tmp[0] = 0; tmp[1] = 1; tmp[2] = 0;
  Rotation rot_pitch(tmp, pitch);
  tmp[0] = 1; tmp[1] = 0; tmp[2] = 0;
  Rotation rot_roll(tmp, roll);
  Rotation rot = (rot_yaw * rot_pitch * rot_roll);
  setValue(rot.quaternion);
}


const double* Rotation::getValue() const
{
  return &quaternion[0];
}

void Rotation::getValue( double &q0, double &q1, double &q2, double &q3 ) const
{
  q0 = quaternion[0];    q1 = quaternion[1];
  q2 = quaternion[2];    q3 = quaternion[3];
}

void Rotation::normalize()
{
  double norm = 1/sqrt(quaternion[0]*quaternion[0] + 
		      quaternion[1]*quaternion[1] + 
		      quaternion[2]*quaternion[2] + 
		      quaternion[3]*quaternion[3]);
  quaternion[0] *= norm;
  quaternion[1] *= norm;
  quaternion[2] *= norm;
  quaternion[3] *= norm;
}

Rotation& Rotation::setValue( double q0, double q1, double q2, double q3 )
{
  quaternion[0] = q0;  quaternion[1] = q1;
  quaternion[2] = q2;  quaternion[3] = q3;

  normalize();
    
  return *this;
}

void Rotation::getValue( Vec3d & axis, double & radians ) const
{
  radians = double ( acos( quaternion[3] ) * 2 );
  if ( radians == 0.f ) {
    axis[0] = 0;
    axis[1] = 0;
    axis[2] = 1;
  } else {
    axis.setValue( quaternion );
    if( axis[0] != 0.f || axis[1] != 0.f || axis[2] != 0.f )
      axis.normalize();
  }
  /*
    SbLinearVector<double> sbaxis;
    double sbradians;
    convertRotToSbRot(*this).getValue(sbaxis, sbradians);
    if (!sbaxis.equals(convertVec3dToSbVec3d(axis), 0.00001) ||
    fabs(radians-sbradians) > 0.00001) {
    printf("getValue inequality: (%f %f %f | %f) (%f %f %f | %f)\n",
    axis[0], axis[1], axis[2], radians,
    sbaxis[0], sbaxis[1], sbaxis[2], sbradians);
    }
  */
}

void Rotation::getValue( Transform & matrix ) const
{
  double s, xs, ys, zs, wx, wy, wz, xx, xy, xz, yy, yz, zz;
   
  s = 2.0 / ( quaternion[0] * quaternion[0]
              + quaternion[1] * quaternion[1]
              + quaternion[2] * quaternion[2]
              + quaternion[3] * quaternion[3] );
   
  xs = quaternion[0] * s;
  ys = quaternion[1] * s;
  zs = quaternion[2] * s;
   
  wx = quaternion[3] * xs;
  wy = quaternion[3] * ys;
  wz = quaternion[3] * zs;
   
  xx = quaternion[0] * xs;
  xy = quaternion[0] * ys;
  xz = quaternion[0] * zs;
   
  yy = quaternion[1] * ys;
  yz = quaternion[1] * zs;
   
  zz = quaternion[2] * zs;
   
  matrix[0][0] = double ( 1.0 - ( yy + zz ) );
  matrix[1][0] = double ( xy + wz );
  matrix[2][0] = double ( xz - wy );
   
  matrix[0][1] = double ( xy - wz );
  matrix[1][1] = double ( 1.0 - ( xx + zz ));
  matrix[2][1] = double ( yz + wx );
   
  matrix[0][2] = double ( xz + wy );
  matrix[1][2] = double ( yz - wx );
  matrix[2][2] = double ( 1.0 - ( xx + yy ));
   
  matrix[3][0] = matrix[3][1] = matrix[3][2] =
    matrix[0][3] = matrix[1][3] = matrix[2][3] = 0.0f;
  matrix[3][3] = 1.0f;
}

Rotation& Rotation::setValue( const double q[4] )
{
  quaternion[0] = q[0];
  quaternion[1] = q[1];
  quaternion[2] = q[2];
  quaternion[3] = q[3];
  normalize();
  return *this;
}

Rotation& Rotation::setValue( const Transform & m )
{
  double tr, s;
  int i, j, k;
  const int nxt[3] = { 1, 2, 0 };
   
  tr = m[0][0] + m[1][1] + m[2][2];
   
  if ( tr > 0.0 ) {
    s = sqrt( tr + 1.0 );
    quaternion[3] = double ( s * 0.5 );
    s = 0.5 / s;
         
    quaternion[0] = double ( ( m[2][1] - m[1][2] ) * s );
    quaternion[1] = double ( ( m[0][2] - m[2][0] ) * s );
    quaternion[2] = double ( ( m[1][0] - m[0][1] ) * s );
  } else {
    i = 0;
    if ( m[1][1] > m[0][0] )
      i = 1;
         
    if ( m[2][2] > m[i][i] )
      i = 2;
         
    j = nxt[i];
    k = nxt[j];

    s = sqrt( ( m[i][i] - ( m[j][j] + m[k][k] )) + 1.0 );
         
    quaternion[i] = double ( s * 0.5 );
    s = 0.5 / s;
         
    quaternion[3] = double ( ( m[k][j] - m[j][k] ) * s );
    quaternion[j] = double ( ( m[j][i] + m[i][j] ) * s );
    quaternion[k] = double ( ( m[k][i] + m[i][k] ) * s );
  }
   
  normalize();
  return *this;
}

Rotation& Rotation::setValue( const Vec3d & axis, double radians )
{
  Vec3d a = axis;
  a.normalize();

  double radsin = sin( radians / 2 );
  quaternion[3] = double ( cos( radians / 2 ) );
  quaternion[0] = double ( radsin * a[0] );
  quaternion[1] = double ( radsin * a[1] );
  quaternion[2] = double ( radsin * a[2] );
  normalize();

  return *this;
}

Rotation& Rotation::setValue(const Vec3d& rotateFrom, 
                             const Vec3d& rotateTo)
{
  /*
  ** Normalize both vectors and take cross product to get rotation axis. 
    */
  Vec3d u1 = rotateFrom;
  u1.normalize();
  Vec3d u2 = rotateTo;
  u2.normalize();
  Vec3d axis = u1.cross(u2); /* axis of rotation */

  /*
    ** | u1 X u2 | = |u1||u2|sin(theta)
    **
    ** Since u1 and u2 are normalized, 
    **
    **  theta = arcsin(|axis|)
    */
  double crossProductMagnitude = sqrt( axis.dot(axis) );

  /*
    ** Occasionally, even though the vectors are normalized, the magnitude will
    ** be calculated to be slightly greater than one.  If this happens, just
    ** set it to 1 or asin() will barf.
    */
  if( crossProductMagnitude > 1.0 )
    crossProductMagnitude = 1.0 ;

    /*
    ** Take arcsin of magnitude of rotation axis to compute rotation angle.
    ** Since crossProductMagnitude=[0,1], we will have theta=[0,pi/2].
    */
  double theta = asin( crossProductMagnitude ) ;
  double theta_complement = M_PI - theta ;

    /*
    ** If cos(theta) < 0, use complement of theta as rotation angle.
    */
  if( u1.dot(u2) < 0.0 ) {
    theta = theta_complement ;
    theta_complement = M_PI - theta ;
  }

  /* if angle is 0, just return identity quaternion   */
  if( theta < Q_EPSILON ) {
    quaternion[X] = 0.0 ;
    quaternion[Y] = 0.0 ;
    quaternion[Z] = 0.0 ;
    quaternion[W] = 1.0 ;
  } else {
    if( theta_complement < Q_EPSILON ) {
      /*
      ** The two vectors are opposed.  Find some arbitrary axis vector.
      ** First try cross prod. with x-axis if u1 not parallel to x-axis.
      */
      if( (u1[Y]*u1[Y] + u1[Z]*u1[Z]) >= Q_EPSILON ) {
        axis[X] = 0.0 ;
        axis[Y] = u1[Z] ;
        axis[Z] = -u1[Y] ;
      } else {
        /*
                ** u1 is parallel to to x-axis.  Use z-axis as axis of rot.
                */
        axis[X] = axis[Y] = 0.0 ;
        axis[Z] = 1.0 ;
      }
    }

    this->setValue(axis, theta);
  }

  normalize();
  return *this;
}

Rotation& Rotation::operator*=(const Rotation& q)
{
  if ( this != &q ) {
    Rotation qq = *this * q;
    setValue(qq.quaternion);
  }

  return *this;
}

int operator ==( const Rotation &q1, const Rotation &q2 )
{
  return (q1.quaternion[0] == q2.quaternion[0]
          && q1.quaternion[1] == q2.quaternion[1]
          && q1.quaternion[2] == q2.quaternion[2]
          && q1.quaternion[3] == q2.quaternion[3] );
}

int operator !=( const Rotation &q1, const Rotation &q2 )
{
  return ! ( q1 == q2 );
}

Rotation operator*(const Rotation & q1, const Rotation & q2)
{
  return Rotation(q1.quaternion[3]*q2.quaternion[0] +
                  q1.quaternion[0]*q2.quaternion[3] + 
                  q1.quaternion[1]*q2.quaternion[2] -
                  q1.quaternion[2]*q2.quaternion[1],
                  q1.quaternion[3]*q2.quaternion[1] +
                  q1.quaternion[1]*q2.quaternion[3] + 
                  q1.quaternion[2]*q2.quaternion[0] -
                  q1.quaternion[0]*q2.quaternion[2],
                  q1.quaternion[3]*q2.quaternion[2] +
                  q1.quaternion[2]*q2.quaternion[3] + 
                  q1.quaternion[0]*q2.quaternion[1] -
                  q1.quaternion[1]*q2.quaternion[0],
                  q1.quaternion[3]*q2.quaternion[3] -
                  q1.quaternion[0]*q2.quaternion[0] - 
                  q1.quaternion[1]*q2.quaternion[1] -
                  q1.quaternion[2]*q2.quaternion[2]);
}

// The opposite rotation about the opposite axis is the same rotation.
bool Rotation::equals( const Rotation & r, double tolerance ) const
{
  tolerance = fabs(tolerance);
  return ( (fabs(quaternion[0] - r.quaternion[0]) < tolerance &&
	    fabs(quaternion[1] - r.quaternion[1]) < tolerance &&
	    fabs(quaternion[2] - r.quaternion[2]) < tolerance &&
	    fabs(quaternion[3] - r.quaternion[3]) < tolerance)
	   ||
	   (fabs(-quaternion[0] - r.quaternion[0]) < tolerance &&
	    fabs(-quaternion[1] - r.quaternion[1]) < tolerance &&
	    fabs(-quaternion[2] - r.quaternion[2]) < tolerance &&
	    fabs(-quaternion[3] - r.quaternion[3]) < tolerance)
	   );
}

#if 1
// All you really need to do to invert is swap the sign of components
// 0..2.  -- ram 4/03
Rotation& Rotation::invert()
{
  for (int i = 0; i < 3; i++)
    quaternion[i] = -quaternion[i];
  return *this;
}

Rotation Rotation::inverse() const
{
  Rotation result = *this;
  return result.invert();
}
#else
Rotation& Rotation::invert()
{
  Vec3d axis;
  double   radians;
  getValue(axis,radians);
  setValue(axis,-radians);
  return *this;
}

Rotation Rotation::inverse() const
{
  Vec3d axis;
  double   radians;
  getValue(axis,radians);
  return Rotation(axis,-radians);
}
#endif

// ### this is a pretty goofy way to rotate using a quaternion.  It
// only makes sense to convert to a transform matrix when there are
// many vectors to be transformed.  rotation is more efficient by
// doing q * vec * q.inverse.  -- ram 4/03
void Rotation::multVec(const Vec3d& src, Vec3d & dst) const
{
  Transform mat;
  getValue(mat);
  mat.multMatrixVec(src,dst);
}

// Spherical linear interpolation of unit quaternions.
// As t goes from 0 to 1, destQuat goes from startQ to endQuat.
// This routine should always return a point along the shorter
// of the two paths between the two.  That is why the vector may be
// negated in the end.   	
Rotation Rotation::slerp(const Rotation& startQuat,
                         const Rotation& endQuat, double t)
{
  Rotation destQuat;
  Rotation  startQ = startQuat;    	/* temp copy of startQuat	*/
  double  omega, cosOmega, sinOmega;
  double  startScale, endScale;
  int     i;

  cosOmega = startQ[X]*endQuat[X] + startQ[Y]*endQuat[Y] + 
    startQ[Z]*endQuat[Z] + startQ[W]*endQuat[W];

  /* If the above dot product is negative, it would be better to
   *  go between the negative of the initial and the final, so that
   *  we take the shorter path.  
   */
  if ( cosOmega < 0.0 ) {
    cosOmega *= -1;
    for (i = X; i <= W; i++)
      startQ[i] *= -1;
  }

  if ( (1.0 + cosOmega) > Q_EPSILON ) {
    /* usual case */
    if ( (1.0 - cosOmega) > Q_EPSILON ) {
      /* usual case */
      omega = acos(cosOmega);
      sinOmega = sin(omega);
      startScale = sin((1.0 - t)*omega) / sinOmega;
      endScale = sin(t*omega) / sinOmega;
    } else {
      /* ends very close */
      startScale = 1.0 - t;
      endScale = t;
    }
    for (i = X; i <= W; i++)
      destQuat[i] = startScale*startQ[i] + endScale*endQuat[i];
  } else {
    /* ends nearly opposite */
    destQuat[X] = -startQ[Y];  
    destQuat[Y] =  startQ[X];
    destQuat[Z] = -startQ[W];  
    destQuat[W] =  startQ[Z];
    
    startScale = sin((0.5 - t) * M_PI);
    endScale = sin(t * M_PI);
    for (i = X; i <= Z; i++)
      destQuat[i] = startScale*startQ[i] + endScale*destQuat[i];
  }

  return destQuat;
} 

Rotation Rotation::identity()
{
  Vec3d tmp(0, 0, 1);
  static Rotation ident( tmp, 0.f );
  return ident;
}

double& Rotation::operator []( int i )
{
  return quaternion[ i ];
}

const double& Rotation::operator []( int i ) const
{
  return quaternion[ i ];
}


///////////////////////////////////////////////////////////////////////////////
// class Line
///////////////////////////////////////////////////////////////////////////////

Line::Line(const Vec3d &p0, const Vec3d &p1): pos(p0), dir(p1 - p0)
{
}

void Line::setValue(const Vec3d &p0, const Vec3d &p1)
{
  pos = p0;
  dir = p1-p0;
  dir.normalize();
}


#ifdef HAVE_IOSTREAM
std::ostream& operator<< (std::ostream& it, const Vec2f &print) {
  return it <<"{" <<print[0] <<", " <<print[1] <<"}";
}

std::ostream& operator<< (std::ostream& it, const Vec2d &print) {
  it << print[0] << ", " << print[1];
  return it;
}
#endif


///////////////////////////////////////////////////////////////////////////////
// class Transform
///////////////////////////////////////////////////////////////////////////////

Transform::Transform()
{
  makeIdentity();
}

Transform::Transform( double a11, double a12, double a13, double a14,
                      double a21, double a22, double a23, double a24,
                      double a31, double a32, double a33, double a34,
                      double a41, double a42, double a43, double a44 )
{
  matrix[0][0] = a11;  matrix[0][1] = a12;
  matrix[0][2] = a13;  matrix[0][3] = a14;
  matrix[1][0] = a21;  matrix[1][1] = a22;
  matrix[1][2] = a23;  matrix[1][3] = a24;
  matrix[2][0] = a31;  matrix[2][1] = a32;
  matrix[2][2] = a33;  matrix[2][3] = a34;
  matrix[3][0] = a41;  matrix[3][1] = a42;
  matrix[3][2] = a43;  matrix[3][3] = a44;

}

Transform::Transform( TransMat m )
{
  matrix[0][0] = m[0][0]; matrix[0][1] = m[0][1];
  matrix[0][2] = m[0][2]; matrix[0][3] = m[0][3];
  matrix[1][0] = m[1][0]; matrix[1][1] = m[1][1];
  matrix[1][2] = m[1][2]; matrix[1][3] = m[1][3];
  matrix[2][0] = m[2][0]; matrix[2][1] = m[2][1];
  matrix[2][2] = m[2][2]; matrix[2][3] = m[2][3];
  matrix[3][0] = m[3][0]; matrix[3][1] = m[3][1];
  matrix[3][2] = m[3][2]; matrix[3][3] = m[3][3];
}


#undef DBG
//#include "utils/message.h"
void Transform::toRPY (double & roll, double & pitch, double & yaw)
{
  yaw = -atan2(-matrix[1][0], matrix[0][0]);

  double cosy = cos(-yaw);
  double siny = sin(-yaw);
  
  double cosr = cosy * matrix[1][1] + siny * matrix[0][1];
  double sinr = cosy * matrix[1][2] + siny * matrix[0][2];
  double cosp = cosy * matrix[0][0] - siny * matrix[1][0];
  
  roll = -atan2(sinr, cosr);
  pitch = -atan2(matrix[2][0], cosp);

#ifdef DBG
  Rotation test(roll, pitch, yaw);
  if (!test.equals(Rotation(*this), 0.001)) {
    printf("toRPY failure: %f %f %f", roll, pitch, yaw);
    abort();
  }
#endif
}


Transform::Transform (Vec3d translate, Rotation rotate) {
  setTransform(translate, rotate);
}


void Transform::setValue( const TransMat &m )
{
  matrix[0][0] = m[0][0]; matrix[0][1] = m[0][1];
  matrix[0][2] = m[0][2]; matrix[0][3] = m[0][3];
  matrix[1][0] = m[1][0]; matrix[1][1] = m[1][1];
  matrix[1][2] = m[1][2]; matrix[1][3] = m[1][3];
  matrix[2][0] = m[2][0]; matrix[2][1] = m[2][1];
  matrix[2][2] = m[2][2]; matrix[2][3] = m[2][3];
  matrix[3][0] = m[3][0]; matrix[3][1] = m[3][1];
  matrix[3][2] = m[3][2]; matrix[3][3] = m[3][3];
}


void Transform::setValue(double a11, double a12, double a13, double a14,
                         double a21, double a22, double a23, double a24, 
                         double a31, double a32, double a33, double a34, 
                         double a41, double a42, double a43, double a44)
{
  matrix[0][0] = a11; matrix[0][1] = a12;
  matrix[0][2] = a13; matrix[0][3] = a14;
  matrix[1][0] = a21; matrix[1][1] = a22;
  matrix[1][2] = a23; matrix[1][3] = a24;
  matrix[2][0] = a31; matrix[2][1] = a32;
  matrix[2][2] = a33; matrix[2][3] = a34;
  matrix[3][0] = a41; matrix[3][1] = a42;
  matrix[3][2] = a43; matrix[3][3] = a44;
}

void Transform::getValue( TransMat &m ) const
{
  m[0][0] = matrix[0][0]; m[0][1] = matrix[0][1];
  m[0][2] = matrix[0][2]; m[0][3] = matrix[0][3];
  m[1][0] = matrix[1][0]; m[1][1] = matrix[1][1];
  m[1][2] = matrix[1][2]; m[1][3] = matrix[1][3];
  m[2][0] = matrix[2][0]; m[2][1] = matrix[2][1];
  m[2][2] = matrix[2][2]; m[2][3] = matrix[2][3];
  m[3][0] = matrix[3][0]; m[3][1] = matrix[3][1];
  m[3][2] = matrix[3][2]; m[3][3] = matrix[3][3];
}

const TransMat& Transform::getValue() const
{
  return matrix;
}

void Transform::makeIdentity()
{
  matrix[0][0] = 1.f;  matrix[0][1] = 0.f;
  matrix[0][2] = 0.f;  matrix[0][3] = 0.f;
  matrix[1][0] = 0.f;  matrix[1][1] = 1.f;
  matrix[1][2] = 0.f;  matrix[1][3] = 0.f;
  matrix[2][0] = 0.f;  matrix[2][1] = 0.f;
  matrix[2][2] = 1.f;  matrix[2][3] = 0.f;
  matrix[3][0] = 0.f;  matrix[3][1] = 0.f;
  matrix[3][2] = 0.f;  matrix[3][3] = 1.f;
}

Transform Transform::identity()
{
  static Transform mident( 1.0f,0.0f,0.0f,0.0f,
                           0.0f,1.0f,0.0f,0.0f,
                           0.0f,0.0f,1.0f,0.0f,
                           0.0f,0.0f,0.0f,1.0f );
  return mident;
}

void Transform::setRotate( const Rotation &q )
{
  makeIdentity();
  q.getValue( *this );
}

void Transform::setScale( double s )
{
  makeIdentity();
  matrix[0][0] = s;    matrix[1][1] = s;       matrix[2][2] = s;
}

void Transform::setScale( const Vec3d &s )
{
  makeIdentity();
  matrix[0][0] = s[0]; matrix[1][1] = s[1];    matrix[2][2] = s[2];
}

void Transform::setTranslate( const Vec3d &t )
{
  makeIdentity();
  matrix[0][3] = t[0]; matrix[1][3] = t[1];    matrix[2][3] = t[2];
}

void Transform::setTransform(const Vec3d&    t, const Rotation& r)
{
  setRotate(r);
  matrix[0][3] = t[0]; matrix[1][3] = t[1];    matrix[2][3] = t[2];
}

void Transform::getTransform( Vec3d & t, Rotation & r) const
{
  t[0] = matrix[0][3];
  t[1] = matrix[1][3];
  t[2] = matrix[2][3];
  r.setValue(*this);
}

inline double det2(double a, double b, double c, double d)
{
  return a*d - b*c;
}

double Transform::det3( int r1, int r2, int r3, int c1, int c2, int c3) const
{
  return
    matrix[r1][c1] * det2(matrix[r2][c2],matrix[r3][c2],
                          matrix[r2][c3],matrix[r3][c3])
    - matrix[r1][c2] * det2(matrix[r2][c1],matrix[r3][c1],
                            matrix[r2][c3],matrix[r3][c3])
    + matrix[r1][c3] * det2(matrix[r2][c1],matrix[r3][c1],
                            matrix[r2][c2],matrix[r3][c2]);
}

double Transform::det3() const
{
  return det3(0,1,2,0,1,2);
}

double Transform::det4() const
{
  return   matrix[0][0] * det3(1,2,3,1,2,3)
    - matrix[0][1] * det3(1,2,3,0,2,3)
    + matrix[0][2] * det3(1,2,3,0,1,3)
    - matrix[0][3] * det3(1,2,3,0,1,2);
}

void Transform::getAdjoint(Transform& out) const
{
  out.matrix[0][0]  =   det3(1,2,3,1,2,3);
  out.matrix[1][0]  = - det3(1,2,3,0,2,3);
  out.matrix[2][0]  =   det3(1,2,3,0,1,3);
  out.matrix[3][0]  = - det3(1,2,3,0,1,2);
                                 
  out.matrix[0][1]  = - det3(0,2,3,1,2,3);
  out.matrix[1][1]  =   det3(0,2,3,0,2,3);
  out.matrix[2][1]  = - det3(0,2,3,0,1,3);
  out.matrix[3][1]  =   det3(0,2,3,0,1,2);
                                 
  out.matrix[0][2]  =   det3(0,1,3,1,2,3);
  out.matrix[1][2]  = - det3(0,1,3,0,2,3);
  out.matrix[2][2]  =   det3(0,1,3,0,1,3);
  out.matrix[3][2]  = - det3(0,1,3,0,1,2);
                                 
  out.matrix[0][3]  = - det3(0,1,2,1,2,3);
  out.matrix[1][3]  =   det3(0,1,2,0,2,3);
  out.matrix[2][3]  = - det3(0,1,2,0,1,3);
  out.matrix[3][3]  =   det3(0,1,2,0,1,2);
}

Transform Transform::inverse() const
{
  // Calculate the adjoint matrix
  Transform out;
  getAdjoint(out);
   
  // Scale the adjoint matrix with the determinant 
  // to get the inverse
  double det = det4();
  if ( det > 0.0f ) {
    for ( int i=0; i<4; i++ )
      for( int j=0; j<4; j++ )
        out.matrix[i][j] /= det;
  }
  else {
    // Simgular matrix
    out.makeIdentity();
  }
   
  return out;
}

Transform& Transform::multRight( const Transform &b )
{
  Transform mt;
  mt.matrix[0][0] = matrix[0][0] * b.matrix[0][0] +
    matrix[0][1] * b.matrix[1][0] + matrix[0][2] * b.matrix[2][0] +
    matrix[0][3] * b.matrix[3][0];
  mt.matrix[0][1] = matrix[0][0] * b.matrix[0][1] +
    matrix[0][1] * b.matrix[1][1] + matrix[0][2] * b.matrix[2][1] +
    matrix[0][3] * b.matrix[3][1];
  mt.matrix[0][2] = matrix[0][0] * b.matrix[0][2] +
    matrix[0][1] * b.matrix[1][2] + matrix[0][2] * b.matrix[2][2] +
    matrix[0][3] * b.matrix[3][2];
  mt.matrix[0][3] = matrix[0][0] * b.matrix[0][3] +
    matrix[0][1] * b.matrix[1][3] + matrix[0][2] * b.matrix[2][3] +
    matrix[0][3] * b.matrix[3][3];
  mt.matrix[1][0] = matrix[1][0] * b.matrix[0][0] +
    matrix[1][1] * b.matrix[1][0] + matrix[1][2] * b.matrix[2][0] +
    matrix[1][3] * b.matrix[3][0];
  mt.matrix[1][1] = matrix[1][0] * b.matrix[0][1] +
    matrix[1][1] * b.matrix[1][1] + matrix[1][2] * b.matrix[2][1] +
    matrix[1][3] * b.matrix[3][1];
  mt.matrix[1][2] = matrix[1][0] * b.matrix[0][2] +
    matrix[1][1] * b.matrix[1][2] + matrix[1][2] * b.matrix[2][2] +
    matrix[1][3] * b.matrix[3][2];
  mt.matrix[1][3] = matrix[1][0] * b.matrix[0][3] +
    matrix[1][1] * b.matrix[1][3] + matrix[1][2] * b.matrix[2][3] +
    matrix[1][3] * b.matrix[3][3];
  mt.matrix[2][0] = matrix[2][0] * b.matrix[0][0] +
    matrix[2][1] * b.matrix[1][0] + matrix[2][2] * b.matrix[2][0] +
    matrix[2][3] * b.matrix[3][0];
  mt.matrix[2][1] = matrix[2][0] * b.matrix[0][1] +
    matrix[2][1] * b.matrix[1][1] + matrix[2][2] * b.matrix[2][1] +
    matrix[2][3] * b.matrix[3][1];
  mt.matrix[2][2] = matrix[2][0] * b.matrix[0][2] +
    matrix[2][1] * b.matrix[1][2] + matrix[2][2] * b.matrix[2][2] +
    matrix[2][3] * b.matrix[3][2];
  mt.matrix[2][3] = matrix[2][0] * b.matrix[0][3] +
    matrix[2][1] * b.matrix[1][3] + matrix[2][2] * b.matrix[2][3] +
    matrix[2][3] * b.matrix[3][3];
  mt.matrix[3][0] = matrix[3][0] * b.matrix[0][0] +
    matrix[3][1] * b.matrix[1][0] + matrix[3][2] * b.matrix[2][0] +
    matrix[3][3] * b.matrix[3][0];
  mt.matrix[3][1] = matrix[3][0] * b.matrix[0][1] +
    matrix[3][1] * b.matrix[1][1] + matrix[3][2] * b.matrix[2][1] +
    matrix[3][3] * b.matrix[3][1];
  mt.matrix[3][2] = matrix[3][0] * b.matrix[0][2] +
    matrix[3][1] * b.matrix[1][2] + matrix[3][2] * b.matrix[2][2] +
    matrix[3][3] * b.matrix[3][2];
  mt.matrix[3][3] = matrix[3][0] * b.matrix[0][3] +
    matrix[3][1] * b.matrix[1][3] + matrix[3][2] * b.matrix[2][3] +
    matrix[3][3] * b.matrix[3][3];

  return operator =( mt );
}

Transform& Transform::multLeft( const Transform &b )
{
  Transform mt;
  mt.matrix[0][0] = b.matrix[0][0] * matrix[0][0] +
    b.matrix[0][1] * matrix[1][0] + b.matrix[0][2] * matrix[2][0] +
    b.matrix[0][3] * matrix[3][0];
  mt.matrix[0][1] = b.matrix[0][0] * matrix[0][1] +
    b.matrix[0][1] * matrix[1][1] + b.matrix[0][2] * matrix[2][1] +
    b.matrix[0][3] * matrix[3][1];
  mt.matrix[0][2] = b.matrix[0][0] * matrix[0][2] +
    b.matrix[0][1] * matrix[1][2] + b.matrix[0][2] * matrix[2][2] +
    b.matrix[0][3] * matrix[3][2];
  mt.matrix[0][3] = b.matrix[0][0] * matrix[0][3] +
    b.matrix[0][1] * matrix[1][3] + b.matrix[0][2] * matrix[2][3] +
    b.matrix[0][3] * matrix[3][3];
  mt.matrix[1][0] = b.matrix[1][0] * matrix[0][0] +
    b.matrix[1][1] * matrix[1][0] + b.matrix[1][2] * matrix[2][0] +
    b.matrix[1][3] * matrix[3][0];
  mt.matrix[1][1] = b.matrix[1][0] * matrix[0][1] +
    b.matrix[1][1] * matrix[1][1] + b.matrix[1][2] * matrix[2][1] +
    b.matrix[1][3] * matrix[3][1];
  mt.matrix[1][2] = b.matrix[1][0] * matrix[0][2] +
    b.matrix[1][1] * matrix[1][2] + b.matrix[1][2] * matrix[2][2] +
    b.matrix[1][3] * matrix[3][2];
  mt.matrix[1][3] = b.matrix[1][0] * matrix[0][3] +
    b.matrix[1][1] * matrix[1][3] + b.matrix[1][2] * matrix[2][3] +
    b.matrix[1][3] * matrix[3][3];
  mt.matrix[2][0] = b.matrix[2][0] * matrix[0][0] +
    b.matrix[2][1] * matrix[1][0] + b.matrix[2][2] * matrix[2][0] +
    b.matrix[2][3] * matrix[3][0];
  mt.matrix[2][1] = b.matrix[2][0] * matrix[0][1] +
    b.matrix[2][1] * matrix[1][1] + b.matrix[2][2] * matrix[2][1] +
    b.matrix[2][3] * matrix[3][1];
  mt.matrix[2][2] = b.matrix[2][0] * matrix[0][2] +
    b.matrix[2][1] * matrix[1][2] + b.matrix[2][2] * matrix[2][2] +
    b.matrix[2][3] * matrix[3][2];
  mt.matrix[2][3] = b.matrix[2][0] * matrix[0][3] +
    b.matrix[2][1] * matrix[1][3] + b.matrix[2][2] * matrix[2][3] +
    b.matrix[2][3] * matrix[3][3];
  mt.matrix[3][0] = b.matrix[3][0] * matrix[0][0] +
    b.matrix[3][1] * matrix[1][0] + b.matrix[3][2] * matrix[2][0] +
    b.matrix[3][3] * matrix[3][0];
  mt.matrix[3][1] = b.matrix[3][0] * matrix[0][1] +
    b.matrix[3][1] * matrix[1][1] + b.matrix[3][2] * matrix[2][1] +
    b.matrix[3][3] * matrix[3][1];
  mt.matrix[3][2] = b.matrix[3][0] * matrix[0][2] +
    b.matrix[3][1] * matrix[1][2] + b.matrix[3][2] * matrix[2][2] +
    b.matrix[3][3] * matrix[3][2];
  mt.matrix[3][3] = b.matrix[3][0] * matrix[0][3] +
    b.matrix[3][1] * matrix[1][3] + b.matrix[3][2] * matrix[2][3] +
    b.matrix[3][3] * matrix[3][3];
  return operator =( mt );
}

/*
 * We allow for src and dst to be the same vector in these
 */ 
void Transform::multVecMatrix( const Vec3d &src, Vec3d &dst ) const
{
  double s0 = src.v[0], s1 = src.v[1], s2 = src.v[2];
  double w = (s0 * matrix[0][3] + s1 * matrix[1][3] +
             s2 * matrix[2][3] + matrix[3][3]);
  dst[0] = (s0 * matrix[0][0] + s1 * matrix[1][0] +
            s2 * matrix[2][0] + matrix[3][0]) / w;
  dst[1] = (s0 * matrix[0][1] + s1 * matrix[1][1] +
            s2 * matrix[2][1] + matrix[3][1]) / w;
  dst[2] = (s0 * matrix[0][2] + s1 * matrix[1][2] +
            s2 * matrix[2][2] + matrix[3][2]) / w;
}

void Transform::multMatrixVec( const Vec3d & src, Vec3d& dst ) const
{
  double s0 = src.v[0], s1 = src.v[1], s2 = src.v[2];
  double w = s0*matrix[3][0] + s1*matrix[3][1] +
    s2*matrix[3][2] + matrix[3][3];
  dst[0] = (s0*matrix[0][0] + s1*matrix[0][1] +
            s2*matrix[0][2] + matrix[0][3])/w;
  dst[1] = (s0*matrix[1][0] + s1*matrix[1][1] +
            s2*matrix[1][2] + matrix[1][3])/w;
  dst[2] = (s0*matrix[2][0] + s1*matrix[2][1] +
            s2*matrix[2][2] + matrix[2][3])/w;
}

void Transform::multMatrixDir( const Vec3d& src, Vec3d& dst) const
{
  double s0 = src.v[0], s1 = src.v[1], s2 = src.v[2];
  double w = s0 * matrix[3][0] + s1 * matrix[3][1] +
    s2 * matrix[3][2] + matrix[3][3];
  dst[0] = (s0 * matrix[0][0] + s1 * matrix[0][1] +
            s2 * matrix[0][2]) / w;
  dst[1] = (s0 * matrix[1][0] + s1 * matrix[1][1] +
            s2 * matrix[1][2]) / w;
  dst[2] = (s0 * matrix[2][0] + s1 * matrix[2][1] +
            s2 * matrix[2][2]) / w;
}

void Transform::multMatrixLine(const Line& src, Line& dst) const
{
  multMatrixVec(src.pos,dst.pos);
  multMatrixDir(src.dir,dst.dir);
}

void Transform::print(FILE* fp) const
{
  ::fprintf(fp, "%f %f %f %f\n",
            matrix[0][0],matrix[0][1],matrix[0][2],matrix[0][3]);
  ::fprintf(fp, "%f %f %f %f\n",
            matrix[1][0],matrix[1][1],matrix[1][2],matrix[1][3]);
  ::fprintf(fp, "%f %f %f %f\n",
            matrix[2][0],matrix[2][1],matrix[2][2],matrix[2][3]);
  ::fprintf(fp, "%f %f %f %f\n",
            matrix[3][0],matrix[3][1],matrix[3][2],matrix[3][3]);
}

#ifdef HAVE_STREAM
std::ostream& operator<< (std::ostream& it, const Transform &print) {
  for (int i = 0; i < 4; i++) {
    for (int j = 0; j < 4; j++) {
      it.width(14);
      double el = print[j][i];
      if (fabs(el) < 1e-6 && el != 0.0)
        it << "~0";
      else 
        it << el;
    }
    it << std::endl;
  }
  return it;
}
#endif

Transform::operator double*()
{
  return &matrix[0][0];
}

Transform::operator TransMat&()
{
  return matrix;
}

double* Transform::operator []( int i )
{
  return &matrix[i][0];
}

const double* Transform::operator []( int i ) const
{
  return &matrix[i][0];
}

Transform& Transform::operator =( const TransMat &m )
{
  setValue( m );
  return *this;
}

Transform& Transform::operator =( const Transform &m )
{
  setValue( m.matrix );
  return *this;
}

Transform& Transform::operator =( const Rotation &q )
{
  setRotate( q );
  return *this;
}

Transform& Transform::operator *=( const Transform &m )
{
  multRight( m );
  return *this;
}

Transform operator*(const Transform& m1, const Transform& m2)
{
  Transform out;

  for ( int i=0; i<4; i++ ) {
    for ( int j=0; j<4; j++ ) {
      out.matrix[i][j] = 0.0f;
      for ( int k=0; k<4; k++) 
        out.matrix[i][j] += m1.matrix[i][k] * m2.matrix[k][j];
    }
  }
   
  return out;
}

int operator ==( const Transform &m1, const Transform &m2 )
{
  return ( m1.matrix[0][0] == m2.matrix[0][0] &&
           m1.matrix[0][1] == m2.matrix[0][1] &&
           m1.matrix[0][2] == m2.matrix[0][2] &&
           m1.matrix[0][3] == m2.matrix[0][3] &&
           m1.matrix[1][0] == m2.matrix[1][0] &&
           m1.matrix[1][1] == m2.matrix[1][1] &&
           m1.matrix[1][2] == m2.matrix[1][2] &&
           m1.matrix[1][3] == m2.matrix[1][3] &&
           m1.matrix[2][0] == m2.matrix[2][0] &&
           m1.matrix[2][1] == m2.matrix[2][1] &&
           m1.matrix[2][2] == m2.matrix[2][2] &&
           m1.matrix[2][3] == m2.matrix[2][3] &&
           m1.matrix[3][0] == m2.matrix[3][0] &&
           m1.matrix[3][1] == m2.matrix[3][1] &&
           m1.matrix[3][2] == m2.matrix[3][2] &&
           m1.matrix[3][3] == m2.matrix[3][3] );
    
}

int operator !=( const Transform &m1, const Transform &m2 )
{
  return !( m1 == m2 );
}

bool Transform::equals(const Transform & m, double tolerance ) const
{
  for (int i = 0; i < 4; i++)
    for (int j = 0; j < 4; j++) {
      double diff = matrix[i][j] - m.matrix[i][j];
      if (fabs(diff) > tolerance)
        return false;
    }

  return true;
}

void Transform::set(ConfigFile& transform)
{
  double values[16];
  if (transform.getDoubles("values", values, 16) == 16) {
    *this=Transform(values[0], values[1], values[2], values[3],
                    values[4], values[5], values[6], values[7],
                    values[8], values[9], values[10],values[11],
                    values[12],values[13],values[14],values[15]);
  } else {
    double angle = transform.getDouble("rotation.angle", 0.0);
    double axis[3];
    axis[0] = axis[1] = 0;
    axis[2] = 1.0;
    if (transform.getDoubles("rotation.axis", axis, 3) < 3) {
      axis[0] = axis[1] = 0;
    }
    double trans[3];
    memset(trans, 0, 3*sizeof(double));
    transform.getDoubles("translation", trans, 3);
    Vec3d tmp(trans), tmp2(axis);
    
    setTransform(tmp,
                 Rotation(tmp2, angle));

    double scale = transform.getDouble("scale", 1.0);
    if (scale != 1.0) {
      utils::Transform s;
      s.setScale(scale);
      (*this) *= s;
    }
  }
}
  

Transform operator*(double f, const Transform& m)
{
  return Transform(f*m.matrix[0][0], f*m.matrix[0][1],
                   f*m.matrix[0][2], f*m.matrix[0][3],
                   f*m.matrix[1][0], f*m.matrix[1][1],
                   f*m.matrix[1][2], f*m.matrix[1][3],
                   f*m.matrix[2][0], f*m.matrix[2][1], 
                   f*m.matrix[2][2], f*m.matrix[2][3],
                   f*m.matrix[3][0], f*m.matrix[3][1],
                   f*m.matrix[3][2], f*m.matrix[3][3]);
}

Transform operator+(const Transform& m1, const Transform& m2)
{
  return Transform(m1.matrix[0][0] + m2.matrix[0][0],
                   m1.matrix[0][1] + m2.matrix[0][1],
                   m1.matrix[0][2] + m2.matrix[0][2],
                   m1.matrix[0][3] + m2.matrix[0][3],
                   m1.matrix[1][0] + m2.matrix[1][0],
                   m1.matrix[1][1] + m2.matrix[1][1],
                   m1.matrix[1][2] + m2.matrix[1][2],
                   m1.matrix[1][3] + m2.matrix[1][3],
                   m1.matrix[2][0] + m2.matrix[2][0],
                   m1.matrix[2][1] + m2.matrix[2][1],
                   m1.matrix[2][2] + m2.matrix[2][2],
                   m1.matrix[2][3] + m2.matrix[2][3],
                   m1.matrix[3][0] + m2.matrix[3][0],
                   m1.matrix[3][1] + m2.matrix[3][1],
                   m1.matrix[3][2] + m2.matrix[3][2],
                   m1.matrix[3][3] + m2.matrix[3][3]);
}

Transform operator-(const Transform& m1, const Transform& m2)
{
  return Transform(m1.matrix[0][0] - m2.matrix[0][0],
                   m1.matrix[0][1] - m2.matrix[0][1],
                   m1.matrix[0][2] - m2.matrix[0][2],
                   m1.matrix[0][3] - m2.matrix[0][3],
                   m1.matrix[1][0] - m2.matrix[1][0],
                   m1.matrix[1][1] - m2.matrix[1][1],
                   m1.matrix[1][2] - m2.matrix[1][2],
                   m1.matrix[1][3] - m2.matrix[1][3],
                   m1.matrix[2][0] - m2.matrix[2][0],
                   m1.matrix[2][1] - m2.matrix[2][1],
                   m1.matrix[2][2] - m2.matrix[2][2],
                   m1.matrix[2][3] - m2.matrix[2][3],
                   m1.matrix[3][0] - m2.matrix[3][0],
                   m1.matrix[3][1] - m2.matrix[3][1],
                   m1.matrix[3][2] - m2.matrix[3][2],
                   m1.matrix[3][3] - m2.matrix[3][3]); 
}



/// Mat2f:

#ifdef HAVE_STREAM
std::ostream& operator<< (std::ostream& it, const Mat2f &print) {
  for (int i = 0; i < 2; i++) {
    for (int j = 0; j < 2; j++) 
      it << print.data[i][j] <<" ";
    it << std::endl;
  }
  return it;
}
#endif

__UTILS_END_NAMESPACE
