///////////////////////////////////////////////////////////////////////////////
//
//                               Generator.cc
//
// Implements routines for parsing specification strings and classes for 
// managing and generating reconfigurable interfaces
//
// Classes implemented for export:
//   GeneratorBase - base class for a table of reconfigurable interfaces
//      that can be chosen from and generated using a specification string
//   SpecElem - a config elem used to read in interface specifications
//
// Classes defined and implemented for internal use
//   Interface - class for creating a reconfigurable interface
//   CreatorCallback - callback for using a function to create a interface
//
///////////////////////////////////////////////////////////////////////////////

#include <utils/Basic.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <utils/SymbolTable.h>
#include <utils/Generator.h>
#include <utils/Creator.h>
#include <utils/String.h>
#include <utils/ConfigFile.h>
#include <utils/SpecElem.h>
#include <utils/Input.h>
#include <utils/Output.h>

__UTILS_BEGIN_NAMESPACE

// Interface - class for creating a reconfigurable interface
class Interface {
  public:
    Interface(GeneratorBase* g, CreatorBase* creator, Token* t);
    ~Interface();
    void* create(ConfigFile* params, SymbolTable* globals);

  private:
    CreatorBase* _creator;
    int _num_tokens;
    Token* _tokens;
    GeneratorBase* _generator;
};

// CreatorCallback - callback for using a function to create a interface
class CreatorCallback : public CreatorBase {
  public:
    CreatorCallback(void* (*c)(GeneratorBase*, ConfigFile*, SymbolTable*))
        { _creator = c; }

    virtual void* make(GeneratorBase* gen,
                       ConfigFile* params, SymbolTable* globals)
        { return (*_creator)(gen, params, globals); }

  private:
    void* (*_creator)(GeneratorBase*, ConfigFile*, SymbolTable*);
};

static SymbolDeleter<Interface>* InterfaceDeleter;

Interface::Interface(GeneratorBase* g, CreatorBase* c, Token* t)
{
    _generator = g;
    _creator = c;

    int count = 0;
    if (t) 
        while (t[count].name) count++;
    _num_tokens = count;

    _tokens = new Token[count];
    for (int i = 0; i<count; i++) {
        _tokens[i].name = String::copy(t[i].name);
        _tokens[i].value = String::copy(t[i].value);
    }
}

Interface::~Interface() {
    for (int i=0;i<_num_tokens;i++) {
        delete [] _tokens[i].name;
        delete [] _tokens[i].value;
    }
    delete [] _tokens;
    delete _creator;
}

void* Interface::create(ConfigFile* params, SymbolTable* globals)
{
    int kill_params = 0;
    if (!params) {
        kill_params = 1;
        params = new ConfigFile;
    }
    // set default values, ignoring them if they are duplicates
    ConfigElem* elem;
    for (int i=0;i<_num_tokens;i++) {
        elem = params->lookup(_tokens[i].name);
        if (!elem)
            params->set(_tokens[i].name, _tokens[i].value);
    }
    
//    printf("From %p, call to creator %p\n", this, _creator);
    void* res = _creator->make(_generator, params, globals);
    if (kill_params)
        delete params;
    
    return res;
}

GeneratorBase::GeneratorBase(int table_size = 20)
{
    _interfaces = new SymbolTable(table_size);
}

GeneratorBase::~GeneratorBase()
{
    delete _interfaces;
}

static bool _Initialized_Deleter = false;

void GeneratorBase::registerInterface(const char* tag,
                                      void* (*creator)(GeneratorBase*,
                                                       ConfigFile*,
                                                       SymbolTable*),
                                      Token* tokens)
{
    if (!_Initialized_Deleter) {
        InterfaceDeleter = new SymbolDeleter<Interface>;
        InterfaceDeleter->makeStatic();
        _Initialized_Deleter = true;
    }

    _interfaces->set(tag, new Interface(this, new CreatorCallback(creator),
                                        tokens),
                     InterfaceDeleter, true);
}

void GeneratorBase::registerInterface(const char* tag, CreatorBase* creator, 
                                      Token* tokens)
{
    if (!_Initialized_Deleter) {
        InterfaceDeleter = new SymbolDeleter<Interface>;
        InterfaceDeleter->makeStatic();
        _Initialized_Deleter = true;
    }

    _interfaces->set(tag, new Interface(this, creator, tokens),
                     InterfaceDeleter, true);
}

void* GeneratorBase::interface(ConfigFile* config, SymbolTable* globals)
{
    const char* tag = config->getString("tag");
    if (!tag)
        return NULL;

    Interface* interface = (Interface*) _interfaces->get(tag);
    if (!interface)
        return 0;
    
    void* res = interface->create(config, globals);

    return res;
}    

static bool _Initialized_Spec = false;

void* GeneratorBase::interface(const char* spec_string,
                               SymbolTable* globals)
{
    if (!spec_string || !strcmp(spec_string, "none") || !*spec_string) {
        return NULL;
    }

    if (!_Initialized_Spec) 
        SpecElem::initClass();

    ConfigFile config(new SpecElem());
    if (!config.parse(spec_string))
        return NULL;
    return interface(&config, globals);
}

///////////////////////////////////////////////////////////////////////////////
// SpecElem - the class for processing interface specifications

UTILS_BASE_SOURCE(SpecElem);

// initialize the SpecElem class
void SpecElem::initClass()
{
    if (_Initialized_Spec)
        return;

    ConfigFile::init();
    UTILS_BASE_INIT_CLASS(SpecElem, "spec", "struct");

    _Initialized_Spec = true;
}

SpecElem::SpecElem() 
{
    UTILS_BASE_CONSTRUCTOR(SpecElem);
}

SpecElem::~SpecElem()
{
    if (_output_buf)
        delete [] _output_buf;
}

// Get specification values.  Can convert to StructElem (its superclass)
// or as a string, thankfully.  Assumes we can only get one value either
// way
int SpecElem::getValue(Type dest_type, void* data, int max_num) const
{
    if (dest_type.isDerivedFrom(StructElem::getClassTypeId()))
        *((const StructElem**) data) = this;
    else if (dest_type == ConfigFile::stringType()) {
        // convert to a string via printing
        Output output;
        SpecElem* elem = (SpecElem*) this;
        if (!_output_buf) {
            elem->_size_output = 15;
            elem->_output_buf = new char[_size_output];
        }
        output.setBuffer(_output_buf, _size_output, Output::standardResize);
        elem->writeData(output);
        output.write('\0');
        int num_bytes;
        void* tmp;
        output.getBuffer(tmp, num_bytes);
        // cache results in output buffer
        elem->_output_buf = (char*) tmp;
        elem->_size_output = output.getBufferSize();
        const char** res = (const char**) data;
        *res = _output_buf;
    } else {
        get_error("Cannot get a non-struct from a struct", this);
        return 0;
    }
    return 1;
}

// get the tag
const char* SpecElem::getTag() const
{
    const char* tag;
    ConfigElem* elem = lookup("tag");
    if (!elem)
      return NULL;
    if (elem->getValue(ConfigFile::stringType(), &tag, 1) == 1)
        return tag;
    return NULL;
}

// helper for SpecElem::writeData which makes sure we don't write the tag
void SpecElem::write_struct_elem(const char* key, ConfigElem* elem, void* data)
{
    if (strcmp(key, "tag"))
        StructElem::write_struct_elem(key, elem, data);
}

// write the specification to output
void SpecElem::writeData(Output& output)
{
    output.write("{ ");
    const char* tag = getTag();
    if (tag) {
      output.write(tag);
      output.write(" :\n");
    } else {
      output.write("\n");
    }
    output.incrementIndent();

    _dict.applyToAll(write_struct_elem, &output);
    output.decrementIndent();
    output.indent();
    output.write("}");
}

bool SpecElem::parseVal(const char* val)
{
  ConfigElem* tag = new StringElem();
  set("tag", tag);
  
  if (!tag->parseVal(val)) {
    return false;
  }

  return true;
};  

__UTILS_END_NAMESPACE
