///////////////////////////////////////////////////////////////////////////////
//
//                                 FormatParser.cc
//
// This file implements a class that takes a format specifying string and
// parses it into the equivalent Format internal structure.
//
// Classes defined for export:
//    FormatParser
//
///////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <utils/Vector.h>
#include <utils/Mutex.h>
#include <utils/StringDict.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/Token.h>
#include <utils/formatting/FormatParser.h>

__UTILS_BEGIN_NAMESPACE

// Displays the format parsing error and allows processing to continue
// It prints the bad token and the string it was expecting
void FormatParser::parserError(Token* badToken, char* expecting)
{ 
    int i;

    if (!_verbose_errors)
        return;
    fprintf(stderr, "Format Parsing Error: Expected %s.\n%s\n", 
            expecting, _parse_string);
    for (i=0; i<badToken->loc; i++) 
        fprintf(stderr, " ");
    fprintf(stderr, "\n");
}

// Creates a list of items with the structure and returns the structure spec
// Appends each token to format list, except for commas
FormatSpec* FormatParser::structFormat()
{
    Vector<FormatSpec*> formatList;
    StructSpec* format = new StructSpec();

    formatList.clear();
    formatList.append(parse(format));

    Token* token = _reader->nextToken();
    while (token->type == Token::COMMA) {
        delete token;
        formatList.append(parse(format));
        token = _reader->nextToken();
    }

    _reader->ungetToken(token);

    format->setStruct(formatList);
    return format;
}

// Creates a format list for items in fixed array and returns the array spec
// r each token, append to format list...except commas
FormatSpec* FormatParser::fixedArrayFormat()
{ 
    Vector<Token*> argList;
    FormatSpec* nextFormat = parse(NULL); /* Formatter */

    Token* token = _reader->nextToken();
    if (token->type != Token::COLON) {
        parserError(token, "':'");
        delete token;
        return NULL;
    }
    delete token;

    argList.clear();
    Token* tmp = _reader->nextToken();
    int numberOfIndexes = 0;
    bool goAhead = true;
    do {
        if (tmp->type != Token::INT) {
            parserError(tmp, "an integer value");
            return NULL;
        } else {
            numberOfIndexes++;
            argList.append(tmp);

            tmp = _reader->nextToken();

            if (tmp->type == Token::COMMA) {
                delete tmp;
                tmp = _reader->nextToken();
                goAhead = true;
            } else if (tmp->type == Token::RBRACK) {
                delete tmp;
                goAhead = false;
            } else {
                parserError(tmp, "a ',' or ']'");
                return NULL;
            }
        }
    } while (goAhead);

    Vector<int>* size_list = new Vector<int>(numberOfIndexes);

    /* this time munch tokens */
    int arrayIndex;
    for (arrayIndex=0; arrayIndex < numberOfIndexes; arrayIndex++) {
        token = argList[arrayIndex];
        size_list->append(token->value.num);
        delete token;
    }

    return new FixedArraySpec(size_list, nextFormat);
}

// makes list of formats for items in variable length array embedded in the
// structure specified by parent
// r each token in array, put format in format list, except commas
FormatSpec* FormatParser::varArrayFormat(StructSpec* parent)
{ 
    Vector<Token*> argList;

    FormatSpec* nextFormat = parse(NULL); /* Formatter */

    Token* token = _reader->nextToken();
    if (token->type != Token::COLON) {
        parserError(token, "':'");
        delete token;
        return NULL;
    }
    delete token;

    argList.clear();
    Token* tmp = _reader->nextToken();
    int numberOfIndexes = 0;
    bool goAhead = true;
    do {
        if (tmp->type != Token::INT) {
            parserError(tmp, "an integer value");
            return NULL;
        } else {
            numberOfIndexes++;
            argList.append(tmp);

            tmp = _reader->nextToken();

            if (tmp->type == Token::COMMA) {
                delete tmp;
                tmp = _reader->nextToken();
                goAhead = true;
            } else if (tmp->type == Token::GT) {
                delete tmp;
                goAhead = false;
            }
            else {
                parserError(tmp, "a ',' or '>'");
                return NULL;
            }
        }
    } while (goAhead);

    Vector<int>* indices = new Vector<int>(numberOfIndexes);
    int arrayIndex;
    for (arrayIndex=0; arrayIndex < numberOfIndexes; arrayIndex++) {
        token = argList[arrayIndex];
        indices->append(token->value.num);
        delete token;
    }

    return new VarArraySpec(parent, indices, nextFormat);
}

// returns a format pointer that signifies it is a self pointer
FormatSpec* FormatParser::selfPtrFormat(StructSpec* parent)
{ 
    return new SelfPtrSpec(parent);
}


// returns a format pointer that signifies it is a pointer within a structure
FormatSpec* FormatParser::ptrFormat()
{ 
    return new PtrSpec(parse(NULL));
}

// returns an integer format for a token value, token
FormatSpec* FormatParser::lengthFormat(Token* token)
{
    return new LengthSpec(token->value.num);
}

// return the format for a token.  If the format is not registered, return NULL
// do a hash lookup on token name, if not there then generate error
FormatSpec* FormatParser::namedFormat(Token* token)
{
    FormatSpec* res;

    if (_format_table->find(token->value.str, res))
        return res;

    parserError(token, "registered format");

    return NULL;
}


// returns the format of a primitive (i.e. int, char...)
// makes a call to namedFormat
FormatSpec* FormatParser::primitiveFormat(Token* token)
{ 
    return namedFormat(token);
}

// parses a user-defined parameterized format spec of the form 
// (name parameters).  How it gets parsed depends on the name
FormatSpec* FormatParser::specialFormat()
{
    // lookup format template
    Token* token = _reader->nextToken();
    if (token->type != Token::STR) {
        parserError(token, "valid type name in parameterized specifier");
        delete token;
        return NULL;
    }
    FormatSpec* spec = namedFormat(token);
    if (!spec)
        return NULL;
    delete token;

    // use the template to create the real specifier from the parameters
    FormatSpec* res = spec->processParameters(this) ;
    if (!res)
        parserError(token, "valid parameters");

    // make sure there is a closing right paranthesis
    token = _reader->nextToken();
    if (token->type != Token::RPAREN)
        parserError(token, ")");
    delete token;

    return res;
}

// parse an input string and returns the format of string
// lex the tokens and parse each one on the fly
FormatSpec* FormatParser::parseString(const char* formString)
{
    if (!formString)
        return NULL;

    if (_mutex)
        _mutex->lock();
    _parse_string = formString;
    _reader->init(formString);  // set the token reader's string

    FormatSpec* res = parse(NULL);    // parse it
    if (_mutex)
        _mutex->unlock();
    return res;
}


// Parse the current token in the token reader into a format specifier and 
// return it.  switch(token->type) and get new formats for string recursively
// If the format is being built inside a structure, that structure is passed
// in in parent.
FormatSpec* FormatParser::parse(StructSpec* parent) 
{ 
    Token* token;
    Token* tmp;
    FormatSpec* returnrm = NULL;

    token = _reader->nextToken();

    switch(token->type) {
      case Token::LBRACE:
        delete token;
        returnrm = structFormat();
        token = _reader->nextToken();
        if (token->type != Token::RBRACE) parserError(token, "'}'");
        break;
      case Token::LBRACK:
        returnrm = fixedArrayFormat();
        break;
      case Token::LPAREN:
        returnrm = specialFormat();
        break;
      case Token::LT:
        if (!parent)
            parserError(token,
                        "var array format '<..>' embedded within a structure '{..}'");
        else
            returnrm = varArrayFormat(parent);
        break;
      case Token::STAR:
        tmp = _reader->nextToken();
        if (tmp->type == Token::BANG) {
            delete tmp;
            if (!parent) 
                parserError(token, 
                            "self pointer '*!' embedded within a structure '{..}'");
            else {
                returnrm = selfPtrFormat(parent);
            }
        }
        else {
            _reader->ungetToken(tmp);
            returnrm = ptrFormat();
        }
        break;
      case Token::INT:
        returnrm = lengthFormat(token);
        break;
      case Token::STR:
        returnrm = namedFormat(token);
        break;
      case Token::EOSTRING:
        if (parent)
            parserError(token, 
                        "additional tokens; premature end of string encountered");
        else
            returnrm = NULL;
        break;
      default:
        parserError(token, "a different token type");
    }

    delete token;

    return returnrm;
}

// Create a format parser 
FormatParser::FormatParser(Mutex* mutex)
{
    _format_table = new StringDict<FormatSpec*>;
    _mutex = mutex;
    _verbose_errors = true;

    registerFormat("string", new StringSpec);
    registerFormat("char", new CharSpec);
    registerFormat("short", new ShortSpec);
    registerFormat("long", new LongSpec);
    registerFormat("int", new IntSpec);
    registerFormat("float", new FloatSpec);
    registerFormat("double", new DoubleSpec);

    _reader = new TokenReader();
}

// Register a format name to be format
bool FormatParser::registerFormat(const char *name, FormatSpec* format)
{
    format->setName(name);
    format->ref();
    if (_format_table->enter(name, format))
        return true;
    else {
        format->unref();
        return false;
    }
}

// helper for deleting a format parser
static void delete_spec(const char*, FormatSpec* fmt)
{
    fmt->unref();
}

// delete a format parser
FormatParser::~FormatParser()
{
    delete _reader;

    _format_table->applyToAll(delete_spec);
    delete _format_table;
    delete _mutex;
}

__UTILS_END_NAMESPACE

