///////////////////////////////////////////////////////////////////////////////
//
//                              FormatAction.cc
//
// Implements the format actions.  A format action implements member functions
// which will process the standard format specifiers to perform an action,
// such as calculating the size of the data defined by a format specifier, or 
// printing out that data, etc.  Custom specifiers can key on the class type
// to implement their own responses to that action.  Format actions also have
// an internal database of tag/value pairs that can be propagated through 
// an action being applied to a set of data.
//
// Classes implemented for export:
//   FormatAction - the base format action class
//   RestrictAction - determines the basic unit for padding of a structure.
//       This will change for different machines.
//   DataSizeAction - calculate the size of the data structure specified
//       by a format specifier
//   DataAction - base class for actions which need to step through the
//       data structure defined by a format specifier
//   PrintAction - print data as specified by a format specifier
//   DeleteAction - delete allocated data as directed by a format specifier
//   ReadAction - read in data from an Input as directed by a format
//       specifier
//   FlatAction - checks if a format specifies a flat data structure
//   BufferSizeAction - calculates size of buffer needed to hold the 
//       flattened data of a data structure specified by a format.
//   PackAction - packs data into a flat buffer with all alignment padding
//       stripped out
//   UnpackAction - takes a flat buffer of data with no alignment padding
//       and builds it into the correct data structure
//   PrintFormatAction - prints formatter in string form to an output
//   FormatEqualsAction - checks if two format specifiers are equal
//   EqualsAction - checks if two formatted data structures are equal
//   CopyAction - makes a copy of a formatted data structure
//   CloneAction - makes a clone of a formatted data structure
//
// Classes implemented for internal use
//   DeferredVarArray - helps out ReadAction::actVarArray by holding info
//       necessary to defer reading in an array until the indices are read
//   
///////////////////////////////////////////////////////////////////////////////

#include <ctype.h>
#include <stdlib.h>

#include <utils/Input.h>
#include <utils/Vector.h>
#include <utils/List.h>
#include <utils/Output.h>
#include <utils/SymbolTable.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/FormatAction.h>
#include <utils/formatting/RestrictAction.h>
#include <utils/formatting/PrintAction.h>
#include <utils/formatting/DeleteAction.h>
#include <utils/formatting/ReadAction.h>
#include <utils/formatting/FlatAction.h>
#include <utils/formatting/BufferSizeAction.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/UnpackSizeAction.h>
#include <utils/formatting/PrintFormatAction.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/EqualsAction.h>
#include <utils/formatting/CopyAction.h>
#include <utils/formatting/CloneAction.h>

__UTILS_BEGIN_NAMESPACE

static Input* fb_std_in = (Input*) NULL;
static Output* fb_std_out = (Output*) NULL;

// format tag/data pair - it's a simple linked list
struct FormatSymbol {
  String tag;       // the tag
  void* data;         // the data
  FormatSymbol* next; // the next in the list
};

///////////////////////////////////////////////////////////////////////////////
//   FormatAction - the base format action class
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_ABSTRACT_SOURCE(FormatAction);

FormatAction::FormatAction()
{
  UTILS_BASE_CONSTRUCTOR(FormatAction);

  _head = 0L;
}

FormatAction::~FormatAction()
{
  clear();
}

// clear the extra value table
void FormatAction::clear()
{
  FormatSymbol* doomed;
  for (FormatSymbol* cur = _head; cur; ) {
    doomed = cur;
    cur = cur->next;
    delete doomed;
  }
  _head = 0L;
}

// add data in with tag
bool FormatAction::set(const char* tag, void* data)
{
  for (FormatSymbol* cur = _head; cur; cur=cur->next) 
    if (cur->tag == tag) {
      cur->data = data;
      return true;
    }
    
  FormatSymbol* sym = new FormatSymbol();
  sym->tag = tag;
  sym->data = data;
  sym->next = _head;
  _head = sym;

  return true;
}

// get the data associated with tag
void* FormatAction::get(const char* tag) const
{
  for (FormatSymbol* cur = _head; cur; cur=cur->next) 
    if (cur->tag == tag)
      return cur->data;

  return NULL;
}

// clear out my table, and inherit all tag/values from other
void FormatAction::inherit(const FormatAction& other)
{
  clear();
  for (FormatSymbol* cur = other._head; cur; cur=cur->next) 
    set(cur->tag.getString(), cur->data);
}

void FormatAction::initClass()
{
  UTILS_BASE_INIT_ABSTRACT_CLASS(FormatAction, "FormatAction", "Base");
}

// calculates the size of the fixed array given by the list of dimensions
int FormatAction::fixedArraySize(const Vector<int>& sizes)
{
  int size = 1;
  for (int i=0;i<sizes.numElems();i++) 
    size *= sizes.getValue(i);
  return size;
}

// default action for a self pointer is the same as for a pointer
bool FormatAction::actSelfPtr(StructSpec* parent)
{
  return actPtr(parent);
}

// default method for acting on a structure with size elements specified by 
// elems.  
bool FormatAction::actStruct(int size, FormatSpec** elems)
{
  if (!startStruct())  // first start the struct
    return false;
  // then act on each element
  for (int i=0;i<size;i++) {
    if (!elems[i])
      return false;
    if (!actStructElem(i, size, elems))
      return false;
  }
  return endStruct();  // finally end the struct
}

// default method for acting on the element of a structure
bool FormatAction::actStructElem(int i, int, FormatSpec** elems)
{
  return elems[i]->act(this);
}


///////////////////////////////////////////////////////////////////////////////
//   RestrictAction - determines the basic unit for padding of a structure.
//       This will change for different machine, i.e., how a structure is
//       padded will depend on whether the machine has word alignment, or
//       no alignment, etc.
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(RestrictAction);

RestrictAction::RestrictAction()
{
  UTILS_BASE_CONSTRUCTOR(RestrictAction);
  resetRestriction();
}

void RestrictAction::initClass()
{
  UTILS_BASE_INIT_CLASS(RestrictAction, "RestrictFormatAction",
                        "FormatAction");
}

// Find the quanta of a structure based on its elements
bool RestrictAction::actStruct(int size, FormatSpec** elems)
{
  // first find the maximum size element
  elems[0]->act(this);
  int maxSize = _restriction;
  for (int i=1;i<size;i++) {
    elems[i]->act(this);
    if (_restriction > maxSize) maxSize = _restriction;
  }
#if (ALIGN & ALIGN_LONGEST)
  _restriction = maxSize;  // some architecture align to the longest elem.
#elif (ALIGN & ALIGN_INT)
  // other architectures align to the largest or the size of an integer,
  // whichever is greater
  _restriction = (maxSize < (int) sizeof(int) ? maxSize : sizeof(int));
#endif
  return true;
}

// fixed arrays are restricted by their element type
bool RestrictAction::actFixedArray(const Vector<int>&, FormatSpec* fmt)
{
  return fmt->act(this);
}

// variable sized arrays don't have to worry about it, since they point to data
bool RestrictAction::actVarArray(StructSpec*, const Vector<int>&,
                                 FormatSpec*)
{
  return primRestrict(sizeof(void*));
}

// the restriction for a primitive type
bool RestrictAction::primRestrict(int size) 
{
#if (ALIGN & ALIGN_LONGEST)
  _restriction = size;
#elif (ALIGN & ALIGN_INT)
  _restriction = (size < (int) sizeof(int) ? size : sizeof(int));
#endif
  return true;
}


///////////////////////////////////////////////////////////////////////////////
// DataSizeAction - calculate the size of the data structure specified
//     by a format specifier
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(DataSizeAction);

DataSizeAction::DataSizeAction()
{
  UTILS_BASE_CONSTRUCTOR(DataSizeAction);
  resetSize();
}

void DataSizeAction::initClass()
{
  UTILS_BASE_INIT_CLASS(DataSizeAction, "DataSizeFormatAction",
                        "FormatAction");
}

// on a structure element, we simply increment by the size of the element,
// and then align the field to the structure restriction
bool DataSizeAction::actStructElem(int i, int size, FormatSpec** elems)
{
  if (!elems[i]->act(this))
    return false;
  _index = alignField(i, size, elems);
  return true;
}

// fixed array sizes are just the sizeof element * num of elements
bool DataSizeAction::actFixedArray(const Vector<int>& sizes,
                                   FormatSpec* fmt)
{
  _index += fixedArraySize(sizes)*fmt->dataSize();
  if (_packed)  // if still no padding detected
    _packed = fmt->isPackedType();   // look for it
  return true;
}

// variable array only takes up a pointer in the data structure
bool DataSizeAction::actVarArray(StructSpec*, const Vector<int>&,
                                 FormatSpec*)
{
  _index += sizeof(void*);
  _packed = false;   // variable arrays cannot be prepacked
  return true;
}

// align a field of a structure as the architecture specifies
int DataSizeAction::alignField(int formatIndex, int num_elems,
                               FormatSpec** elems)
{
#if (ALIGN & ALIGN_WORD)
  // if a structure needs to be aligned along word boundaries, i.e.,
  // odd indices into the structure need to be dealt with
  if (!ODDPTR(_index)) // at an even index, do nothing
    return _index;
  else {   // at an odd index
    int nextField = 1 + formatIndex;
    if (nextField == num_elems) {
      /* end of structure, simply pad by one byte */
      _packed = false;   // indicate padding detected
      return _index+1;
    } else {
      // in the middle of a structure
      FormatSpec* nextFormat = elems[nextField];
      // if next format is simple
      if (nextFormat->isSimpleType()) {
        // and odd, then the problem will take care of itself
        if (ODDPTR(nextFormat->dataSize())) 
          return _index; // so return the index
      }
      // otherwise pad out the data size
      _packed = false;
      return _index+1;
    }
  }
#elif ((ALIGN & ALIGN_LONGEST) | (ALIGN & ALIGN_INT))
  // in other architectures, we can use the RestrictAction to figure out
  // how to align fields

  int nextField = 1+formatIndex;
  RestrictAction restrict;
  int appropriateSize;
  if (nextField == num_elems) {
    restrict.actStruct(num_elems, elems);
  } else {
    /* on Sparc (and Mach/486 machines), element must start on boundary
       compatible with size of largest element within the sub-structure */
    elems[nextField]->act(&restrict);
  }
  appropriateSize = restrict.getRestriction();
  /* Round up to the next even multiple of "appropriateSize" */
  int rem = _index % appropriateSize;
  if (rem != 0) {   // things need padded
    _packed = false;
    _index += appropriateSize - rem;
  }
    
  return _index;
#else
  /* Should be an error. */
#endif
}


///////////////////////////////////////////////////////////////////////////////
// DataAction - base class for actions which need to step through the
//     data structure defined by a format specifier
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(DataAction);

DataAction::DataAction(void* data)
{
  UTILS_BASE_CONSTRUCTOR(DataAction);
  resetData(data);
  _sizes = 0L;
}

DataAction::~DataAction()
{
  delete _sizes;
}

void DataAction::initClass()
{
  UTILS_BASE_INIT_CLASS(DataAction, "DataFormatAction",
                        "DataSizeFormatAction");
}

// Assumes that at the current index there is a pointer to other data.
// Returns that pointer
unsigned char* DataAction::getSubData() const
{
  unsigned char** src = (unsigned char**) curData();
  if (!src)
    return (unsigned char*) NULL;
    
  unsigned char* data;
  memcpy((char*) &data, (char*) src, sizeof(unsigned char*));
  return data;
}

// When a data action starts a struct we store the index on the 
// _sizes stack so that we can find the indices for variable array sizes.
bool DataAction::startStruct()
{
  List<void>* sizes = (List<void>*) get("sizes");
  if (!sizes) {
    sizes = _sizes = new List<void>;
    set("sizes", sizes);
  }
  sizes->prepend((void*) (long) getSize());
  //    printf("pushing %d\n", getSize());

  return true;
}

// indicate a structure is ended by popping the size of the stack
bool DataAction::endStruct()
{
  List<void>* sizes = (List<void>*) get("sizes");
  if (!sizes)
    return false;

  bool res = sizes->length() > 0;
  sizes->pop();
  //    printf("popping\n");

  return res;
}

// Find the number of elements for a variable array dimension.  The variable 
// array size should be embedded at sizeIndex of the structure parent
int DataAction::varArraySize(StructSpec* parent, int sizeIndex)
{
  // make sure sizeIndex is in range and that DataAction::startStruct
  // has been called
  int start;
  List<void>* sizes = (List<void>*) get("sizes");
  if (!sizes || sizeIndex < 0 || sizeIndex > parent->numElems() ||
      !sizes->length())
    return 0;
  start = (int) (long) sizes->first();

  // go through the parent to get to the variable array dimension
  DataSizeAction sizeAct;
  for (int i=0; i<sizeIndex-1; i++) 
    sizeAct.actStructElem(i, parent->numElems(), parent->formats());

    // get the dimension and return it
  int size;
  memcpy((char*) &size, (char*) getData()+start+sizeAct.getSize(),
         sizeof(int));
  if (size < 0) // sanity check
    return 0;
  else
    return size;
}

// Find the number of elems for a variable array given the structure the 
// array is embedded in and the list of indices into the structure at which
// to find the dimensions
int DataAction::varArraySize(StructSpec* parent,
                             const Vector<int>& indices)
{
  int size = 1;
  for (int i=0;i<indices.numElems();i++)
    size *= varArraySize(parent, indices.getValue(i));
  return size;
}


///////////////////////////////////////////////////////////////////////////////
// PrintAction - print data to a file as specified by a format specifier
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PrintAction);

// static convenience method for deleting a structure
bool PrintAction::print(FormatSpec* spec, void* data,
                        const Output& output)
{
  // conveniently, if the spec is flat, we don't have to do anything
  if (!spec || !data)
    return false;

    // otherwise, set up the action 
  PrintAction printer(data, output);
  // and print away
  return spec->act(&printer);
}

PrintAction::PrintAction(void* data, FILE* fp)
  : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(PrintAction);
  if (!fp)
    fp = stdout;

  _output.setFilePointer(fp);
}

PrintAction::PrintAction(void* data, void* buffer, int size,
                         bool (*resize_proc)(void*&,int&,char*&,void*),
                         void* callback_data, int offset)
  : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(PrintAction);
  _output.setBuffer(buffer, size, resize_proc, callback_data, offset);
}

PrintAction::PrintAction(void* data, const Output& output)
  : DataAction(data), _output(output)
{
  UTILS_BASE_CONSTRUCTOR(PrintAction);
}

void PrintAction::initClass()
{
  UTILS_BASE_INIT_CLASS(PrintAction, "PrintAction", "DataFormatAction");
}

// extract and print an integer
bool PrintAction::actInt()
{
  int data;
  memcpy((char*) &data, (char*) curData(), sizeof(int));
  _output.write(data);
  return DataAction::actInt();  // increment index
}

// extract and print a short
bool PrintAction::actShort()
{
  short data;
  memcpy((char*) &data, (char*) curData(), sizeof(short));
  _output.write(data);
  return DataAction::actShort(); // increment index
}

// extract and print a long
bool PrintAction::actLong()
{
  long data;
  memcpy((char*) &data, (char*) curData(), sizeof(long));
  _output.write((int) data);
  return DataAction::actLong();  // increment index
}

// extract and print a float
bool PrintAction::actFloat()
{
  float data;
  memcpy((char*) &data, (char*) curData(), sizeof(float));
  _output.write(data);
  return DataAction::actFloat();  // increment index
}

// extract and print a double
bool PrintAction::actDouble()
{
  double data;
  memcpy((char*) &data, (char*) curData(), sizeof(double));
  _output.write(data);
  return DataAction::actDouble();  // increment index
}

// extract and print a character
bool PrintAction::actChar()
{
  char data;
  memcpy((char*) &data, (char*) curData(), sizeof(char));
  char buffer[10];
  if (isprint(data) && !isspace(data))
    sprintf(buffer, "%c", data);  // printable characters just printed
  else
    sprintf(buffer, "%02x", data); // unprintable characters printed as hex
  _output.write(buffer);
  return DataAction::actChar();  // increment index
}

// extract and print the string 
bool PrintAction::actString()
{
  _output.write('"');
  _output.write((char*) getSubData());
  _output.write('"');

  return DataAction::actString();
}

// extract and print a run of bytes of length size
bool PrintAction::actLength(int size)
{
  // run of bytes printed as hex with no spaces
  char buffer[20];
  for (int i=0;i<size;i++) {
    sprintf(buffer, "%02x", *(curData()+i));
    _output.write(buffer);
  }
  return DataAction::actLength(size);
}

// print data that is pointed to at the current index specified by fmt
bool PrintAction::actPtr(FormatSpec* fmt)
{
  unsigned char* data = getSubData();
  bool res;
  if (data) {
    _output.write('*');
    PrintAction subaction(data, _output);
    subaction.inherit(*this);
    res = fmt->act(&subaction);
  } else { // if data is null, just end
    res = true;
    _output.write("NULL");
  }
  return res && DataAction::actPtr(fmt);
}

// print elements separated by spaces
bool PrintAction::actStructElem(int i, int size, FormatSpec** elems)
{
  _output.write(' ');
  if (!DataAction::actStructElem(i, size, elems))
    return false;
  if (i != size-1)
    _output.write(',');
  return true;
}

// start structures with a bracket
bool PrintAction::startStruct()
{
  if (!DataAction::startStruct())
    return false;
  _output.write('{');
  return true;
}

// end structures with a bracket
bool PrintAction::endStruct()
{
  _output.write(" }");
  return DataAction::endStruct();
}

// print fixed arrays inside of square brackets with elements separated by
// spaces
bool PrintAction::actFixedArray(const Vector<int>& sizes,
                                FormatSpec* fmt)
{
  int size = fixedArraySize(sizes);
  _output.write('[');
  for (int i=0; i<size-1; i++) {
    _output.write(' ');
    if (!fmt->act(this))
      return false;
    _output.write(',');
  }
  _output.write(' ');
  if (size > 0) 
    if (!fmt->act(this))
      return false;
  _output.write(" ]");
  return true;
}

// print variable sized arrays inside of angle brackets with elements separated
// by spaces.  
bool PrintAction::actVarArray(StructSpec* parent,
                              const Vector<int>& indices,
                              FormatSpec* fmt)
{
  int size = varArraySize(parent, indices);
  _output.write('<');
  PrintAction subaction(getSubData(), _output);
  subaction.inherit(*this);
  for (int i=0; i<size-1; i++) {
    _output.write(' ');
    if (!fmt->act(&subaction))
      return false;
    _output.write(',');
  }
  _output.write(' ');
  if (size > 0) 
    if (!fmt->act(&subaction))
      return false;
  _output.write(" >");
  return DataAction::actVarArray(parent, indices, fmt);
}


///////////////////////////////////////////////////////////////////////////////
// DeleteAction - delete allocated data as directed by a format specifier
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(DeleteAction);

DeleteAction::DeleteAction(void* data,
                           unsigned char* buffer, int buf_size)
  : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(DeleteAction);

  _buffer = buffer;
  _buf_size = buf_size;
}

void DeleteAction::initClass()
{
  UTILS_BASE_INIT_CLASS(DeleteAction, "DeleteFormatAction",
                        "DataFormatAction");
}

// static convenience method for deleting a structure
bool DeleteAction::deleteStruct(FormatSpec* spec, void* data,
                                unsigned char* buffer, int buf_size)
{
  // conveniently, if the spec is flat, we don't have to do anything
  if (!spec || !data || spec->isFlatType())
    return false;

    // otherwise, set up the action 
  DeleteAction deletor(data, buffer, buf_size);
  // and delete away
  return spec->act(&deletor);
}

// static convenience method for deleting a structure
bool DeleteAction::deleteData(FormatSpec* spec, void* data,
                              unsigned char* buffer, int buf_size)
{
  if (!data)
    return false;

  bool res = deleteStruct(spec, data, buffer, buf_size);
  delete [] (unsigned char*) data;
  return res;
}

// returns true if the data is inside the given buffer
bool DeleteAction::inBounds(unsigned char* data) const
{
  return ((_buf_size != 0) &&
          (data >= _buffer) && (data < _buffer+_buf_size));
}

// strings have data to be deleted
bool DeleteAction::actString()
{
  unsigned char* data = getSubData();
  DataAction::actString();
  if (!inBounds(data))
    delete [] data;
  return true;
}

// delete data pointed to at current index
bool DeleteAction::actPtr(FormatSpec* fmt) 
{
  unsigned char* data = getSubData();
  if (data) {
    DeleteAction subaction(data, _buffer, _buf_size);
    subaction.inherit(*this);
    fmt->act(&subaction);
    if (!inBounds(data))
      delete [] data;
  }
  return DataAction::actPtr(fmt);
}

// fixed arrays don't get deleted themselves, but their guts might
bool DeleteAction::actFixedArray(const Vector<int>& sizes,
                                 FormatSpec* fmt)
{
  if (fmt->isFlatType()) { // if the fixed array is flat
    // no deleting needs to be done
    return DataAction::actFixedArray(sizes, fmt);
  }

  // otherwise we need to take care of each element
  int size = fixedArraySize(sizes);
  for (int i=0; i<size; i++) 
    if (!fmt->act(this))
      return false;

  return true;
}

// delete a varaible array, and maybe its guts        
bool DeleteAction::actVarArray(StructSpec* parent,
                               const Vector<int>& indices,
                               FormatSpec* fmt)
{
  unsigned char* data = getSubData();  // get the array data
  if (!fmt->isFlatType()) { //if the guts are not flat
    // otherwise take care of each subelement
    int size = varArraySize(parent, indices);
    DeleteAction subaction(data, _buffer, _buf_size);
    subaction.inherit(*this);
    for (int i=0; i<size; i++)
      if (!fmt->act(&subaction))
        return false;
  }
  if (!inBounds(data))   // if the data isn't in a pre-existing buffer
    delete [] data;    // delete the array data

  return DataAction::actVarArray(parent, indices, fmt);
}


///////////////////////////////////////////////////////////////////////////////
//   ReadAction - read in data from an Input as directed by a format
//       specifier
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(ReadAction);

// create a read action which will read from standard input into data
ReadAction::ReadAction(void* data)
  : DataAction(data), _input(*fb_std_in)
{
  init();
}

// create a read action which will read from input into data
ReadAction::ReadAction(Input& input, void* data)
  : DataAction(data), _input(input)
{
  init();
}

void ReadAction::init()
{
  UTILS_BASE_CONSTRUCTOR(ReadAction);
  _deferred_list = 0L;   // hopefully we won't need a deferred list
}

bool ReadAction::read(FormatSpec* spec, Input& input, void* data)
{
  if (!spec || !data)
    return false;

  ReadAction action(input, data);
  return spec->act(&action);
}

void ReadAction::initClass()
{
  UTILS_BASE_INIT_CLASS(ReadAction, "ReadFormatAction", "DataFormatAction");
  if (!fb_std_in)
    fb_std_in = new Input();
}

ReadAction::~ReadAction()
{
  if (_deferred_list)  
    delete _deferred_list;
}

// copy size bytes from data into the current position in the internal buffer
bool ReadAction::putData(void* data, int size)
{
  if (getData()) {
    memcpy((char*) curData(), (char*) data, size);
    return true;
  } else
    return false;
}

// read in an integer from the input
bool ReadAction::actInt()
{
  int data;
  if (!_input.read(data)) {
    data = -1;
    return false;
  }

  return putData(&data, sizeof(int)) && DataAction::actInt();
}

// read in a short from the input
bool ReadAction::actShort()
{
  short data;
  if (!_input.read(data)) {
    data = -1;
    return false;
  }

  return putData(&data, sizeof(short)) && DataAction::actShort();
}

// read in a long from the input
bool ReadAction::actLong()
{
  int data;
  if (!_input.read(data)) {
    data = -1;
    return false;
  }
  long ld = (long) data;

  return putData(&ld, sizeof(long)) && DataAction::actLong();
}

// read in a float from the input
bool ReadAction::actFloat()
{
  float data;
  if (!_input.read(data)) {
    data = -1;
    return false;
  }

  return putData(&data, sizeof(float)) && DataAction::actFloat();
}

// read in a double from the input
bool ReadAction::actDouble()
{
  double data;
  if (!_input.read(data)) {
    data = -1;
    return false;
  }

  return putData(&data, sizeof(double)) && DataAction::actDouble();
}

// read in a double from the input
bool ReadAction::actChar()
{
  char c;
  if (!_input.read(c)) {
    c = (char) -1;
    return false;
  } else {
    if (isxdigit(c)) {   // if read a hexadecimal digit
      char nextC;  // check to see if we are reading an unprintable char
      if (_input.get(nextC)) {
        if (isxdigit(nextC)) {  // that would be another hex digit
          char hexNum[3];     // build the hex number
          hexNum[0] = c;
          hexNum[1] = nextC;
          hexNum[2] = '\0';
          char* res;
          c = (char) strtol(hexNum, &res, 16); // extract it
          if (res == &hexNum[0]) {
            c = (char) -1;
            return false;
          }
        } else
          _input.putBack(nextC);  // not building an unprintable char
      } 
    }
  }
  return putData(&c, sizeof(char)) && DataAction::actChar();   
}

static void read_bracketed(Input& input, Vector<char>& buffer, char start)
{
  char end;
  switch (start) {
  case '{':
    end = '}';
    break;
  case '<':
    end = '>';
    break;
  case '(':
    end = ')';
    break;
  case '[':
    end = ']';
    break;
  default:
    return;
  }

  int depth = 1;
  int size = buffer.numElems();
  char c;
  while (input.get(c)) {
    if (c == start)
      depth++;
    else if (c == end)
      depth--;
    if (!depth) {
      break;
    }

    if (size >= buffer.getMaxNum()) {
      buffer.setMaxNum(buffer.getMaxNum() + 100);
    }

    buffer.append(c);
    size++;
  }
}
    
// read in a string from the input
bool ReadAction::actString()
{
  Vector<char> buffer;
  char c;
  char* data;
  if (!_input.read(c)) {
    data = (char*) NULL;
    return false;
  } else {
    bool putback = false;
    if (c == '"')  { // see if a quoted string
      while (_input.get(c) && (c != '"')) buffer.append(c);
    } else if (c == '{' || c == '[' || c == '(' || c == '<') {
      // if a bracketed read until the matching bracket character 
      read_bracketed(_input, buffer, c);
    } else {  // otherwise is a string delineated by whitespace
      buffer.append(c);
      while (_input.get(c) && !isspace(c) 
             && c!=',' && c!=']' && c !=')' && c !='>' && c!= '}'
             && c!= '{' && c!='[' && c != '(' && c != '<')
        buffer.append(c);

      putback = true;
    }

    // create new room for the data
    data = new char[buffer.numElems()+1];
    memcpy(data, buffer.getData(), buffer.numElems());
    // terminate it the data
    data[buffer.numElems()] = '\0';
    if (putback && !isspace(c))
      _input.putBack(c);
  }

  // store the pointer to the string 
  return putData(&data, sizeof(char*)) && DataAction::actString();
}

// read in a run of bytes of length size
bool ReadAction::actLength(int size)
{
  char c;
  unsigned char data;
  if (!_input.read(c)) {  // if we have a problem
    data = 0xff;     // read in -1's
    for (int i=0;i<size;i++)
      putData(&data, sizeof(char));
    return false;
  } else {  // read in the hex digits which are the bytes
    for (int i=0;i<size;i++) {
      data = 0xff;
      if ((i!=0) && !_input.get(c))
        ;
      else if (isxdigit(c)) {
        char nextC;
        if (_input.get(nextC)) {
          if (isxdigit(nextC)) {
            char hexNum[3];
            hexNum[0] = c;
            hexNum[1] = nextC;
            hexNum[2] = '\0';
            char* res;
            data = (unsigned char) strtol(hexNum, &res, 16);
            if (res == &hexNum[0])
              return false;
          } else {
            _input.putBack(nextC);
          }
        } 
      }
      if (!putData(&data, sizeof(unsigned char)) ||
          !DataAction::actLength(1))
        return false;
    }
  }
  return true;
}

// allocate the room needed to store data specifed by fmt and return it
unsigned char* ReadAction::allocFmt(FormatSpec* fmt)
{
  return new unsigned char[fmt->dataSize()];
}

// reading a structue starts with a left curly bracket
bool ReadAction::startStruct()
{
  if (!DataAction::startStruct())
    return false;
  char c;
  if (!_input.read(c) || c != '{') {
    fprintf(stderr, "Error reading structure brace start\n");
    return false;
  }
  return true;
}

// reading a structure ends with a right curly bracket
bool ReadAction::endStruct()
{
  if (_deferred_list)
    for (int i=0; i<_deferred_list->numElems(); i++)
      finalizeVarArray((*_deferred_list)[i]);
  return DataAction::endStruct();
}

// read in a structure element, they are separated by commas and terminated
// by a right curly bracket
bool ReadAction::actStructElem(int i, int size, FormatSpec** fmts)
{
  _cur_index = i+1;    // indicate where we are for variable sized arrays

  if (!DataAction::actStructElem(i, size, fmts))
    return false;
  char c;
  if (_input.read(c)) {
    if (i == size-1 &&  c == '}')
      return true;
    else if (c == ',')
      return true;
  }
        
  fprintf(stderr, "Error reading structure separator\n");
  return false;
}

// Read in pointer to data specifed by fmt
bool ReadAction::actPtr(FormatSpec* fmt)
{
  char c;
  unsigned char* data;
    
  if (!_input.read(c)) {
    DataAction::actPtr(fmt);
    return false;
  }

  // check for word NULL terminating the self referencing structure read
  char* null_word = "NULL";
  char putbacks[4];
  int i = 0;;
  putbacks[0] = c;
  while (1) {
    if (i >= 3) {
      i = 4;
      break;
    }
    if (c != null_word[i] || (!_input.get(c)))
      break;
    i++;
    putbacks[i] = c;
  }

  bool res = true;
  if (i==4)
    data = 0L;  // if it matches, the current data is null
  else {  // other wise read in another structure
    for (int j=i; j>=0; j--) 
      _input.putBack(putbacks[j]);
    if (!_input.read(c) || c!= '*') {
      fprintf(stderr, "Error reading self ptr\n");
      data = 0L;
      res = false;
    } else {
      data = allocFmt(fmt);
      ReadAction subaction(_input, data);
      subaction.inherit(*this);
      res = fmt->act(&subaction);
    }
  }
  putData(&data, sizeof(unsigned char*));  // store the data

  return res && DataAction::actPtr(fmt);
}

// read in a fixed array
bool ReadAction::actFixedArray(const Vector<int>& sizes,
                               FormatSpec* fmt)
{
  char c;
  if (!_input.read(c) || c!= '[') {
    DataAction::actFixedArray(sizes, fmt);
    printf("Error reading fixed array left bracket\n");
    return false;
  }
    
  int size = fixedArraySize(sizes);
  for (int i=0; i<size; i++) {
    if (!fmt->act(this))
      return false;
    if (i != size-1) {
      if (!_input.read(c) || c!= ',') {
        DataAction::actFixedArray(sizes, fmt);
        printf("Error reading fixed array separator\n");
        return false;
      }
    }
  }

  if (!_input.read(c) || c!= ']') {
    DataAction::actFixedArray(sizes, fmt);
    printf("Error reading fixed array right bracket\n");
    return false;
  }

  return true;
}

// read in a variable sizes array
bool ReadAction::actVarArray(StructSpec* parent,
                             const Vector<int>& indices, FormatSpec* fmt)
{
  int i;

  for (i=0;i<indices.numElems(); i++) 
    /* if the index for the dimension of a variable sized array is after
       the index at which the variable sized array is defined (a very
       stupid design, by the way, then we must defer reading of the 
       array until the dimension is read */
    if (indices.getValue(i) > _cur_index) {
      if (indices.getValue(i)-1 != _cur_index)
        deferVarArray(parent, indices, fmt);
      return DataAction::actVarArray(parent, indices, fmt);
    }

  // otherwise, process and store the array
  unsigned char* data = processVarArray(parent, indices, fmt);
  return (putData(&data, sizeof(unsigned char*)) &&
          DataAction::actVarArray(parent, indices, fmt));
}

// returns true if the data specified by fmt can be placed at index bytes into
// an array
static bool placeable(FormatSpec* fmt, int index)
{
#if (ALIGN & ALIGN_WORD)
  if (!ODDPTR(index)) 
    return true;
  else {
    if (ODDPTR(fmt->dataSize()))
      return true;
    else return false;
  }
#elif ((ALIGN & ALIGN_LONGEST) | (ALIGN & ALIGN_INT))
  int appropriateSize, rem;

  /* on Sparc (and Mach/486 machines), element must start on boundary
     compatible with size of largest element within the sub-structure */
  RestrictAction restrict;
  fmt->act(&restrict);
  appropriateSize = restrict.getRestriction();

  /* Round up to the next even multiple of "appropriateSize" */
  rem = index % appropriateSize;
  return (rem == 0) ;
#else
  /* Should be an error. */
#endif
}

// defer the reading of a variable sized array by adding a new element
// to the _deferred_list, including the string containing the uninterpreted
// variable sized array data
void ReadAction::deferVarArray(StructSpec* parent,
                               const Vector<int>& indices,
                               FormatSpec* fmt)
{
  if (!_deferred_list) 
    _deferred_list = new Vector<DeferredVarArray>(1);

  DeferredVarArray& elem = _deferred_list->append();
  elem.parent = parent;
  elem.indices = &indices;
  elem.fmt = fmt;
  elem.index = getSize();
  elem.buffer = new Vector<char>(100);

  char c;
  if (!_input.read(c) || c!= '<') {
    printf("Error reading variable array left bracket\n");
    _deferred_list->remove(_deferred_list->numElems()-1);
    return;
  }
  elem.buffer->append('<');
    
  int char_count = 1;
  int level = 0;
  while (level >= 0) {
    if (!_input.get(c)) {
      printf("Error reading variable array\n");
      _deferred_list->remove(_deferred_list->numElems()-1);
      return;
    }
    elem.buffer->append(c);

    char_count++;
    if (char_count >= 100) {
      elem.buffer->setMaxNum(elem.buffer->numElems() + 100);
      char_count = 0;
    }

    if (c == '<')
      level++;
    else if (c == '>')
      level--;
  }
}

// read in a deferred variable size array
void ReadAction::finalizeVarArray(DeferredVarArray& deferred)
{
  Input temp;
  temp.setBuffer(deferred.buffer->getData(), deferred.buffer->numElems());

  ReadAction tmpaction(temp, getData());
  tmpaction.inherit(*this);
  unsigned char* data =
    tmpaction.processVarArray(deferred.parent,
                              *deferred.indices, deferred.fmt);
  memcpy((char*) getData()+deferred.index,(char*) &data,
         sizeof(unsigned char*));
  delete deferred.buffer;
}

// read the variable sized array from the input
unsigned char* ReadAction::processVarArray(StructSpec* parent,
                                           const Vector<int>& indices,
                                           FormatSpec* fmt)
{
  char c;
  if (!_input.read(c) || c!= '<') {
    printf("Error reading variable array left bracket\n");
    return (unsigned char*) NULL;
  }
    
  int size = varArraySize(parent, indices);
  int fmt_size = fmt->dataSize();

  unsigned char* data = new unsigned char[size*fmt_size];
  ReadAction subaction(_input, data);
  subaction.inherit(*this);
  for (int i=0; i<size; i++) {
    fmt->act(&subaction);
    if (i != size-1) {
      if (!_input.read(c) || c!= ',') {
        printf("Error reading variable array separator\n");
        delete [] data;
        return (unsigned char*) NULL;
      }
    }
  }

  if (!_input.read(c) || c!= '>') 
    printf("Error reading variable array right bracket\n");

  return data;
}

///////////////////////////////////////////////////////////////////////////////
// FlatAction - checks if a format specifies a flat data structure
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(FlatAction);

FlatAction::FlatAction()
{
  UTILS_BASE_CONSTRUCTOR(FlatAction);
  _flat = false;
}

void FlatAction::initClass()
{
  UTILS_BASE_INIT_CLASS(FlatAction, "FlatFormatAction", "FormatAction");
}

bool FlatAction::actStruct(int size, FormatSpec** elems)
{
  for (int i=0; i<size; i++) 
    if (!elems[i]->isFlatType()) {
      _flat = false;
      return true;
    }
  _flat = true;
  return true;
}

bool FlatAction::actFixedArray(const Vector<int>&, FormatSpec* fmt)
{
  _flat = fmt->isFlatType();
  return true;
}

bool FlatAction::actVarArray(StructSpec*,
                             const Vector<int>&,FormatSpec*)
{
  _flat = false;
  return true;
}

///////////////////////////////////////////////////////////////////////////////
// BufferSizeAction - calculates size of buffer needed to hold the 
//   flattened data of a data structure specified by a format.  
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(BufferSizeAction);

int BufferSizeAction::size(FormatSpec* spec, void* data)
{
  if (!spec || !data)
    return 0;

  BufferSizeAction action(data);
  if (!spec->act(&action))
    return 0;
    
  return action.getBufSize();
}

BufferSizeAction::BufferSizeAction(void* data) : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(BufferSizeAction);

  _buf_size = 0;
}

void BufferSizeAction::initClass()
{
  UTILS_BASE_INIT_CLASS(BufferSizeAction, "BufferSizeFormatAction",
                        "FormatAction");
}

// calculate the buffer space needed for a particular string
bool BufferSizeAction::actString()
{
  unsigned char* data = getSubData();
  if (!DataAction::actString())
    return false;

  if (data) 
    _buf_size += (strlen((char*) data) + 1);
  else  // null pointers translate to empty strings
    _buf_size++;

  return true;
}

// calculate the buffer space needed for a pointer
bool BufferSizeAction::actPtr(FormatSpec* fmt)
{
  unsigned char* data = getSubData();
  _buf_size += sizeof(char);
  if (data) {
    BufferSizeAction subaction(data);
    subaction.inherit(*this);
    if (!fmt->act(&subaction))
      return false;
    _buf_size += subaction._buf_size;
  } 
  return DataAction::actPtr(fmt);
}

// calculate the buffer space needed for a fixed array
bool BufferSizeAction::actFixedArray(const Vector<int>& sizes,
                                     FormatSpec* fmt)
{
  if (fmt->isPackedType()) {
    _buf_size += fmt->dataSize()*fixedArraySize(sizes);
    return DataAction::actFixedArray(sizes, fmt);
  } else {
    int size = fixedArraySize(sizes);
    int i;
    for (i=0;i<size;i++)
      if (!fmt->act(this))
        return false;
    return true;
  }
}

// calculate the buffer space needed for a variable array
bool BufferSizeAction::actVarArray(StructSpec* parent,
                                   const Vector<int>& indices,
                                   FormatSpec* fmt)
{
  int size = varArraySize(parent, indices);
  _buf_size += sizeof(int);
  if (fmt->isPackedType()) 
    _buf_size += size * fmt->dataSize();
  else {
    BufferSizeAction subaction(getSubData());
    subaction.inherit(*this);
    for (int i = 0; i < size; i++) {
      if (!fmt->act(&subaction))
        return false;
    }
    _buf_size += subaction._buf_size;
  }
  return DataAction::actVarArray(parent, indices, fmt);
}


///////////////////////////////////////////////////////////////////////////////
// PackAction - packs data into a flat buffer with all alignment padding
//                stripped out
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PackAction);

bool PackAction::pack(FormatSpec* spec, unsigned char* buffer,
                      int buf_size, void* data, int index)
{
  if (!spec || !data)
    return false;

  if (spec->isFlatType()) { // flat things are easy
    memcpy((char*) buffer, (char*) data, spec->dataSize());
    return true;
  } else {  
    PackAction action(buffer, buf_size, data, index);
    return spec->act(&action);
  }
}

// create a packer action.  We will pack data, starting at index, into buffer
// (which is buf_size bytes long).
PackAction::PackAction(unsigned char* buffer, int buf_size, void* data,
                       int index) : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(PackAction);

  setBuffer(buffer, buf_size, index);
}

void PackAction::initClass()
{
  UTILS_BASE_INIT_CLASS(PackAction, "PackFormatAction", "DataFormatAction");
}

// this action will pack into buffer starting at index.  buffer is buf_size
// bytes long
void PackAction::setBuffer(unsigned char* buffer, int buf_size, int index)
{
  if (index >= buf_size) {
    _buffer = (unsigned char*) NULL;
    _buf_size = _buf_index = 0;
  } else {
    _buffer = buffer;
    _buf_size = buf_size;
    _buf_index = index;
  }
}

bool PackAction::actLength(int len)
{
  return toBuffer(curData(), len) && DataAction::actLength(len);
}

bool PackAction::actChar()
{
  return toBuffer(curData(), sizeof(char)) &&  DataAction::actChar();
}

bool PackAction::actShort()
{
  return toBuffer(curData(), sizeof(short)) && DataAction::actShort();
}

bool  PackAction::actLong()
{
  return toBuffer(curData(), sizeof(long)) && DataAction::actLong();
}

bool  PackAction::actDouble()
{
  return toBuffer(curData(), sizeof(double)) && DataAction::actDouble();
}

bool PackAction::actFloat()
{
  return toBuffer(curData(), sizeof(float)) && DataAction::actFloat();
}

bool PackAction::actInt()
{
  return toBuffer(curData(), sizeof(int)) && DataAction::actInt();
}

// strings are just packed as strings
bool PackAction::actString()
{
  char* str = (char*) getSubData();
  bool res;
  if (!str) {
    char c = '\0';
    res = toBuffer((unsigned char*) &c, sizeof(char));
  } else {
    res = toBuffer((unsigned char*) str, strlen(str)+1);
  }
  return res && DataAction::actString();
}

bool PackAction::actPtr(FormatSpec* fmt)
{
  unsigned char* data = getSubData();

  // Packs a pointer to a structure specified by fmt
  // pointed to structures have a one byte header, '\0' indicates a NULL 
  // value and 'Z' indicates a non-NULL value
  if (data) {
    char c = 'Z';
    if (!toBuffer((unsigned char*) &c, sizeof(char)))
      return false;
    PackAction subaction(_buffer, _buf_size, data, _buf_index);
    subaction.inherit(*this);
    if (!fmt->act(&subaction))
      return false;
    _buf_index = subaction._buf_index;
  } else {
    char c = '\0';
    if (!toBuffer((unsigned char*) &c, sizeof(char)))
      return false;
  }

  return DataAction::actPtr(fmt);
}

bool PackAction::actFixedArray(const Vector<int>& sizes, FormatSpec* fmt)
{
  if (fmt->isPackedType()) {
    return (toBuffer(curData(), fmt->dataSize()*fixedArraySize(sizes)) &&
            DataAction::actFixedArray(sizes, fmt));
  } else {
    int size = fixedArraySize(sizes);
    int i;
    for (i=0;i<size;i++)
      if (!fmt->act(this))
        return false;
    return true;
  }
}

bool PackAction::actVarArray(StructSpec* parent,
                             const Vector<int>& indices, FormatSpec* fmt)
{
  int size = varArraySize(parent, indices);
  if (!toBuffer((unsigned char*) &size, sizeof(int)))
    return false;
  if (fmt->isPackedType()) {
    if (!toBuffer(getSubData(), size*fmt->dataSize()))
      return false;
  } else {
    PackAction subaction(_buffer, _buf_size, getSubData(), _buf_index);
    subaction.inherit(*this);
    for (int i = 0; i < size; i++) {
      if (!fmt->act(&subaction))
        return false;
    }
    _buf_index = subaction._buf_index;
  }
  return DataAction::actVarArray(parent, indices, fmt);
}


///////////////////////////////////////////////////////////////////////////////
// UnpackAction - takes a flat buffer of data with no alignment padding
//                  and builds it into the correct data structure
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(UnpackAction);

bool UnpackAction::unpack(FormatSpec* spec,
                          unsigned char* buffer, int buf_size, void* data,
                          int index, int align, int byte_order)
{
  if (!spec || !buffer || !data)
    return false;

  if (spec->isPackedType() && byte_order == BYTE_ORDER) {
    memcpy((char*) data, (char*) buffer, spec->dataSize());
    return true;
  } else {
    UnpackAction action(buffer, buf_size, index, data,
                        align, byte_order);
    return spec->act(&action);
  }
}

void* UnpackAction::unpack(FormatSpec* spec,
                           unsigned char* buffer, int buf_size, 
                           int align, int byte_order)
{
  if (!spec || !buffer)
    return NULL;

  int size = spec->dataSize();
  if (!size)
    return NULL;
  unsigned char* data = new unsigned char[size];
  if (!unpack(spec, buffer, buf_size, data, 0, align, byte_order)) {
    delete [] data;
    return NULL;
  }
  return data;
}
    
// Create action to unpack from a buffer.  If buf_size is 0, then all data
// must be allocated.  If it is non-zero, then all data possible will be used
// "in place," i.e., pointers will reference the data in buffer directly, if
// possible. align and byte_order specify the alignment and byte order of the 
// data in buffer.
UnpackAction::UnpackAction(unsigned char* buffer, int buf_size, int index,
                           void* data, int align, int byte_order)
  : DataAction(data)
{
  UTILS_BASE_CONSTRUCTOR(UnpackAction);

  setBuffer(buffer, buf_size, index, align, byte_order );
}

void UnpackAction::initClass()
{
  UTILS_BASE_INIT_CLASS(UnpackAction, "UnpackFormatAction",
                        "DataFormatAction");
}

// set the buffer to unpack from buffer.  If buf_size is 0, then all data
// must be allocated.  If it is non-zero, then all data possible will be used
// "in place," i.e., pointers will reference the data in buffer directly, if
// possible. align and byte_order specify the alignment and byte order of the 
// data in buffer.
void UnpackAction::setBuffer(unsigned char* buffer, int buf_size, int index,
                             int align, int byte_order)
{
  if (buf_size && index >= buf_size) {
    _buffer = (unsigned char*) NULL;
    _buf_size = _buf_index = 0;
  } else {
    _buffer = buffer;
    _buf_size = buf_size;
    _buf_index = index;
  }
  _align = align;
  _byte_order = byte_order;
}

// convenience method for unpacking an integer and returning it
int UnpackAction::getInt()
{
  int res;
  bytesToInt((unsigned char*) &res);
  return res;
}

bool UnpackAction::actLength(int len)
{
  fromBuffer(curData(), len);
  return DataAction::actLength(len);
}

bool UnpackAction::actChar()
{
  fromBuffer(curData(), sizeof(char));
  return DataAction::actChar();
}

bool UnpackAction::actShort()
{
  bytesToShort(curData());
  return DataAction::actShort();
}

bool UnpackAction::actLong()
{
  bytesToLong(curData());
  return DataAction::actLong();
}

bool UnpackAction::actDouble()
{
  bytesToDouble(curData());
  return DataAction::actDouble();
}

bool UnpackAction::actFloat()
{
  bytesToFloat(curData());
  return DataAction::actFloat();
}

bool UnpackAction::actInt()
{
  bytesToInt(curData());
  return DataAction::actInt();
}

bool UnpackAction::actString()
{
  int length = strlen((char*) (_buffer + _buf_index)) + 1;
  char* str;
  if (isInline()) {   // try and reference strings if possible
    str = (char*) (_buffer + _buf_index);
    advanceBuffer(length);
  } else {  // oh well, we have to allocate and copy
    str = new char[length];
    fromBuffer(str, length);
  }
  toData(&str, sizeof(unsigned char*));
  return DataAction::actString();
}

bool UnpackAction::actPtr(FormatSpec* fmt)
{
  char c;
  fromBuffer((unsigned char*) &c, sizeof(char));
  unsigned char* newStruct;
  if (c == '\0')
    newStruct = (unsigned char*) NULL;
  else {
    if (fmt->isPackedType() && isInline() && placeable(fmt, getIndex())) {
      // we can only reference structures that are already aligned 
      // correctly in the buffer
      newStruct = _buffer+_buf_index;
      if (inNativeFormat())  
        advanceBuffer(fmt->dataSize());
      else {   // some inplace reording may need to be done
        UnpackAction subaction(_buffer, _buf_size, _buf_index,
                               newStruct, _align, _byte_order);
        subaction.inherit(*this);
        if (!fmt->act(&subaction))
          return false;
        _buf_index = subaction._buf_index;
      }
    } else {
      newStruct = new unsigned char[fmt->dataSize()];
      UnpackAction subaction(_buffer, _buf_size, _buf_index,
                             newStruct, _align, _byte_order);
      subaction.inherit(*this);
      if (!fmt->act(&subaction))
        return false;
      _buf_index = subaction._buf_index;
    }
  }
  toData(&newStruct, sizeof(unsigned char*));

  return DataAction::actPtr(fmt);
}

// calculate the buffer space needed for a fixed array
bool UnpackAction::actFixedArray(const Vector<int>& sizes,
                                 FormatSpec* fmt)
{
  if (fmt->isPackedType() && inNativeFormat()) {
    // if we're lucky, we can just copy the data
    fromBuffer(curData(), fmt->dataSize()*fixedArraySize(sizes));
  } else {
    // no, we have to go through piece by piece
    transferArrayData(fmt, fixedArraySize(sizes));
  }
  return DataAction::actFixedArray(sizes, fmt);
}

// Tries to efficiently transfer an array of data to a data structure.  Does 
// not increment data pointers, just buffer pointers.
bool UnpackAction::transferArrayData(FormatSpec* fmt, int arraySize)
{
  bool inplace = dataInPlace();
  int i;

  if (inNativeFormat()) {
    if (fmt->isPackedType()) // try and do things in a block
      fromBuffer(curData(), fmt->dataSize()*arraySize);
    else   // oh well, need to do it piece by piece
      for(i=0;i < arraySize;i++) 
        if (!fmt->act(this))
          return false;
  } else if (fmt->isSimpleType()) {
    // arrays are usually primitives, so we can inline them for efficiency
    switch (fmt->dataSize()) {
    case sizeof(char): 
      /* Characters, do not need to do anything but copy. */
      if (inplace)
        advanceBuffer(arraySize);
      else
        fromBuffer(curData(), arraySize);
      break;
    case sizeof(short): /* Shorts, . */ 
      if (inplace) {
        /* The copy is in place, just need to rearrange. */
        register unsigned short *datas = (unsigned short*) curData();
        for(i=0; i<arraySize; i++) {
          *datas = (short)(*datas << 8) | (short)(*datas >> 8);
        }
        advanceBuffer(arraySize*sizeof(short));
      } else {
        short* datas = (short*) curData();
        for(i=0;i < arraySize;i++) {
          bytesToShort((unsigned char*) datas++);
        }
      }
      break;
    case sizeof(int): 
      if (inplace) {
        /* The copy is in place, just need to rearrange. */
        register unsigned int *datai = (unsigned int *) curData();
        for(i=0; i<arraySize; i++) {
          *datai = (((*datai & 0x000000FF) << 24) |
                    ((*datai >> 24 ) & 0x000000FF) |
                    ((*datai & 0x0000FF00) << 8) |
                    ((*datai >> 8 ) & 0x0000FF00));
        }
        advanceBuffer(arraySize*sizeof(int));
      } else {
        register int *datai = (int*) curData();
        for(i=0;i < arraySize;i++) 
          bytesToInt((unsigned char*) datai++);
      }
      break;
    case sizeof(double): 
      if (inplace) {
        /* The copy is in place, just need to rearrange. */
        register unsigned int *datad = (unsigned int *) curData();
        for(i=0; i<2*arraySize; i++) {
          *datad = (((*datad & 0x000000FF) << 24) |
                    ((*datad >> 24 ) & 0x000000FF) |
                    ((*datad & 0x0000FF00) << 8) |
                    ((*datad >> 8 ) & 0x0000FF00));
        }
        advanceBuffer(arraySize*sizeof(double));
      } else {
        double *datad = (double*) curData();
        for(i=0;i < arraySize;i++) 
          bytesToDouble((unsigned char*) datad++);
      }
      break;
    default:
      fprintf(stderr, "Unhanded primitive element of size %d \n",
              fmt->dataSize());
      return false;
    }
  } else {
    // do things piece by piece as a last resort
    for(i=0;i < arraySize;i++) 
      if (!fmt->act(this))
        return false;
  }
  return true;
}

bool UnpackAction::actVarArray(StructSpec* parent,
                               const Vector<int>& indices,
                               FormatSpec* fmt)
{
  int arraySize;
  bytesToInt((unsigned char*) &arraySize);

  unsigned char* newStruct;
  bool inplace = fmt->isPackedType() && placeable(fmt, getIndex());
  if (!arraySize) {
    newStruct = NULL;
  } else {
    if (inplace)
      newStruct = ((arraySize == 0) ? 0L : _buffer+_buf_index);
    else
      newStruct = ((arraySize == 0) ? 0L :
                   new unsigned char[(arraySize*fmt->dataSize())]);
  }
  toData(&newStruct, sizeof(unsigned char*));

  if (inplace && inNativeFormat() || !arraySize) 
    advanceBuffer(arraySize*fmt->dataSize());
  else {
    UnpackAction subaction(_buffer, _buf_size, _buf_index,
                           newStruct, _align, _byte_order);
    subaction.inherit(*this);
    if (!subaction.transferArrayData(fmt, arraySize))
      return false;
    _buf_index = subaction._buf_index;
  }

  return DataAction::actVarArray(parent, indices, fmt);
}


///////////////////////////////////////////////////////////////////////////////
// UnpackSizeAction - takes a flat buffer of data with no alignment padding
//                    and figures how much of that data would be used
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(UnpackSizeAction);

int UnpackSizeAction::unpackSize(FormatSpec* spec, unsigned char* buffer,
                                 int byte_order)
{
  if (!spec || !buffer)
    return 0;

  if (spec->isPackedType()) {
    return spec->dataSize();
  } else {
    UnpackSizeAction action(buffer, 0, byte_order);
    if (!spec->act(&action))
      return 0;
    return action.getIndex();
  }
}

// Create action to size a buffer.  byte_order specifies the byte order of the
// data in buffer.
UnpackSizeAction::UnpackSizeAction(unsigned char* buffer, int buf_index,
                                   int byte_order)
{
  UTILS_BASE_CONSTRUCTOR(UnpackSizeAction);

  setBuffer(buffer, buf_index, byte_order);
}

void UnpackSizeAction::initClass()
{
  UTILS_BASE_INIT_CLASS(UnpackSizeAction, "UnpackSizeFormatAction",
                        "FormatAction");
}

void UnpackSizeAction::setBuffer(unsigned char* buffer, int buf_index,
                                 int byte_order)
{
  _buffer = buffer;
  _buf_index = _buf_index;
  _byte_order = byte_order;
}

// convenience method for unpacking an integer and returning it
int UnpackSizeAction::getInt()
{
  int res;
  bytesToInt((unsigned char*) &res);
  return res;
}

bool UnpackSizeAction::actLength(int len)
{
  advanceBuffer(len);
  return true;
}

bool UnpackSizeAction::actChar()
{
  advanceBuffer(sizeof(char));
  return true;
}

bool UnpackSizeAction::actShort()
{
  advanceBuffer(sizeof(short));
  return true;
}

bool UnpackSizeAction::actLong()
{
  advanceBuffer(sizeof(long));
  return true;
}

bool UnpackSizeAction::actDouble()
{
  advanceBuffer(sizeof(double));
  return true;
}

bool UnpackSizeAction::actFloat()
{
  advanceBuffer(sizeof(float));
  return true;
}

bool UnpackSizeAction::actInt()
{
  advanceBuffer(sizeof(int));
  return true;
}

bool UnpackSizeAction::actString()
{
  int length = strlen((char*) (_buffer + _buf_index)) + 1;
  advanceBuffer(length);
  return true;
}

bool UnpackSizeAction::actPtr(FormatSpec* fmt)
{
  char c;
  fromBuffer((unsigned char*) &c, sizeof(char));
  if (c != '\0') {
    if (fmt->isPackedType()) {
      advanceBuffer(fmt->dataSize());
    } else {
      UnpackSizeAction subaction(_buffer, _buf_index, _byte_order);
      subaction.inherit(*this);
      if (!fmt->act(&subaction))
        return false;
      _buf_index = subaction._buf_index;
    }
  }

  return true;
}

// calculate the buffer space needed for a fixed array
bool UnpackSizeAction::actFixedArray(const Vector<int>& sizes,
                                     FormatSpec* fmt)
{
  int arraySize = fixedArraySize(sizes);
  if (fmt->isPackedType()) {
    // if we're lucky, we can just copy the data
    advanceBuffer(fmt->dataSize()*arraySize);
  } else {
    // no, we have to go through piece by piece
    for(int i=0;i < arraySize;i++) 
      if (!fmt->act(this))
        return false;
  }
  return true;
}

bool UnpackSizeAction::actVarArray(StructSpec* parent,
                                   const Vector<int>& indices,
                                   FormatSpec* fmt)
{
  int arraySize;
  bytesToInt((unsigned char*) &arraySize);

  if (fmt->isPackedType()) {
    advanceBuffer(arraySize*fmt->dataSize());
  } else {
    // no, we have to go through piece by piece
    for(int i=0;i < arraySize;i++) 
      if (!fmt->act(this))
        return false;
  }

  return true;
}


///////////////////////////////////////////////////////////////////////////////
// PrintFormatAction - prints formatter in string form to an output
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(PrintFormatAction);

// default constructor to build a format printer
PrintFormatAction::PrintFormatAction() : _output(*fb_std_out)
{
  UTILS_BASE_CONSTRUCTOR(PrintFormatAction);
}

// constructor to build a format printer which will output to output
PrintFormatAction::PrintFormatAction(Output& output) : _output(output)
{
  UTILS_BASE_CONSTRUCTOR(PrintFormatAction);
}

void PrintFormatAction::initClass()
{
  UTILS_BASE_INIT_CLASS(PrintFormatAction, "PrintFormatAction",
                        "FormatAction");
  if (!fb_std_out)
    fb_std_out = new Output();
}

// most simple actions just print the action tag
bool PrintFormatAction::actInt()
{
  _output.write("int");
  return true;
}

bool PrintFormatAction::actShort()
{
  _output.write("short");
  return true;
}

bool PrintFormatAction::actLong()
{
  _output.write("long");
  return true;
}

bool PrintFormatAction::actFloat()
{
  _output.write("float");
  return true;
}

bool PrintFormatAction::actDouble()
{
  _output.write("double");
  return true;
}

bool PrintFormatAction::actChar()
{
  _output.write("char");
  return true;
}

bool PrintFormatAction::actString()
{
  _output.write("string");
  return true;
}

bool PrintFormatAction::actLength(int size)
{
  _output.write(size);
  return true;
}

bool PrintFormatAction::actPtr(FormatSpec* fmt)
{
  _output.write('*');
  return fmt->act(this);
}

bool PrintFormatAction::actSelfPtr(StructSpec*)
{
  _output.write("*!");
  return true;
}

bool PrintFormatAction::actStructElem(int i, int size,
                                      FormatSpec** elems)
{
  _output.write(' ');
  if (!elems[i]->act(this))
    return false;
  if (i != size-1)
    _output.write(',');
  return true;
}

bool PrintFormatAction::startStruct()
{
  _output.write('{');
  return true;
}

bool PrintFormatAction::endStruct()
{
  _output.write(" }");
  return true;
}

bool PrintFormatAction::actFixedArray(const Vector<int>& sizes,
                                      FormatSpec* fmt)
{
  if (sizes.numElems() < 1)
    return false;

  _output.write("[ ");
  if (!fmt->act(this))
    return false;
  _output.write(" : ");

  for (int i=0;i<sizes.numElems()-1;i++) {
    _output.write(sizes[i]);
    _output.write(", ");
  }
  _output.write(sizes[sizes.numElems()-1]);
  _output.write(" ]");
  return true;
}

bool PrintFormatAction::actVarArray(StructSpec*,
                                    const Vector<int>& indices,
                                    FormatSpec* fmt)
{
  if (indices.numElems() < 1)
    return false;

  _output.write("< ");
  if (!fmt->act(this))
    return false;
  _output.write(" : ");
    
  for (int i=0;i<indices.numElems()-1;i++) {
    _output.write(indices[i]);
    _output.write(", ");
  }
  _output.write(indices[indices.numElems()-1]);
  _output.write(" >");

  return true;
}

// static convenience routine for printing a format to an output
bool PrintFormatAction::print(FormatSpec* fmt, Output& output)
{
  if (!fmt) {
    return false;
  } else {
    PrintFormatAction act(output);
    return fmt->act(&act);
  }
}

// static convenience routine for printing a format to a string and returning
// it.  The result must be deleted
char* PrintFormatAction::printToString(FormatSpec* fmt)
{
  char* res = new char[20];
  Output output;
  output.setBuffer(res, 20, Output::standardResize);
  if (!print(fmt, output)) {
    delete res;
    return (char*) NULL;
  }
  int size;
  void* out_buf;
  output.write('\0');
  output.getBuffer(out_buf, size);
  return (char*) out_buf;
}


///////////////////////////////////////////////////////////////////////////////
// FormatEqualsAction - checks if two format specifiers are equal
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(FormatEqualsAction);

// Construct an action which will check if one specifier equals other
FormatEqualsAction::FormatEqualsAction(FormatSpec* other) 
{
  UTILS_BASE_CONSTRUCTOR(FormatEqualsAction);

  reset(other);
}

bool FormatEqualsAction::equals(FormatSpec* one, FormatSpec* other)
{
  if (one == other)
    return true;
  if (!one || !other)
    return false;
    
  FormatEqualsAction action(other);
  if (!one->act(&action))
    return false;

  return action.result();
}
    
void FormatEqualsAction::initClass()
{
  UTILS_BASE_INIT_CLASS(FormatEqualsAction, "FormatEqualsAction",
                        "FormatAction");
}

// in most simple cases, we can just check the other specifier type
bool FormatEqualsAction::actInt()
{
  _res = _other->isType(IntSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actShort()
{
  _res = _other->isType(ShortSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actLong()
{
  _res = _other->isType(LongSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actFloat()
{
  _res = _other->isType(FloatSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actDouble()
{
  _res = _other->isType(DoubleSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actChar()
{
  _res = _other->isType(CharSpec::getClassTypeId());
  return true;
}

bool FormatEqualsAction::actString()
{
  _res = _other->isType(StringSpec::getClassTypeId());
  return true;
}

// here we check the type, and the length
bool FormatEqualsAction::actLength(int size)
{
  _res = _other->isType(LengthSpec::getClassTypeId());
  if (!_res)
    return true;

  _res = (size == ((LengthSpec*) _other)->getLength());
  return true;
}

// for a pointer, we check the type, and the format type
bool FormatEqualsAction::actPtr(FormatSpec* fmt)
{
  _res = _other->isType(PtrSpec::getClassTypeId());
  if (!_res)
    return true;

  FormatEqualsAction sub_action(((PtrSpec*) _other)->getPointerFormat());
  if (!fmt->act(&sub_action))
    return false;

  _res = sub_action._res;
  return true;
}

// self pointer types just check type
bool FormatEqualsAction::actSelfPtr(StructSpec*)
{
  _res = _other->isType(SelfPtrSpec::getClassTypeId());
  return true;
}

// check the type and all the structure elements
bool FormatEqualsAction::actStruct(int size, FormatSpec** elems)
{
  _res = _other->isType(StructSpec::getClassTypeId());
  if (!_res)
    return true;

  StructSpec* other_struct = (StructSpec*) _other;
  if (other_struct->getSize() != size) {
    _res = false;
    return true;
  }

  if (!startStruct())
    return false;
  for (int i=0;i<size;i++) {
    FormatEqualsAction sub_action(other_struct->getFormat(i));
    if (!elems[i])
      return false;
    if (!elems[i]->act(&sub_action))
      return false;
    _res = sub_action._res;
    if (!_res)
      return true;
  }
  return endStruct();
}

// check the type, the sizes of the array, and the element type
bool FormatEqualsAction::actFixedArray(const Vector<int>& sizes,
                                       FormatSpec* fmt)
{
  _res = _other->isType(FixedArraySpec::getClassTypeId());
  if (!_res)
    return true;

  FixedArraySpec* other_array = (FixedArraySpec*) _other;
  if (sizes.numElems() != other_array->getSizes()->numElems()) {
    _res = false;
    return true;
  }

  for (int i=0;i<sizes.numElems();i++)
    if (sizes[i] != other_array->getSizes()->getValue(i)) {
      _res = false;
      return true;
    }

  FormatEqualsAction sub_action(other_array->getSubFormat());
  if (!fmt->act(&sub_action))
    return false;
  _res = sub_action._res;
  return true;
}

// check the type, the indices of the array, and the array type
bool FormatEqualsAction::actVarArray(StructSpec*,
                                     const Vector<int>& indices,
                                     FormatSpec* fmt)
{
  _res = _other->isType(VarArraySpec::getClassTypeId());
  if (!_res)
    return true;

  VarArraySpec* other_array = (VarArraySpec*) _other;
  if (indices.numElems() != other_array->getIndices()->numElems()) {
    _res = false;
    return true;
  }

  for (int i=0;i<indices.numElems();i++)
    if (indices[i] != other_array->getIndices()->getValue(i)) {
      _res = false;
      return true;
    }

  FormatEqualsAction sub_action(other_array->getSubFormat());
  if (!fmt->act(&sub_action))
    return false;
  _res = sub_action._res;
  return true;
}


///////////////////////////////////////////////////////////////////////////////
//   EqualsAction - checks if two formatted data structures are equal
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(EqualsAction);

// construct an action to check if the formatted data structure in data equals
// the formatted data structure in other
EqualsAction::EqualsAction(void* data, void* other)
  : DataAction(data), _other(other)
{
  UTILS_BASE_CONSTRUCTOR(EqualsAction);

  _result = true;  // assume equal by default
}

void EqualsAction::initClass()
{
  UTILS_BASE_INIT_CLASS(EqualsAction, "EqualsFormatAction",
                        "DataFormatAction");
}

// static convenience method to see if the formatted data structure in data
// equals the formatted data structure in other, as specified by the formatter
// spec.
bool EqualsAction::equals(FormatSpec* spec, void* data, void* other)
{
  if (!spec)
    return false;

  if (data == other)  // pointer equality is data equality
    return true;

  if (spec->isPackedType()) // pre-packed data can be quickly checked
    return (memcmp((char*) data, (char*) other, spec->dataSize()) == 0);
  else {
    // note that we have to check flat, but non-prepacked data because
    // alignment padding may have arbitrary content
    EqualsAction action(data, other);
    if (!spec->act(&action))
      return false;
    return action.result();
  }
}

// accumulate the result of checking if len bytes of the current data and
// the other data are equal.  Advance both data pointers by len.
bool EqualsAction::checkBytes(int len)
{
  if (!getData() || !_other.getData())
    return false;
  _result = (memcmp((char*) curData(), (char*) _other.curData(), len) == 0);
  return DataAction::actLength(len) && _other.actLength(len);
}

// string data must be checked
bool EqualsAction::actString()
{
  char* str1 = (char*) getSubData();
  char* str2 = (char*) _other.getSubData();
  _result = ((str1 == str2) || (str1 && str2 && !strcmp(str1, str2)));

  return DataAction::actString() && _other.actString();
}

// pointer data must be checked
bool EqualsAction::actPtr(FormatSpec* fmt)
{
  void* data = getSubData();
  void* other = _other.getSubData();
  if (data == other)
    return true;

  EqualsAction sub_action(data, other);
  return fmt->act(&sub_action) &&
    DataAction::actPtr(fmt) &&
    _other.actPtr(fmt);
}

// we must look at all of a fixed array
bool EqualsAction::actFixedArray(const Vector<int>& sizes,
                                 FormatSpec* fmt)
{
  if (fmt->isPackedType()) {  // packed types can quickly be checked
    _result = (memcmp((char*) curData(), (char*) _other.curData(),
                    fmt->dataSize()*fixedArraySize(sizes)) == 0);
    return DataAction::actFixedArray(sizes, fmt) &&
      _other.actFixedArray(sizes, fmt);
  } else {  // otherwise we have to look at each element inidivually
    int size = fixedArraySize(sizes);
    int i;
    for (i=0;i<size;i++) {
      if (!fmt->act(this))
        return false;
      if (!_result)
        return true;
    }
    return true;
  }
}

bool EqualsAction::actVarArray(StructSpec* parent,
                               const Vector<int>& indices,
                               FormatSpec* fmt)
{
  int size = varArraySize(parent, indices);
  unsigned char* data = getSubData();
  unsigned char* other = _other.getSubData();
  if (fmt->isPackedType()) // packed types can be checked all at once
    _result = (memcmp((char*)data, (char*)other, fmt->dataSize()*size) == 0);
  else {  // otherwise we must look at each one individually
    EqualsAction sub_action(data, other);
    for (int i = 0; i < size; i++) {
      if (!fmt->act(&sub_action))
        return false;
      if (!_result)  // abort check if result is false
        return true;
    }
  }
  return DataAction::actVarArray(parent, indices, fmt) &&
    _other.actVarArray(parent, indices, fmt);
}

bool EqualsAction::actStruct(int size, FormatSpec** elems)
{
  if (!startStruct())
    return false;

  // check each element
  for (int i=0;i<size;i++) {
    if (!actStructElem(i, size, elems))
      return false;
    if (!_result)  // abort check if result is false
      return true;
    _other.setIndex(alignField(i, size, elems));
  }

  return endStruct();
}

    
///////////////////////////////////////////////////////////////////////////////
// CopyAction - makes a copy of a formatted data structure
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(CopyAction);

// Action to take formatted data structure data and copy it to dest
CopyAction::CopyAction(void* data, void* dest)
  : DataAction(data), _dest(dest)
{
  UTILS_BASE_CONSTRUCTOR(CopyAction);
}

void CopyAction::initClass()
{
  UTILS_BASE_INIT_CLASS(CopyAction, "CopyFormatAction", "DataFormatAction");
}

// static convenience method for copying formatted data structure data to
// dest with the format structure specified by spec
bool CopyAction::copy(FormatSpec* spec, void* data, void* dest)
{
  if (!spec || (data == dest))
    return false;

  if (spec->isFlatType()) // flat things are easy
    memcpy((char*) dest, (char*) data, spec->dataSize());
  else {  
    CopyAction action(data, dest);
    if (!spec->act(&action))
      return false;
  }
  return true;
}

// static convenience method for copying formatted data structure data to
// a new destination with the format structure specified by spec
void* CopyAction::copy(FormatSpec* spec, void* data)
{
  if (!spec || !data)
    return NULL;

  unsigned char* res = new unsigned char[spec->dataSize()];
  if (!copy(spec, data, res)) {
    delete res;
    return NULL;
  }

  return res;
}

// copy len bytes from data to the destination without advancing any indices
void CopyAction::toDest(void* data, int len)
{
  memcpy((char*) _dest.curData(), (char*) data, len);
}

// copy len bytes from the current data to the current destination and advance
// both indices appropriately
bool CopyAction::copyBytes(int len)
{
  memcpy((char*) _dest.curData(), (char*) curData(), len);
  _dest.actLength(len);
  return DataAction::actLength(len);
}

// advance both the current data pointer and the destination pointer by len
void CopyAction::advance(int len)
{
  addSize(len);
  _dest.addSize(len);
}

// copy a string
bool CopyAction::actString()
{
  char* old_str = (char*) getSubData();
  char* str;
  if (!old_str)
    str = (char*) NULL;
  else {
    int length = strlen(old_str)+1;
    str = new char[length];
    memcpy(str, old_str, length);
  }

  toDest(&str, sizeof(char*));
  return DataAction::actString() && _dest.actString();
}

// copy a pointer
bool CopyAction::actPtr(FormatSpec* fmt)
{
  unsigned char* newStruct;
  void* data = getSubData();
  if (!data)
    newStruct = (unsigned char*) NULL;
  else {
    newStruct = new unsigned char[fmt->dataSize()];
    CopyAction subaction(data, newStruct);
    subaction.inherit(*this);
    if (!fmt->act(&subaction))
      return false;
  }

  toDest(&newStruct, sizeof(unsigned char*));
  return DataAction::actPtr(fmt) && _dest.actPtr(fmt);
}

// copy a fixed array
bool CopyAction::actFixedArray(const Vector<int>& sizes, FormatSpec* fmt)
{
  if (fmt->isFlatType()) {  // flat things can be done all at once
    toDest(curData(), fmt->dataSize()*fixedArraySize(sizes));
    return DataAction::actFixedArray(sizes, fmt) &&
      _dest.actFixedArray(sizes, fmt);
  } else {  // non flat things only one at a time
    int size = fixedArraySize(sizes);
    int i;
    for (i=0;i<size;i++) 
      if (!fmt->act(this))
        return false;
    return true;
  }
}

// copy a variable sized array
bool CopyAction::actVarArray(StructSpec* parent,
                             const Vector<int>& indices, FormatSpec* fmt)
{
  int size = varArraySize(parent, indices);
  unsigned char* data = getSubData();
  unsigned char* dest = new unsigned char[fmt->dataSize()*size];
  if (fmt->isFlatType())  // flat arrays can be done all at once
    memcpy((char*) dest, (char*) data, fmt->dataSize()*size);
  else {  // non-flat arrays need to be done one at a time
    CopyAction sub_action(data, dest);
    for (int i = 0; i < size; i++) 
      if (!fmt->act(&sub_action))
        return false;
  }

  toDest(&dest, sizeof(unsigned char*));
  return DataAction::actVarArray(parent, indices, fmt) &&
    _dest.actVarArray(parent, indices, fmt);
}

// every structure element needs to align the destination index as well as
// the normal data index alignment done by the superclass
bool CopyAction::actStructElem(int i, int size, FormatSpec** elems)
{
  if (!DataAction::actStructElem(i, size, elems))
    return false;
  _dest.setIndex(alignField(i, size, elems));
  return true;
}


///////////////////////////////////////////////////////////////////////////////
// CloneAction - makes a clone of a formatted data structure
///////////////////////////////////////////////////////////////////////////////

UTILS_BASE_SOURCE(CloneAction);

// Action to take formatted data structure data and clone it to dest
CloneAction::CloneAction(void* data, void* dest)
  : CopyAction(data, dest)
{
  UTILS_BASE_CONSTRUCTOR(CloneAction);
}

void CloneAction::initClass()
{
  UTILS_BASE_INIT_CLASS(CloneAction, "CloneFormatAction", "CopyFormatAction");
}

// static convenience method for cloneing formatted data structure data to
// dest with the format structure specified by spec
bool CloneAction::clone(FormatSpec* spec, void* data, void* dest)
{
  if (!spec || (data == dest))
    return false;

  if (spec->isFlatType()) // flat things are easy
    memcpy((char*) dest, (char*) data, spec->dataSize());
  else {  
    CloneAction action(data, dest);
    if (!spec->act(&action))
      return false;
  }
  return true;
}

// initialize FoFormatAction and its subclasses
void FormatAction::initClasses()
{
  static int initialized = 0;
  if (initialized)
    return;
  initialized = 1;

  Name::initClass();
  Type::init();
  Base::initClass();

  FormatAction::initClass();
  FlatAction::initClass();
  RestrictAction::initClass();
  DataSizeAction::initClass();
  DataAction::initClass();
  DeleteAction::initClass();
  ReadAction::initClass();
  PrintAction::initClass();
  BufferSizeAction::initClass();
  PackAction::initClass();
  UnpackAction::initClass();
  UnpackSizeAction::initClass();
  FormatEqualsAction::initClass();
  EqualsAction::initClass();
  CopyAction::initClass();
  CloneAction::initClass();
  PrintFormatAction::initClass();
}

__UTILS_END_NAMESPACE
