///////////////////////////////////////////////////////////////////////////////
//
//                               CustomSpec.cc
//
// Implements classes to assist in creating custom format specifiers
//
// Classes implemented for export:
//     CustomSpec - a general framework for custom format specifiers
//     CustomPtrSpec - a subclass specifically for entities that are pointers
//     VerbatimSpec - a subclass for dealing with "verbatim" entries,
//                      essentially treated as strings except they are
//                      read and printed enclosed in brackets, not quotes,
//                      and there is no built in delete
//
// Class implemented for internal use
//     PrintNameAction - default action for printing a custom specifier name
//     FormatEqualsAction - default action for testing custom specifier
//                          format equality
//
///////////////////////////////////////////////////////////////////////////////

#include <utils/Dict.h>
#include <utils/Vector.h>
#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/String.h>
#include <utils/formatting/VerbatimSpec.h>
#include <utils/formatting/RestrictAction.h>
#include <utils/formatting/FlatAction.h>
#include <utils/formatting/DeleteAction.h>
#include <utils/formatting/PrintAction.h>
#include <utils/formatting/ReadAction.h>
#include <utils/formatting/PrintFormatAction.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/EqualsAction.h>
#include <utils/formatting/Format.h>

__UTILS_BEGIN_NAMESPACE

/////////////////////
// class CustomSpec
/////////////////////

// PrintNameAction - default action for printing a custom specifier name
class CustPrintNameAction : public ApplyFormatAction
{
  public:
    CustPrintNameAction(FormatSpec* spec) { _spec = spec; }

    // just prints the specifier name to the output
    virtual bool execute(FormatAction* action) {
        ((PrintFormatAction*) action)->getOutput().
            write(_spec->getName());
        return true;
    }

  private:
    FormatSpec* _spec;   // the custom specifier
};

// CustFormatEqualsAction - default action for testing custom specifier
//                      format equality
class CustFormatEqualsAction : public ApplyFormatAction
{
  public:
    CustFormatEqualsAction(FormatSpec* spec) { _spec = spec; }

    // just tests to see if the two specifiers are of the same type
    virtual bool execute(FormatAction* action) {
        FormatEqualsAction* ea = (FormatEqualsAction*) action;
        if (ea->getOther()->getClassTypeId() != _spec->getClassTypeId())
            ea->indicateInequality();
        return true;
    }

  private:
    FormatSpec* _spec;   // the custom specifier
};

UTILS_BASE_SOURCE(CustomSpec);

void CustomSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(CustomSpec, "CustomSpec", "FormatSpec");
}

CustomSpec::CustomSpec()
{
    UTILS_BASE_CONSTRUCTOR(CustomSpec);

    // create the action dictionary
    _actions = new Dict<ApplyFormatAction*>(20);

    addFormatAction(PrintFormatAction::getClassTypeId(),
                    new CustPrintNameAction(this));
    addFormatAction(FormatEqualsAction::getClassTypeId(),
                    new CustFormatEqualsAction(this));
}

// used in deleting a CustomSpec action
static void unref_action(unsigned long, ApplyFormatAction* action)
{
    action->unref();
}

// unref all the actions
CustomSpec::~CustomSpec()
{
    _actions->applyToAll(unref_action);
    delete _actions;
}

// add a format action.  When the CustomSpec is actd upon by something of
// type action_type, execute imp.  Return true for success
bool CustomSpec::addFormatAction(Type action_type,
                                     ApplyFormatAction* imp)
{
    // check for bad types (usually caused by ommision from Init.cc)
    if (action_type.isBad()) {
        printf("Warning: adding bad format action\n");
        imp->ref();
        imp->unref();
        return false;
    }

    // if the action type is an OK type
    if (action_type.isDerivedFrom(FormatAction::getClassTypeId())) {
        ApplyFormatAction* old = (ApplyFormatAction*) NULL;
        // see if we already respond to that action
        if (_actions->find(action_type.getKey(), old)) {
            // yes? remove old response from table
            if (old != imp) {
                old->unref();
                _actions->remove(action_type.getKey());
            } else
                return true;
        }
        // add new response into table
        if (!_actions->enter(action_type.getKey(), imp)) 
            printf("Big problems re-entering an action\n");
        else {
            imp->ref();
            return true;
        }
    }

    // error occured
    imp->ref();
    imp->unref();
    return false;
}

// convenience routine to add an action by function address
bool CustomSpec::addFormatRoutine(Type action_type,
                                      bool (*func)(FormatAction*),
                                      bool external)
{
    ApplyFormatAction* imp = new ApplyFormatRoutine(func, external);
    return addFormatAction(action_type, imp);
}

// apply action to the custom element using the dictionary.  If there is
// an entry in the dictionary, apply it and return true, else return false
bool CustomSpec::applyAction(FormatAction* action)
{
    ApplyFormatAction* imp;
    Type type = action->getTypeId();
    if (_actions->find(type.getKey(), imp)) {
        return imp->execute(action);
    } else {
        printf("Warning: No custom specification for %s\n",
               type.getName().getString());
    }
    return false;
}

// returns true if the action can be applied to this custom spec.
bool CustomSpec::canApplyAction(FormatAction* action)
{
    ApplyFormatAction* imp;
    Type type = action->getTypeId();
    if (_actions->find(type.getKey(), imp)) {
        return imp != NULL;
    } else {
        return false;
    }
}

// the virtual act function, which just applies the appropriate action, if 
// there is one
bool CustomSpec::act(FormatAction* action)
{
    return applyAction(action);
}

// static helper function used by CustomSpec::copyExternals
static void copy_external(unsigned long key,
                          ApplyFormatAction* act, void* cb)
{
    if (act->isExternal()) 
        ((CustomSpec*) cb)->addFormatAction(Type((short) key), act);
}

// copy all of the actions marked "external" to other
void CustomSpec::copyExternalsTo(CustomSpec* other)
{
    _actions->applyToAll(copy_external, (void*) other);
}


///////////////////
// CustomPtrSpec
///////////////////

UTILS_BASE_SOURCE(CustomPtrSpec);

void CustomPtrSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(CustomPtrSpec, "CustomPtrSpec", "CustomSpec");
}

// all pointers are restricted in the same way
#if ((ALIGN & ALIGN_LONGEST) || (ALIGN & ALIGN_INT))
static bool restrict_ptr(FormatAction* act)
{
    ((RestrictAction*) act)->restrict(sizeof(void*));
    return true;
}
#endif

// all pointers are not flat
static bool flat_ptr(FormatAction* act)
{
    ((FlatAction*) act)->setFlat(false);
    return true;
}

// all pointers have the same data size
static bool data_size_ptr(FormatAction* act)
{
    ((DataSizeAction*) act)->addSize(sizeof(void*));
    ((DataSizeAction*) act)->needsPacking();
    return true;
}

// generic equality test for pointers
static bool equals_ptr(FormatAction* act)
{
    EqualsAction* eq_act = (EqualsAction*) act;
    eq_act->setResult(eq_act->getSubData() == eq_act->getOther().getSubData());
    return true;
}

CustomPtrSpec::CustomPtrSpec()
{
    UTILS_BASE_CONSTRUCTOR(CustomPtrSpec);

#if ((ALIGN & ALIGN_LONGEST) || (ALIGN & ALIGN_INT))
    addFormatRoutine(RestrictAction::getClassTypeId(), restrict_ptr);
#endif
    addFormatRoutine(FlatAction::getClassTypeId(), flat_ptr);
    addFormatRoutine(DataSizeAction::getClassTypeId(), data_size_ptr);
    addFormatRoutine(EqualsAction::getClassTypeId(), equals_ptr);
}


///////////////////////
// class VerbatimSpec
///////////////////////

// verbatims print enclosed by brackets
static bool print_verbatim(FormatAction* action)
{
    PrintAction* paction = (PrintAction*) action;
    Output& output = paction->getOutput();
    output.write('{');
    output.write((char*) paction->getSubData());
    output.write('}');
    paction->addSize(sizeof(char*));
    return true;
}

// verbatims are read enclosed by brackets
static bool read_verbatim(FormatAction* action)
{
    Vector<char> buffer(100);
    ReadAction* raction = (ReadAction*) action;
    Input& input = raction->getInput();

    char c;
    if (!input.read(c) || c!= '{') {
        printf("Error reading verbatim left brace\n");
        return false;
    }

    int level = 0;
    while (1) {
        if (!input.get(c)) {
            printf("Error reading verbatim\n");
            return false;
        }

        if (c == '{')
            level++;
        else if (c == '}')
            level--;

        if (level < 0)
            break;
        
        buffer.append(c);
    }

    char* data = new char[buffer.numElems()+1];
    memcpy(data, buffer.getData(), buffer.numElems());
    data[buffer.numElems()] = '\0';
    raction->putData(&data, sizeof(char*));
    raction->addSize(sizeof(char*));

    return true;
}

UTILS_BASE_SOURCE(VerbatimSpec);

void VerbatimSpec::initClass()
{
    UTILS_BASE_INIT_CLASS(VerbatimSpec, "VerbatimSpec", "CustomPtrSpec");
}

VerbatimSpec::VerbatimSpec()
{
    UTILS_BASE_CONSTRUCTOR(VerbatimSpec);

    addFormatRoutine(PrintAction::getClassTypeId(), print_verbatim);
    addFormatRoutine(ReadAction::getClassTypeId(),  read_verbatim);
}

bool VerbatimSpec::applyAction(FormatAction* action)
{
    if (CustomPtrSpec::applyAction(action))
        return true;

    // if nothing in the verbatim spec table responds, treat this verbatim
    // as a string
    StringSpec spec;
    spec.act(action);

    return true;
}

__UTILS_END_NAMESPACE
