/***************************************************************************

                                ConfigFile.cc

  Implements a class which reads a hierarchical, typed parameter file and
  can be queried for the values set in that parameter file.  In addition,
  the user can set the values after the parameter file has been read.

  ConfigFile recognizes a variety of data types:
    int, float, double, char, string, and bool
  are atomic types.  In addition there are complex primitives,
    struct, structArray
  Also, users can use ConfigFile::registerElem to define new primitive types, 
    such as the "spec" primitive seen in Generator.cc

  Each type corresponds to a subclass of ConfigElem which processes that type.

  The configuration files themselves have a fairly flexible syntax.  

    <type decl> <name>[num_elems] = <list of types of length num elem>;

  If <type decl> is ommitted, everything after the equal sign up to the 
  first semicolon not inside a pair of brackets is stored in an UnknownElem,
  which will resolve to a type upon user querying the ConfigFile.

  If [num_elems] is present the list of data must be of length num elem,
  otherwise the list can be of any length:  num_elems is purely for redundancy.

  The atomic types are fairly straightforward, except for strings. A string
  is either a white-space delimited word, or enclosed in quotes or enclosed
  in matching curly brackets.  For example, the following is a valid string
  declaration
    string str_example = first "second string" { third { string { here }}};

  Structures can be declared using the "struct" type, and have a slightly
  different syntax:

    struct name {
       int example = 1;
       <other declarations>
    }

  Note no trailing semicolon is necessary, but stylistically is probably
  a good idea (and certainly won't cause any harm).

  Just for grins, structure values can be added and reset after the structure
  is declared.  For example, a config file which contained the above structure
  could have later in the file a statement

     name.example = 2;

  which would override the original definition of name.  You could also do

     int name.new_thing = 1;

  Which will add the integer element "new_thing" to the structure name.  If
  name does not exist, a structure element will be created.

  You can declare a named array of structures with the structArray type:

     structArray array {
       {
         int thing=1;
         < struct 1 declarations >
       }
       .
       .
       {
        < struct n declarations >
       }
     }

  A wrinkle is that you can refer later to individual elements of a structure
  array:

     array[0].thing = 2;

  will reset "thing" in the first element of array.  You cannot do this
  with arbitrary types, i.e., setting the nth element of an array of ints.

  The comment character in the parameter files is '#' (inherited from Input)

  The procedure for using a ConfigFile is fairly straightforward.  Use
  a default constructor to create a ConfigFile and use the open method
  to parse a file or the parse method to parse a string.  Once the data is
  parsed use the methods such as getInt, getBool, etc. to get data by name.
  Data in structures can be referenced with "."s as in the above example where
  the element example of structure name was referenced with name.example
  The get* functions will do their best to convert data from the type specified
  in the parameter file to the requested type (and for UnknownElem's with
  no specified type in the file will parse on the fly, even structures)

  That's the general gist, but many other useful things can be learned about
  ConfigFile by examining the header file and this source code.

  Classes implemented for export:
    ConfigFile - the configuration file class
    ConfigElem - the basic unit of a configuration file
    PrimElem - the basic primitive element class, super class of IntElem etc.
    StructElem - the compositional configuration element
    UnknownElem - holder for untyped data terminated by ; to be parsed later
    StructArrayElem - holds an array of StructElem's

  Classes implemented for internal use:
    RootElem - used as top level struct for ConfigFile cause it prints nice
    IntElem - holds an array of ints
    FloatElem - holds an array of floats
    DoubleElem - holds an array of doubles
    StringElem - holds an array of strings
    BoolElem - holds an array of boolean values
    CharElem - holds an array of characters
    ConfigFileInitializer - used as C++ trickery to call ConfigFile::init
      upon creation of first ConfigFile

***************************************************************************/

#include <stdlib.h>
#include <math.h>

#include <utils/ConfigFile.h>
#include <utils/ConfigElem.h>
#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/Dict.h>
#include <utils/StructElem.h>
#include <utils/UnknownElem.h>
#include <utils/ConfigWatch.h>

#include "ConfigParser.h"

__UTILS_BEGIN_NAMESPACE

// verbosity level.  Reset to result of looking at the CONFIG_VERBOSE
// level, i.e., 0 for none, 1, for just include files, 2 for debugging
// getting values
static int _Verbosity = -1;

// RootElem - used as top level struct for ConfigFile cause it prints nice
class RootElem : public StructElem {
public:    
  virtual void writeData(Output& output);
};

UTILS_BASE_SOURCE(StructArrayElem);

StructArrayElem::StructArrayElem() : _array(5)
{
  UTILS_BASE_CONSTRUCTOR(StructArrayElem);
}

void StructArrayElem::initClass()
{
  UTILS_BASE_INIT_CLASS(StructArrayElem, "structArray", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// IntElem - holds an array of ints
class IntElem : public PrimElem
{
  UTILS_BASE_HEADER(IntElem);
public:
  IntElem();

  virtual void clear() { _data.clear(); setNumValues(0); }
  virtual bool parsePrim(const char* val);
  virtual void writePrim(Output& output, int index);
  virtual bool setPrim(Type type, void* data, int index);
  virtual int getValue(Type dest_type, void* data, int max_num=1) const;
  virtual bool copy(ConfigElem* src);
  virtual unsigned char* getData() {
    return (unsigned char*) _data.getData();
  }

  static void initClass();

private:
  Vector<int> _data;   // the data
};

UTILS_BASE_SOURCE(IntElem);

IntElem::IntElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(IntElem);
}

void IntElem::initClass()
{
  UTILS_BASE_INIT_CLASS(IntElem, "int", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// FloatElem - holds an array of floats
class FloatElem : public PrimElem
{
  UTILS_BASE_HEADER(FloatElem);
public:
  FloatElem();

  virtual void clear() { _data.clear(); setNumValues(0); }
  virtual bool parsePrim(const char* val);
  virtual void writePrim(Output& output, int index);
  virtual bool setPrim(Type type, void* data, int index);
  virtual int getValue(Type dest_type, void* data, int max_num=1) const;
  virtual bool copy(ConfigElem* src);
  virtual unsigned char* getData() {
    return (unsigned char*) _data.getData();
  }

  static void initClass();

private:
  Vector<float> _data;    // the data
};

UTILS_BASE_SOURCE(FloatElem);

FloatElem::FloatElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(FloatElem);
}

void FloatElem::initClass()
{
  UTILS_BASE_INIT_CLASS(FloatElem, "float", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// DoubleElem - holds an array of doubles
class DoubleElem : public PrimElem
{
  UTILS_BASE_HEADER(DoubleElem);
public:
  DoubleElem();

  virtual void clear() { _data.clear(); setNumValues(0); }
  virtual bool parsePrim(const char* val);
  virtual void writePrim(Output& output, int index);
  virtual bool setPrim(Type type, void* data, int index);
  virtual int getValue(Type dest_type, void* data, int max_num=1) const;
  virtual bool copy(ConfigElem* src);
  virtual unsigned char* getData() {
    return (unsigned char*) _data.getData();
  }

  static void initClass();

private:
  Vector<double> _data;    // the data
};

UTILS_BASE_SOURCE(DoubleElem);

DoubleElem::DoubleElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(DoubleElem);
}

void DoubleElem::initClass()
{
  UTILS_BASE_INIT_CLASS(DoubleElem, "double", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// CharElem - holds an array of characters
class CharElem : public PrimElem
{
  UTILS_BASE_HEADER(CharElem);
public:
  CharElem();

  virtual void clear() { _data.clear(); setNumValues(0); }
  virtual bool parsePrim(const char* val);
  virtual void writePrim(Output& output, int index);
  virtual bool setPrim(Type type, void* data, int index);
  virtual int getValue(Type dest_type, void* data, int max_num=1) const;
  virtual bool copy(ConfigElem* src);
  virtual unsigned char* getData() {
    return (unsigned char*) _data.getData();
  }

  static void initClass();

private:
  Vector<char> _data;    // the data
};

UTILS_BASE_SOURCE(CharElem);

CharElem::CharElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(CharElem);
}

void CharElem::initClass()
{
  UTILS_BASE_INIT_CLASS(CharElem, "char", "ConfigElem");
}

UTILS_BASE_SOURCE(StringElem);

StringElem::StringElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(StringElem);
}

void StringElem::initClass()
{
  UTILS_BASE_INIT_CLASS(StringElem, "string", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// BoolElem - holds an array of boolean values
class BoolElem : public PrimElem
{
  UTILS_BASE_HEADER(BoolElem);
public:
  BoolElem();

  virtual void clear() { _data.clear(); setNumValues(0); }
  virtual bool parsePrim(const char* val);
  virtual void writePrim(Output& output, int index);
  virtual bool setPrim(Type type, void* data, int index);
  virtual int getValue(Type dest_type, void* data, int max_num=1) const;
  virtual bool copy(ConfigElem* src);
  virtual unsigned char* getData() {
    return (unsigned char*) _data.getData();
  }

  static void initClass();

private:
  Vector<bool> _data;    // the data
};

UTILS_BASE_SOURCE(BoolElem);

BoolElem::BoolElem() : _data(1)
{
  UTILS_BASE_CONSTRUCTOR(BoolElem);
}

void BoolElem::initClass()
{
  UTILS_BASE_INIT_CLASS(BoolElem, "bool", "ConfigElem");
}

///////////////////////////////////////////////////////////////////////////
// ConfigFileInitializer - used as C++ trickery to call ConfigFile::init
//  upon creation of first ConfigFile
class ConfigFileInitializer {
public:
  ConfigFileInitializer() { ConfigFile::init(); }
};


///////////////////////////////////////////////////////////////////////////
// ConfigElem implementation - most is abstract

UTILS_BASE_ABSTRACT_SOURCE(ConfigElem);

void ConfigElem::initClass()
{
  UTILS_BASE_INIT_ABSTRACT_CLASS(ConfigElem, "ConfigElem", "Base");
}

ConfigElem::ConfigElem()
{
  UTILS_BASE_CONSTRUCTOR(ConfigElem);
  _line_num = 0;
  _watcher = NULL;
}

ConfigElem::~ConfigElem()
{
  utils::Managed::unref(_watcher);
}

void ConfigElem::setWatcher(ConfigWatch* watcher)
{
  if (watcher == _watcher)
    return;
  if (watcher)
    watcher->ref();
  if (_watcher)
    _watcher->unref();
  _watcher = watcher;
}

void ConfigElem::invokeSetWatch()
{
  if (_watcher)
    _watcher->set(this);
}

// print an error message on parsing a file
void ConfigElem::input_error(const char* err_msg, Input& input)
{
  if (input.fromBuffer()) {
    printf("Config error in cached file at line %d: %s\n",
           input.getLineNum(), err_msg);
  } else {
    printf("Config error in %s at line %d: %s\n", input.getCurFileName(),
           input.getLineNum(), err_msg);
  }
}

// print an error in getting a value 
void ConfigElem::get_error(const char* msg, const ConfigElem* elem)
{
  printf("Config error in line %d: %s\n", elem->getLineNum(), msg);
}

ConfigElem* ConfigElem::clone() const
{
  ConfigElem* res = (ConfigElem*) getTypeId().createInstance();
  if (res)
    res->copy((ConfigElem*) this);
  return res;
}


///////////////////////////////////////////////////////////////////////////
// ConfigFile implementation

static bool _ConfigFileInitialized = false;

// create a config file based on the structure elem root.  If NULL, an 
// empty root will be created later
ConfigFile::ConfigFile(StructElem* root)
{
  _root = NULL;
  initRoot(root);
}

// Same as above, just try and typecheck the root to ensure it is a structure
// element
ConfigFile::ConfigFile(ConfigElem* root)
{
  _root = NULL;
  if (!root ||
      !root->getTypeId().isDerivedFrom(StructElem::getClassTypeId()))
    initRoot((StructElem*) NULL);
  else
    initRoot((StructElem*) root);
}

// delete the configuration file, i.e., close it
ConfigFile::~ConfigFile()
{
  close();
}

// return global verbosity level
int ConfigFile::verbosity()
{
  // if we haven't checked the CONFIG_VERBOSE variable
  if (_Verbosity == -1) {
    const char* verbosity = getenv("CONFIG_VERBOSE");
    if (!verbosity)
      _Verbosity = 0;
    else if (sscanf(verbosity, "%d", &_Verbosity) != 1) {
      _Verbosity = 1;
    }
  }

  return _Verbosity;
}

// setup the root of the configuration file, i.e. the top level structure
void ConfigFile::initRoot(StructElem* root)
{
  static ConfigFileInitializer initializer;

  if (!root)
    root = new RootElem();
  if (_root)
    _root->unref();
  _root = root;
  _root->ref();
}

// open a file named "name" and read parameters from it.  returns true
// for success and false for failure.
bool ConfigFile::open(const char* name)
{
  Input input;
  if (!input.openFile(name)) {
    printf("Cannot open config file %s\n", name);
    return false;
  }

  if (verbosity() > 0) {
    printf("ConfigFile:  Opening %s\n", input.getCurFileName());
  }

  bool res = read(input);
  input.closeFile();
  return res;
}


// release the root
void ConfigFile::close()
{
  if (_root)
    _root->unref();
  _root = (StructElem*) NULL;
}

void ConfigFile::clear()
{
  if (_root)
    _root->clear();
}

// parse parameter file from string "source" 
bool ConfigFile::parse(const char* source)
{
  Input input;
  input.setBuffer((void*) source, strlen(source));
  return read(input);
}

// Parse data from input.  If context is NULL, we are using root as the
// current structure.  If root doesn't exist, create it (as a RootElem so
// it prints nice).  If context is non-NULL, it is the context within which
// we read from input.  The function also initializes the ConfigFile class
// for reading if it hasn't been done before
bool ConfigFile::read(Input& input, StructElem* context)
{
  // get appropriate context
  if (!context) {
    if (!_root) {
      _root = new RootElem();
      _root->ref();
    }
    context = _root;
  }
  // make sure ConfigFile is initialized
  if (!_ConfigFileInitialized) {
    init();
  }

  ConfigParser p(input, _root);
  if (p.yyparse() == 0)
    return true;
  if (!p.errorPrinted) {
    printf("File %s, line %d, syntax error\n", input.getCurFileName(),
           input.getLineNum());
  }
  return false;
}

// Ouptut the configuration elements rooted at _root, if any
void ConfigFile::write(Output& output)
{
  if (_root)
    _root->writeData(output);
}

// if necessary, initialize the standard element types
void ConfigFile::init()
{
  if (_ConfigFileInitialized)
    return;
  _ConfigFileInitialized = true;

  Name::initClass();
  Type::init();
  Base::initClass();
  ConfigElem::initClass();
  StructElem::initClass();
  IntElem::initClass();
  FloatElem::initClass();
  DoubleElem::initClass();
  CharElem::initClass();
  StringElem::initClass();
  BoolElem::initClass();
  StructArrayElem::initClass();
}

static char* string_val(ConfigElem* elem) {
  utils::Output output;
  output.setBuffer(NULL, 0, utils::Output::standardResize);
  elem->writeData(output);
  output.write('\0');
  void* bufptr;
  int size;
  output.getBuffer(bufptr, size);

  // get rid of trailing semicolon if there
  char* val = (char*) bufptr;
  if (val[size-2] == ';')
    val[size-2] ='\0';

  return val;
}

// look up an element specified by "name".  "name" could be a single string
// indicating look up in the root context, or it could be a list of
// strings separated by .'s to specifying elements of named sub-structures
// the identifying string or strings could be preceded by a type name, such 
// as int or bool, to indicate what type to expect the data to be.
// If the specified element is not there, it is created and inserted
// with the default data given by "default_data" and "num_defaults" of the
// type with ID "type," and "data" is filled with the default data up to
// "max_num" number of elements, and the number of elements put into "data"
// is returned.  If the specified element is there, then the
// data is converted to the type with ID "type" and up to "max_num" elements
// of the data of that type are put into "data."  The number successfully
// converted is returned.  If the data can not be converted to "type", then
// 0 is returned.
int ConfigFile::get(Type type, const char* name, void* data, int max_num,
                    void* default_data, int num_defaults)

{
  if (!_root)  // no context -> no data
    return 0;

  // cache the verbosity
  int v = verbosity();
  bool using_default = false;
  
  // does the element already exist?
  ConfigElem* elem = lookup(name);
  if (!elem) {  // no element by that name exists
    if (default_data) {  // if we have default data
      // create a new element specified by "name"
      elem = _root->makeElem(name);
      if (!elem) {
        printf("Problem making default for %s\n", name);
        return 0;
      }
      // if no type was specified in name
      if (elem->getTypeId() == UnknownElem::getClassTypeId()) {
        // coerce it to be "type"
        elem->ref();
        elem->getValue(type, NULL, 0);
        elem->unref();
        elem = lookup(name);
      }
      // if the created type does not equal the requested type
      // we have a problem
      // NOTE:  in future maybe we could do some conversion rather
      // than producing an error
      if (!elem || elem->getTypeId() != type) {
        printf("Problem using default for %s\n", name);
        return 0;
      }
      // set the new element data to be the defaults
      elem->setValue(type, default_data, num_defaults);
    }
    using_default = true;
  }
  if (!elem) {  // if the element is still NULL, we have a problem
    if (v > 1) {
      printf("ConfigFile: get error looking up %s\n", name);
    }
    return 0;
  }

  elem->ref();  // reference it to be safe for unknown replacing
  // get the data, coercing to be type
  int res = elem->getValue(type, data, max_num);
  if (v > 1) {
    char* val = string_val(elem);
    if (using_default) {
      printf("ConfigFile: get default %s = %s\n", name, val);
    } else {
      printf("ConfigFile: get %s = %s\n", name, val);
    }
    delete [] val;
  }

  elem->unref();
  return res;  // and return the result
}

bool ConfigFile::getStruct(const char* name, utils::ConfigFile& params)
{
  ConfigElem* root = getStruct(name);
  if (!name)
    return false;
  params.initRoot((utils::StructElem*) root);
  return true;
}

// Convenience routine for getting an array of structures specified by "name"
// The results (up to length "max_num") are copied into data, and the number
// of pointers copied is returned.  Note no referencing is done.
int ConfigFile::getStructs(const char* name, ConfigElem** data, int max_num)
{
  return get(StructElem::getClassTypeId(), name, data, max_num);
}

// Convenience routine for getting a single string "name" with default def
// Copies the result (up to length max_len) into dest.
bool ConfigFile::getString(const char* name, const char* def, char* dest,
                           int max_len)
{
  const char* data = getString(name, def);
  if (!data) {
    *dest = '\0';
    return false;
  }

  int len = strlen(data)+1;
  if (len > max_len) {
    memcpy(dest, data, max_len-1);
    dest[max_len-1] = '\0';
    return true;
  }

  memcpy(dest, data, len);
  return true;
}

// Convenience routine for public access of the integer type ID
Type ConfigFile::intType()
{
  return IntElem::getClassTypeId();
}

// Convenience routine for public access of the float type ID
Type ConfigFile::floatType()
{
  return FloatElem::getClassTypeId();
}

// Convenience routine for public access of the double type ID
Type ConfigFile::doubleType()
{
  return DoubleElem::getClassTypeId();
}

// Convenience routine for public access of the character type ID
Type ConfigFile::charType()
{
  return CharElem::getClassTypeId();
}

// Convenience routine for public access of the boolean type ID
Type ConfigFile::boolType()
{
  return BoolElem::getClassTypeId();
}

// Convenience routine for public access of the string type ID
Type ConfigFile::stringType()
{
  return StringElem::getClassTypeId();
}

// set the element specified by "name" (which may be created) to the
// num_values stored in data.  If they are not of the correct type,
// bad things will happen silently, thus you should always use a type
// value with name so that it will not fail silently.
// If ignore_duplicates is false, then setting an element which already
// exists will print a warning and return NULL
// The element that is successfully set is returned, or NULL for an error.
ConfigElem* ConfigFile::set(Type type, const char* name, void* data,
                            int num_values, 
                            bool ignore_duplicates)
{
  ConfigElem* elem = _root->makeElem(name, true, ignore_duplicates);
  if (!elem)
    return (ConfigElem*) NULL;
  if (elem->getTypeId() == UnknownElem::getClassTypeId()) {
    // coerce it to be "type"
    elem->ref();
    elem->getValue(type, NULL, 0);
    elem->unref();
    elem = lookup(name);
  }
  elem->setValue(type, data, num_values);
  return elem;
}

ConfigElem* ConfigFile::set(const char* name, const char* value,
                            bool ignore_duplicates)
{
  ConfigElem* elem = _root->makeElem(name, ignore_duplicates, true);
  if (!elem)
    return (ConfigElem*) NULL;

  Input input;
  input.setBuffer((void*) value, strlen(value));
  if (!elem->readData(input))
    return NULL;

  return elem;
}

// Lookup the element specified by "name."  Do not create if not there.
ConfigElem* ConfigFile::lookup(const char* name)
{
  // true for ignore duplicates, false for don't create if not there
  return _root->makeElem(name, true, false);
}

// get the root element of the configuration file
ConfigElem* ConfigFile::getRoot() const
{
  return (ConfigElem*) _root;
}

// static function to make a copy of the configuration file src and return it
ConfigFile* ConfigFile::copy(ConfigFile* src)
{
  return new ConfigFile((StructElem*) src->_root->clone());
}

void ConfigFile::copy(const ConfigFile& src, ConfigFile& dest)
{
  dest.initRoot((StructElem*) src._root->clone());
}

int ConfigFile::numValues(const char* name)
{
  ConfigElem* elem = lookup(name);
  if (!elem)
    return 0;
  return elem->numValues();
}


///////////////////////////////////////////////////////////////////////////
// StructElem implementation

// StructElem destructor helper function 
static void unref_elem(const char*, ConfigElem* elem)
{
  elem->unref();
}

UTILS_BASE_SOURCE(StructElem);

void StructElem::initClass()
{
  UTILS_BASE_INIT_CLASS(StructElem, "struct", "ConfigElem");
}

StructElem::StructElem()  : _dict(30) {
  UTILS_BASE_CONSTRUCTOR(StructElem);
  _output_buf = 0L;
  _size_output = 0;
  _me = this;
} 

StructElem::~StructElem()
{
  clear();
}

void StructElem::clear()
{
  _dict.applyToAll(unref_elem);
  _dict.clear();
}

// coerce value into type "dest_type" and put up to max_num into data
// StructElems cannot be coerced into anything except StructElems
// StructElems are always singletons for the abstract getValue method
int StructElem::getValue(Type dest_type, void* data, int max_num) const
{
  if (dest_type == ConfigFile::stringType()) {
    // convert to a string via printing
    Output output;
    StructElem* elem = (StructElem*) this;
    if (!_output_buf) {
      elem->_size_output = 15;
      elem->_output_buf = new char[_size_output];
    }
    output.setBuffer(_output_buf, _size_output, Output::standardResize);
    elem->writeData(output);
    output.write('\0');
    int num_bytes;
    void* tmp;
    output.getBuffer(tmp, num_bytes);
    // cache results in output buffer
    elem->_output_buf = (char*) tmp;
    elem->_size_output = output.getBufferSize();
    const char** res = (const char**) data;
    *res = _output_buf;
  } else if (!dest_type.isDerivedFrom(getClassTypeId())) {
    get_error("Cannot get a non-struct from a struct", this);
    return 0;
  } else
    if (data)
      *((const StructElem**) data) = this;
  return 1;
}

bool StructElem::setValue(Type dest_type, void* data, int num_values)
{
  if (num_values != 1) {
    printf("StructElem:  cannot set with more than one value\n");
    return false;
  }
  if (!dest_type.isDerivedFrom(getClassTypeId())) {
    printf("StructElem:  cannot set with a non-structure\n");
    return false;
  }
  if (!data) {
    clear();
    return false;
  } else {
    if (copy(*(ConfigElem**) data)) {
      invokeSetWatch();
      return true;
    } else
      return false;
  }
}

// Translates "spec" into an element of this structure.  If ignore_duplicates
// is false, then print an error if "spec" refers to an existing element
// and return NULL.
// If "create_new" is true, if the element specified by spec does not exist,
// create it and add it to the structure.  Return the new/referenced element
ConfigElem* StructElem::makeElem(const char* spec, bool ignore_duplicates,
                                 bool create_new)
{
  Input input;
  input.setBuffer((void*) spec, strlen(spec));
  String tag;
  return makeElem(input, tag, ignore_duplicates, create_new);
}

// Reads a element specification from input.  If ignore_duplicates
// is false, then print an error if "spec" refers to an existing element
// and return NULL.
// If "create_new" is true, if the element specified by spec does not exist,
// create it and add it to the structure.  Return the new/referenced element,
// and puts the read tag into "tag".
ConfigElem* StructElem::makeElem(Input& input, String& tag,
                                 bool ignore_duplicates, bool create_new)
{
  const char* err_msg = "EOF while reading struct";
  char err_buf[200];
  ConfigElem* elem = (ConfigElem*) NULL;
  char c;
  StructElem* context = this;
  Type expected_type;
  bool good_type;

  // get the first "tag"
  if (!input.readWord(tag, "_")) 
    goto error;

  if (tag.getString() == "mergeStruct") {
    expected_type = StructElem::getClassTypeId();
    good_type = true;
  } else {
    // see if first word of tag is a type identifier
    expected_type = Type::fromName(tag.getString());
    good_type = expected_type.isDerivedFrom(ConfigElem::getClassTypeId());
  }

  if (good_type) { //if so, next element should be the first tag
    if (!input.readWord(tag, "_")) 
      goto error;
    good_type = expected_type.canCreateInstance();
  }

  while (1) {
    c = ' ';  // default value if read fails
    input.read(c);  // read next character
    // see if there is a subelement named "tag"
    ConfigElem* sub = context->lookup(tag.getString());
    if (sub) {  // yes
      if (c == '.') {  // next character indicates more structure refs
        sub->ref();  // protect against unknown conversion
        // get the sub-structure
        if (!sub->getValue(StructElem::getClassTypeId(), &context)) {
          // if not a substructure, it is an error
          sub->unref();
          err_msg = "Primitive type cannot have structure";
          goto error;
        }
        sub->unref();
      } else if (c == '[') { // next character indicates an index 
        // into an array of structures
        // read a number and the closing bracket
        int index;   
        if (!input.read(index) || !input.read(c) || c != ']')
          goto error;
        // make sure size in range
        if (index < 0 || index > sub->numValues()-1) {
          err_msg = "Index out of range";
          goto error;
        }
        // See if we can get an array of structures out of "sub"
        ConfigElem** array = new ConfigElem*[index+1];
        sub->ref();
        bool res = sub->getValue(StructElem::getClassTypeId(), array,
                                 index+1);
        sub->unref();
        // context will be the indexth structure
        context = (StructElem*) array[index];
        delete [] array;
        if (!res) {  // unless we can't get the right number of structs
          sprintf(err_buf, "Cannot get at index %d of %s\n", index,
                  tag.getString());
          err_msg = err_buf;
          goto error;
        }
        // we can't access anything except structure arrays
        if (!input.read(c) || c != '.') {
          err_msg = "Indexing only valid for structure arrays";
          goto error;
        }
      } else {
        // next character indicates this is the end of the spec
        if (ignore_duplicates)  // if wew can ignore duplicates
          return sub;  // return the element
        // otherwise we have an error.
        sprintf(err_buf, "Duplicate definition of '%s'\n",
                tag.getString());
        err_msg = err_buf;
        goto error;
      }
    } else {  // a new element
      if (c == '.') {  // expects sub elements?
        StructElem* sub;  // create new structure
        sub = new StructElem();  
        sub->setLineNum(input.getLineNum());
        // and recurse
        context->set(tag.getString(), sub);
        context = sub;
      } else {  // expect new primitive
        if (!create_new)  // if invalid to create new prim, return
          return (ConfigElem*) NULL;

        // if the type is specified, generate it
        if (good_type) 
          elem = (ConfigElem*) expected_type.createInstance();
        else {  // otherwise create a new unknown element with the data
          UnknownElem* unknown =
            new UnknownElem(context, tag.getString());
          elem = unknown;
        }
        elem->setLineNum(input.getLineNum());
        context->set(tag.getString(), elem);
        break;
      }
    }
    // read a word (words also include underscores here) as the next tag
    if (!input.readWord(tag, "_"))
      goto error;
  }

  // finished with type spec and name spec, now read optional expected size
  if (c == '[') {  // indicated by open bracket
    int expected_size = 0;
    // read the expected size
    if (!input.read(expected_size) || expected_size < 1) {
      err_msg = "Problem reading size";
      goto error;
    }
    // read the end bracket
    if (!input.read(c) || c != ']') {
      err_msg = "Malformed size";
      goto error;
    }
    // set the expected size, if possible
    if (!elem->setExpectedSize(expected_size)) {
      sprintf(err_buf, "Cannot have array of %s\n",
              elem->getTypeName());
      err_msg = err_buf;
      goto error;
    }
  } else
    input.putBack(c);  // undo any damage, if necessary

  return elem;

 error:
  input_error(err_msg, input);

  return (ConfigElem*) NULL;
}

// static helper function applied StructElem::writeData to its element dict.
// data is void* cast Output file.  
void StructElem::write_struct_elem(const char* key, ConfigElem* elem,
                                   void* data)
{
  Output* output = (Output*) data;
  output->indent();
  // output type
  const char* type = elem->getTypeId().getName().getString();
  if (type && elem->getTypeId() != UnknownElem::getClassTypeId()) {
    output->write(type);
    output->write(' ');
  }
  // output element name
  output->write(key);
  if (elem->getExpectedSize() > 1) {
    output->write('[');
    output->write(elem->getExpectedSize());
    output->write(']');
  }
  if (elem->getTypeId().isDerivedFrom(StructElem::getClassTypeId()))
    output->write(' ');
  else
    output->write(" = ");
  // output element data
  elem->writeData(*output);
  output->write('\n');
}

// write a text representation of the structure to output
void StructElem::writeData(Output& output)
{
  output.write("{\n");
  output.incrementIndent();

  _dict.applyToAll(write_struct_elem, &output);
  output.decrementIndent();
  output.indent();
  output.write("}");
}

bool StructElem::readData(Input& input)
{
  clear();
  ConfigParser p(input, this);
  if (p.yyparse() == 0) {
    invokeSetWatch();
    return true;
  }
  if (!p.errorPrinted) {
    printf("File %s, line %d, syntax error\n", input.getCurFileName(),
           input.getLineNum());
  }
  return false;
}

// lookup the name "name" in the structure dictionary.  No type or substructure
// parsing is done.
ConfigElem* StructElem::lookup(const char* name) const
{
  ConfigElem* result;
  if (_dict.find(name, result))
    return result;
  else
    return (ConfigElem*) NULL;
}

// Set the simple name "name" to be the element "elem".  Override previous
// definitions of name.
void StructElem::set(const char* name, ConfigElem* elem)
{
  ConfigElem* prev;
  if (_dict.find(name, prev)) {
    if (elem == prev)
      return;
    _dict.remove(name);
    prev->unref();
  }
  _dict.enter(name, elem);
  elem->ref();
  if (watcher()) 
    watcher()->add(this, name, elem);
}

// static helper function for copying from one structure to another (passed
// in as void* cbd)
static void copy_struct_member(const char* name, ConfigElem* elem, void* cbd)
{
  StructElem* new_struct = (StructElem*) cbd;
  ConfigElem* new_elem;
  if (elem->getTypeId() == UnknownElem::getClassTypeId())
    new_elem = new UnknownElem(new_struct, name, (UnknownElem*) elem);
  else
    new_elem = elem->clone();
  new_struct->set(name, new_elem);
}

// Create a copy of the structure element
bool StructElem::copy(ConfigElem* src)
{
  if (src == this)
    return true;
  if (!src->getTypeId().isDerivedFrom(StructElem::getClassTypeId()))
    return false;
  utils::Managed::deferDeletions();
  clear();
  ((StructElem*) src)->_dict.applyToAll(copy_struct_member, this);
  utils::Managed::undeferDeletions();
  return true;
}

bool StructElem::merge(ConfigElem* src)
{
  if (src == this)
    return true;
  if (!src->getTypeId().isDerivedFrom(StructElem::getClassTypeId()))
    return false;
  ((StructElem*) src)->_dict.applyToAll(copy_struct_member, this);
  return true;
}

///////////////////////////////////////////////////////////////////////////
// RootElem implementation.  RootElem is used as the root element of
//  ConfigFiles so that when we print a config file it is not enclosed
//  in spurious brackets

// and this is what does that, simple prints the elements without brackets
// or extra indentation overriding the StructElem::writeData
void RootElem::writeData(Output& output)
{
  _dict.applyToAll(write_struct_elem, &output);
}


///////////////////////////////////////////////////////////////////////////
// StructArray implementation.  Holds an array of structures

StructArrayElem::~StructArrayElem()
{
  clear();
}

void StructArrayElem::clear() {
  for (int i=0; i<_array.numElems(); i++) 
    _array[i]->unref();
  _array.clear();
}

void StructArrayElem::add(StructElem* elem)
{
  elem->ref();
  _array.append(elem);
}

// write the structure to output
void StructArrayElem::writeData(Output& output)
{
  //  print the elements
  output.incrementIndent();
  output.write('\n');
  for (int i=0; i<_array.numElems();i++) {
    output.indent();
    _array[i]->writeData(output);
    output.write('\n');
  }

  // and mark last with semicolon
  output.decrementIndent();
  output.indent();
  output.write(";\n");
}

bool StructArrayElem::readData(Input& input)
{
  clear();
  char c;
  while (1) {
    if (!input.read(c) || c == ';')
      return true;
    input.putBack(c);

    StructElem* elem = new StructElem();
    if (!elem->readData(input)) {
      delete elem;
      return false;
    }
    add(elem);
  }
}

// Get the structures.  We can only convert to StructElem's
int StructArrayElem::getValue(Type dest_type, void* data, int max_num) const
{
  if (!dest_type.isDerivedFrom(StructElem::getClassTypeId())) {
    get_error("Cannot get a non-struct from a struct array\nP.S. Most likely you have put an = in front of a structure\n",
              this);
    return 0;
  }

  StructElem** res = (StructElem**) data;
  int num = MIN(max_num, _array.numElems());
  for (int i=0;i<num;i++) 
    res[i] = _array[i];

  return num;
}

// Set the data.  
bool StructArrayElem::setValue(Type type, void* data, int num_values)
{
  if (!type.isDerivedFrom(getClassTypeId())) 
    return false;
  StructElem** res = (StructElem**) data;
  Managed::deferDeletions();
  int i;
  for (i=0;i<_array.numElems();i++)
    _array[i]->unref();
  _array.clear();
  _array.setMaxNum(num_values);
  for (i=0;i<num_values;i++) {
    _array.append(res[i]);
    res[i]->ref();
  }
  Managed::undeferDeletions();
  invokeSetWatch();
  
  return true;
}

StructElem* StructArrayElem::get(int index)
{
  if (index < 0 || index > _array.numElems())
    return NULL;
  return _array[index];
}

bool StructArrayElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(StructArrayElem::getClassTypeId()))
    return false;
  StructArrayElem* src_elem = (StructArrayElem*) src;
  Managed::deferDeletions();
  clear();
  _array.setMaxNum(src_elem->_array.numElems());
  for (int i=0;i<src_elem->_array.numElems();i++) {
    StructElem* new_array = (StructElem*) src_elem->_array[i]->clone();
    _array.append(new_array);
    new_array->ref();
  }
  Managed::undeferDeletions();
            
  return true;
}

///////////////////////////////////////////////////////////////////////////
// Implementation of PrimElem - provides common implementation for concrete
//   subclasses

bool PrimElem::parseVal(const char* val)
{
  if (!parsePrim(val))
    return false;
  _num_values++;
  return true;
}

// write the array of primitive data to output
void PrimElem::writeData(Output& output)
{
  int i=0;
  if (_num_values) {
    while (1) {
      writePrim(output, i++);
      if (i >= _num_values)
        break;
      output.write(' ');
    }
  }
  output.write(';');
}

bool PrimElem::readData(Input& input)
{
  String s;
  char c;

  clear();
  while (1) {
    if (!input.read(c) || c == ';') {
      invokeSetWatch();
      return true;
    }
    input.putBack(c);
    input.read(s);
    int len = s.getLength();
    if (s.getString()[len-1] == ';') {
      input.putBack(';');
      s = String(s.getString(), 0, len-2);
    }
    if (!parseVal(s.getString()))
      return false;
  }
}

// set the primitive data to the num_values of pieces of data
bool PrimElem::setValue(Type type, void* data, int num_values)
{
  _num_values = num_values;
  for (int i=0;i<num_values;i++) 
    if (!setPrim(type, data, i))
      return false;

  invokeSetWatch();
  return true;
}

///////////////////////////////////////////////////////////////////////////
// Implementation of UnknownElem - caches data with unspecified type

UTILS_BASE_ABSTRACT_SOURCE(UnknownElem);

// create an unknown element as part of structure context with name
// If other is non-null use other's data and expected size (essentially
// making this a crufty copy-constructor)
UnknownElem::UnknownElem(StructElem* context, const char* name,
                         UnknownElem* other)
  : _name(name) {
  UTILS_BASE_CONSTRUCTOR(UnknownElem);
  if (other) {
    for (int i=0;i<other->numValues();i++) 
      parseVal(other->get(i));
    invokeSetWatch();
    setExpectedSize(other->getExpectedSize());
  }
  _context = context;
}

void UnknownElem::initClass()
{
  UTILS_BASE_INIT_ABSTRACT_CLASS(UnknownElem, "Unknown", "string");
}

void UnknownElem::writePrim(Output& output, int i)
{
  const char* val = get(i);
  if (strchr(val, ' ')) {  // hack to fix most cases
    output.write('"');
    output.write(get(i));
    output.write('"');
  } else 
    output.write(get(i));
}

// Convert the unknown element to type dest_type and put the resulting
// data in data (up to max_num elements of it at least).
// Also, replace the UnknownElement in its structure with the equivalent
// parsed element of type dest_type.  This will cause a deref of 
// this, which is why we do the ref/unref pair around the getValue's in
// StructElem::getValue
int UnknownElem::getValue(Type dest_type, void* data, int max_num) const
{
  // generate an element of the appropriate type
  ConfigElem* elem = (ConfigElem*) dest_type.createInstance();
  if (!elem) {
    get_error("Illegal destination type", this);
    return 0;
  }

  // if the expected size needs to be set and can't for this element,
  // return an error
  if (getExpectedSize() != -1)
    if (!elem->setExpectedSize(getExpectedSize())) {
      elem->ref();   // delete the new element
      elem->unref();
      get_error("Cannot set expected size",this);
      return 0;
    }

  // parse the cached unknown data into the new element
  for (int i=0;i<numValues();i++) 
    elem->parseVal(get(i));

    // replace the UnknownElem with the now known element
  UnknownElem* unknown = (UnknownElem*) this;
  elem->setWatcher(watcher());
  unknown->_context->set(_name.getString(), elem);
  elem->invokeSetWatch();

    // and put the requested values into data
  return elem->getValue(dest_type, data, max_num);
}

///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type IntElem

// read an integer
bool IntElem::parsePrim(const char* val)
{
  char* end_ptr;
  int i = strtol(val, &end_ptr, 10);
  if (*end_ptr != '\0' && *end_ptr != ';')
    return false;

  _data.append(i);
  return true;
}

// write an integer
void IntElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write(_data[i]);
}

// set an integer, assuming data is an (int*)
bool IntElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append(0);
    _data.append(((int*) data)[i]);
  } else
    _data[i] = ((int*) data)[i];

  return true;
}

// get integer values.  Can convert to float, double, character, or boolean
// destination types
int IntElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    memcpy((char*) data, (char*) _data.getData(), num_values*sizeof(int));
    return num_values;
  } else if (dest_type == FloatElem::getClassTypeId()) {
    float* output = (float*) data;
    for (i=0;i<num_values;i++)
      output[i] = (float) _data[i];
    return num_values;
  } else if (dest_type == DoubleElem::getClassTypeId()) {
    double* output = (double*) data;
    for (i=0;i<num_values;i++)
      output[i] = (double) _data[i];
    return num_values;
  } else if (dest_type == CharElem::getClassTypeId()) {
    char* output = (char*) data;
    for (i=0;i<num_values;i++)
      output[i] = (char) _data[i];
    return num_values;
  } else if (dest_type == BoolElem::getClassTypeId()) {
    bool* output = (bool*) data;
    for (i=0;i<num_values;i++)
      output[i] = (_data[i] != 0);
    return num_values;
  } else {
    get_error("Invalid destination type for int", this);
    return 0;
  }
}

// Create a integer element copy
bool IntElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(IntElem::getClassTypeId()))
    return false;
  setNumValues(src->numValues());
  _data = ((IntElem*) src)->_data;
  return true;
}


///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type FloatElem

// read a float
bool FloatElem::parsePrim(const char* val)
{
  char* end_ptr;
  float f = strtod(val, &end_ptr);
  if (*end_ptr != '\0' && *end_ptr != ';')
    return false;

  _data.append(f);
  return true;
}

// write a float
void FloatElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write(_data[i]);
}

// set a float, assuming data is an (float*)
bool FloatElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append(0.0);
    _data.append(((float*) data)[i]);
  } else
    _data[i] = ((float*) data)[i];

  return true;;
}

// get float values.  Can convert to integer, double, or boolean
// destination types
int FloatElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    memcpy((char*) data, (char*) _data.getData(),num_values*sizeof(float));
    return num_values;
  } else if (dest_type == IntElem::getClassTypeId()) {
    int* output = (int*) data;
    for (i=0;i<num_values;i++)
      output[i] = (int) _data[i];
    return num_values;
  } else if (dest_type == DoubleElem::getClassTypeId()) {
    double* output = (double*) data;
    for (i=0;i<num_values;i++)
      output[i] = (double) _data[i];
    return num_values;
  } else if (dest_type == BoolElem::getClassTypeId()) {
    bool* output = (bool*) data;
    for (i=0;i<num_values;i++)
      output[i] = (_data[i] != 0.0);
    return num_values;
  } else {
    get_error("Invalid destination type for float", this);
    return 0;
  }
}

// Create a float element copy
bool FloatElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(FloatElem::getClassTypeId()))
    return false;
  setNumValues(src->numValues());
  _data = ((FloatElem*) src)->_data;
  return true;
}


///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type DoubleElem

// read a double
bool DoubleElem::parsePrim(const char* val)
{
  char* end_ptr;
  double d = strtod(val, &end_ptr);
  if (*end_ptr != '\0' && *end_ptr != ';')
    return false;

  _data.append(d);
  return true;
}

// write a double
void DoubleElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write(_data[i]);
}

// set a double, assuming data is an (double*)
bool DoubleElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append(0.0);
    _data.append(((double*) data)[i]);
  } else
    _data[i] = ((double*) data)[i];

  return true;
}

// get float values.  Can convert to integer, float, or boolean
// destination types
int DoubleElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    memcpy((char*) data, (char*) _data.getData(),
           num_values*sizeof(double));
    return num_values;
  } else if (dest_type == IntElem::getClassTypeId()) {
    int* output = (int*) data;
    for (i=0;i<num_values;i++)
      output[i] = (int) _data[i];
    return num_values;
  } else if (dest_type == FloatElem::getClassTypeId()) {
    float* output = (float*) data;
    for (i=0;i<num_values;i++)
      output[i] = (float) _data[i];
    return num_values;
  } else if (dest_type == BoolElem::getClassTypeId()) {
    bool* output = (bool*) data;
    for (i=0;i<num_values;i++)
      output[i] = (_data[i] != 0.0);
    return num_values;
  } else {
    get_error("Invalid destination type for double", this);
    return 0;
  }
}

// Create a double element copy
bool DoubleElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(DoubleElem::getClassTypeId()))
    return false;
  setNumValues(src->numValues());
  _data = ((DoubleElem*) src)->_data;
  return true;
}


///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type CharElem

// read a character
bool CharElem::parsePrim(const char* val)
{
  if (!*val || 
      (*(val+1) && (*(val+1) != ';')))
    return false;
  _data.append(*val);
  return true;
}

// write a character
void CharElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write(_data[i]);
}

// set a character, assuming data is an (char*)
bool CharElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append('\0');
    _data.append(((char*) data)[i]);
  } else
    _data[i] = ((char*) data)[i];

  return true;
}

// get character values.  Can convert to integer destination types
int CharElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    memcpy((char*) data, (char*) _data.getData(), num_values*sizeof(char));
    return num_values;
  } else if (dest_type == IntElem::getClassTypeId()) {
    int* output = (int*) data;
    for (i=0;i<num_values;i++)
      output[i] = (int) _data[i];
    return num_values;
  } else {
    get_error("Invalid destination type for char", this);
    return 0;
  }
}

// Create a character element copy
bool CharElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(CharElem::getClassTypeId()))
    return false;
  setNumValues(src->numValues());
  _data = ((CharElem*) src)->_data;
  return true;
}


///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type BoolElem

// read a boolean value.  Any string starting with "f" or "F" is a false
// value and any string starting with "t" or "T" is a true value.  The string
// "0" is false and the string "1" is true as well.
bool BoolElem::parsePrim(const char* val)
{
  String s(val);

  if (s == "0" || *s.getString() == 'F' || *s.getString() == 'f' ||
      s == "0;") {
    _data.append(false);
    return true;
  }
  if (s == "1" || *s.getString() == 'T' || *s.getString() == 't' ||
      s == "1;") {
    _data.append(true);
    return true;
  }
  return false;
}

// write a boolean (as 0 or 1)
void BoolElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write((int) _data[i]);
}

// set a boolean, assuming data is an (bool*)
bool BoolElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append(false);
    _data.append(((bool*) data)[i]);
  } else
    _data[i] = ((bool*) data)[i];
  return true;
}

// get integer values.  Can convert to integer destination types
int BoolElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    memcpy((char*) data, (char*) _data.getData(), num_values*sizeof(bool));
    return num_values;
  } else if (dest_type == IntElem::getClassTypeId()) {
    int* output = (int*) data;
    for (i=0;i<num_values;i++)
      output[i] = (int) _data[i];
    return num_values;
  } else {
    get_error("Invalid destination type for bool", this);
    return 0;
  }
}

// Create a boolean element copy
bool BoolElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(BoolElem::getClassTypeId()))
    return false;
  setNumValues(src->numValues());
  _data = ((BoolElem*) src)->_data;
  return true;
}


///////////////////////////////////////////////////////////////////////////
// Implementation of primitive type StringElem

StringElem::~StringElem()
{
  clear();
}

// clear memory associated with a string
void StringElem::clear()
{
  for (int i=0;i<_data.numElems();i++) 
    delete _data[i];
  _data.clear();
  setNumValues(0);
}

bool StringElem::parsePrim(const char* val)
{
  // first strip off any extraneous external quotes added by the 
  // assignment process in ConfigParser
  int len = strlen(val);
  String* new_val;
  if (len > 2 && *val=='"' && val[len-1]==';' && val[len-2]=='"') {
    new_val = new String(val, 1, len-3);
  } else
    new_val = new String(val, 0, len-1);
    
  _data.append(new_val);
  return true;
}

// write a string
void StringElem::writePrim(Output& output, int i)
{
  if (i<0 || i >= _data.numElems())
    return;
  output.write('"');
  output.write(_data[i]->getString());
  output.write('"');
}

// set a string, assuming data is an (char**)
bool StringElem::setPrim(Type type, void* data, int i)
{
  if (type != getClassTypeId())
    return false;
  if (i <0)
    return false;
  if (i>= _data.numElems()) {
    for (int index=_data.numElems();index<i;index++) 
      _data.append(new String());
    _data.append(new String(((char**) data)[i]));
  } else {
    if (_data[i])
      delete _data[i];
    _data[i] = new String(((char**) data)[i]);
  }
  return true;
}

// get integer values.  Cannot convert to other destination types
int StringElem::getValue(Type dest_type, void* data, int max_num) const
{
  int num_values = MIN(max_num, _data.numElems());
  int i;
  if (dest_type == getClassTypeId()) {
    const char** output = (const char**) data;
    for (i=0;i<num_values;i++)
      output[i] = _data[i]->getString();
    return num_values;
  } else {
    get_error("Invalid destination type for string", this);
    return 0;
  }
}

// Create a string element copy
bool StringElem::copy(ConfigElem* src)
{
  if (!src->getTypeId().isDerivedFrom(StringElem::getClassTypeId()))
    return false;
  StringElem* string_src = (StringElem*) src;
  _data.setMaxNum(string_src->_data.numElems());
  clear();
  setNumValues(src->numValues());
  for (int i=0;i<string_src->_data.numElems();i++)
    _data.append(new String(*(string_src->_data[i])));
  return true;
}

ConfigFile& ConfigFile::operator=(const ConfigFile& in)
{
  initRoot(in._root);
  return *this;
}

ConfigElem* ConfigFile::setStruct(const char* name, utils::ConfigFile& params)
{
  ConfigElem* elem = params.getRoot();
  return set(StructElem::getClassTypeId(), name, &elem, 1);
}

ConfigElem* ConfigFile::mergeStruct(const char* name,
                                    utils::ConfigFile& params)
{
  StructElem* dest = (StructElem*) getStruct(name);
  if (!dest) 
    setStruct(name, params);
  else
    if (!dest->merge(params.getRoot()))
      return NULL;
  return dest;
}

ConfigElem* ConfigFile::makeElem(const char* name, bool ignore_duplicates,
                                 bool create_new)
{
  if (!_root)
    return NULL;
  return _root->makeElem(name, ignore_duplicates, create_new);
}

UTILS_BASE_ABSTRACT_SOURCE(ConfigWatch);
 
void ConfigWatch::initClass()
{
  UTILS_BASE_INIT_ABSTRACT_CLASS(ConfigWatch, "ConfigWatch", "Base");
}

ConfigWatch::ConfigWatch()
{
  UTILS_BASE_CONSTRUCTOR(ConfigWatch);
}

__UTILS_END_NAMESPACE
