#include <utils/Input.h>
#include <utils/CannedDataAccess.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

__UTILS_BEGIN_NAMESPACE

char *CannedDataReader::_type_descriptions[] = 
{
	"Unknown",
	"Test",
	"Video",
	"Sick"      
};

void CannedDataReader::_init()
{
	_type          = CDT_UNKNOWN;
	_major_version = 0;
	_minor_version = 0;
	_id            = 0;
	_creation_time = 0;
	_header_size   = 0;
}

CannedDataReader::CannedDataReader()
{
	_init();
}

CannedDataReader::CannedDataReader(const char *filename)
{
	_init();
	open(filename);
}

CannedDataReader::~CannedDataReader()
{
        close();
}


void CannedDataReader::_openDataFiles(const char *filename, int last_data_file) 
{
	char buf[16];
	int i;
	String data_filename;
	assert(_data_files.empty());
        Input input;
        if (!input.openFile(filename))
          throw CannedDataError("Couldn't open %s for input: %s\n", filename,
                                strerror(errno));

	_data_files.append(input.getCurFile());
        input.releaseFile();
	
	for (i = 1; i <= last_data_file; i++) {
		snprintf(buf, 16, "%i", i - 1);
		buf[15] = 0;
		data_filename = filename;
		data_filename += ".data.";
		data_filename += buf;
                if (!input.openFile(data_filename.getString()))
                  throw CannedDataError("Couldn't open %s for input: %s\n",
                                        filename,
                                        strerror(errno));
		_data_files.append(input.getCurFile());
                input.releaseFile();
	}
}

/*
 * Open a file for reading.  Load an index of the contents, so
 * this could take a non-trivial amount of time.  
 */
void CannedDataReader::open(const char *filename)
{
	char buf[5];
	char *desc;
	uint16_t id, length;
	uint32_t tmp;
	uint32_t cd_major_version, cd_minor_version;
	String idx_filename;
	int last_data_file = 0;
	/* 
	 * Playing fast and loose with this, methinks. 
	 * But it *should* work.  Damn c++! 
	 */
	CannedDataIdxRecord rec;
	int partial_read = 0;
	idx_filename = filename;
        Input input;
        if (!input.openFile(idx_filename.getString()))
          throw CannedDataError("Couldn't open %s for reading: %s", 
                                filename, strerror(errno));
        _idx_file = input.getCurFile();
        input.releaseFile();

	if (!((fread(buf, 1, 4, _idx_file) == 4) 
	      && fread(&cd_major_version, 4, 1, _idx_file)
	      && fread(&cd_minor_version, 4, 1, _idx_file))) {
		throw CannedDataError("Unexpected end of file %s", filename);
	}
	cd_major_version = le_to_host_uint32(cd_major_version);
	cd_minor_version = le_to_host_uint32(cd_minor_version);
	buf[4] = 0;
	if (strcmp(buf, "DWHO")) {
		throw CannedDataError("No valid signature found in file %s", 
				      filename);
	}
	
	/*
	 * We'll have to change this later if we want to do compatibility
	 * work
	 */
	if (!((cd_major_version == CANNED_DATA_MAJOR_VERSION)
	      && (cd_minor_version == CANNED_DATA_MINOR_VERSION))) {
		throw CannedDataError("Version mismatch.  File %s was created "
				      "with CannedData library version %x.%x."
				      "  Library in use is version %x.%x.",
				      filename,
				      cd_major_version,
				      cd_minor_version,
				      CANNED_DATA_MAJOR_VERSION,
				      CANNED_DATA_MINOR_VERSION);
	}
	    

	while (!(feof(_idx_file) || partial_read)) {
		if (!(fread(&id, 2, 1, _idx_file)
		      && fread(&length, 2, 1, _idx_file))) {
			partial_read = 1;
			break;
		}
		id = le_to_host_uint16(id);
		length = le_to_host_uint16(length);
		
		switch(id) {
		case CDID_INVALID:
			throw CannedDataError("Encountered invalid field in file %s", filename);
		case CDID_TYPE:
			if (length != 4) {
				throw CannedDataError("Type field appears to be screwed in file %s (%i)", filename, length);
			}
			if (!fread(&tmp, 4, 1, _idx_file)) {
				partial_read = 1;
				break;
			}
			_type = (CannedDataType)le_to_host_uint32(tmp);
			break;
		case CDID_VERSION:
			if (length != 8) {
				throw CannedDataError("Version field appears to be screwed in file %s", filename);
			}
			if (!(fread(&_major_version, 4, 1, _idx_file)
			      && fread(&_minor_version, 4, 1, _idx_file))) {
				partial_read = 1;
			}
			_major_version = le_to_host_uint32(_major_version);
			_minor_version = le_to_host_uint32(_minor_version);
			break;
		case CDID_ID:
			if (length != 4) {
				throw CannedDataError("ID field appears to be screwed in file %s", filename);
			}
			if (!fread(&_id, 4, 1, _idx_file)) {
				partial_read = 1;
			}
			break;
		case CDID_CREATION_TIME:
			if (length != 8) {
				throw CannedDataError("Creation field appears to "
						      "be screwed in file %s", 
						      filename);
			}
			if (!fread(&_creation_time, 8, 1, _idx_file)) {
				partial_read = 1;
			}
			_creation_time = le_to_host_uint64(_creation_time);
			break;
		case CDID_DESCRIPTION:
			desc = (char *)calloc(1, length + 1);
			if (!desc) {
				throw CannedDataError("Out of memory");
			}
			if (fread(desc, 1, length, _idx_file) != length) {
				partial_read = 1;
			}
			_desc = desc;
			free(desc);
			break;
		case CDID_HEADER:
			_header_size = length;
			_header_offset = ftell(_idx_file);
			fseek(_idx_file, length, SEEK_CUR);
			break;
		case CDID_IDX:
			if (length != 25) {
				throw CannedDataError("Index field screwed!");
			}
			if (!(fread(&rec.time_stamp.secs, 8, 1, _idx_file)
			      && fread(&rec.time_stamp.usecs, 8, 1, _idx_file)
			      && fread(&rec.ofs, 4, 1, _idx_file)
			      && fread(&rec.size, 4, 1, _idx_file)
			      && fread(&rec.file_no, 1, 1, _idx_file))) {
				partial_read = 1;
			} else {
				rec.time_stamp.secs = le_to_host_uint64(rec.time_stamp.secs);
				rec.time_stamp.usecs = le_to_host_uint64(rec.time_stamp.usecs);
				rec.ofs = le_to_host_uint32(rec.ofs);
				rec.size = le_to_host_uint32(rec.size);
				if (rec.file_no >= last_data_file) {
					last_data_file = rec.file_no;
				}
				_index.add(rec);
			}
			break;
		case CDID_END:
			/*
			 * It's not really a partial read, but this is a simple way to get
			 * out of this loop...
			 */
			partial_read = 1;
			break;
		default:
		  throw CannedDataError("Unknown field ID: %i\n", (int)id);
		  break;
		}
	}
	if (_index.size() == 0) {
		throw CannedDataError("No records found in file %s\n", filename);
	}
	_openDataFiles(filename, last_data_file);
}



void CannedDataReader::close()
{
        for (int i=0;i<_data_files.numElems();i++) 
                fclose(_data_files[i]);
        _data_files.clear();
}

/* 
 * Return the filetype that we think this is 
 */
CannedDataType CannedDataReader::getType()
{
	return _type;
}


/* 
 * Return the filetype that we think this is 
 */
String CannedDataReader::getTypeDescription()
{
	int idx = (int)_type;
	if ((idx < 0) || (idx >= CDT_INVALID)) {
		throw CannedDataError("Type is out of bounds!");
	}
	String retval = _type_descriptions[idx];
	return retval;
}


/*
 * Return the version of data format used in this file
 */
void CannedDataReader::getVersion(uint32_t *major, uint32_t *minor)
{
	(*major) = _major_version;
	(*minor) = _minor_version;
}


/* 
 * Return a file pointer to the beginning of the header information.  
 * The file pointer is only valid until the next member function
 * call.  If no header was found, returns a null pointer
 */
FILE *CannedDataReader::getHeader()
{
	if (!_header_size) {
		return 0;
	}
	fseek(_idx_file, _header_offset, SEEK_SET);
	return _idx_file;
}


/*
 * Return the total size of the header
 */
size_t CannedDataReader::getHeaderSize()
{
	return _header_size;
}


/* 
 * Adjust reading position to be the record
 * closest to the passed time.  
 */
void CannedDataReader::seek(uint64_t secs, uint64_t usecs)
{
        _warn_deprecated();
	_index.seek(secs, usecs);
}

void CannedDataReader::seek(const Time &stamp)
{
        long secs, usecs;
        stamp.getValue(secs, usecs);
        _index.seek(secs, usecs);
}

/*
 * Adjust reading position to be the record
 * _in the past_ closest to the passed time.
 * 
 * Returns -1 and doesn't move the file
 * pointer if there is no record earlier
 * than this one.  Returns 0 on success
 */
int CannedDataReader::seekBefore(uint64_t secs, uint64_t usecs)
{
        _warn_deprecated();
	return _index.seekBefore(secs, usecs);
}

int CannedDataReader::seekBefore(const Time &stamp)
{
	return _index.seekBefore(stamp.sec, stamp.usec);
}

	


/* 
 * Returns 1 if the last time requested was
 * beyond the end of the file.  Note the
 * seek will still point at the last 
 * record of the file and return valid
 * results in a read, though.
 */
int CannedDataReader::eof()
{
	return _index.atEnd();
}


/* 
 * Rewind to the first record of the file
 */
void CannedDataReader::first()
{
	_index.first();
}

/* 
 * Go to the last record in the file
 */
void CannedDataReader::last()
{
	_index.last();
}


/*
 * Select the next record.  If at the end, this is a nop.
 */
void CannedDataReader::next()
{
	_index.next();
}


/* 
 * Select the previous record.  If at the start, this is a nop.
 */
void CannedDataReader::prev()
{
	_index.previous();
}


/*
 * Get the timestamp associated with the current
 * record
 */
void CannedDataReader::getCurTimeStamp(uint64_t *secs, uint64_t *usecs)
{
        _warn_deprecated();
	CannedDataIdxRecord tmp = _index.current();
	(*secs) = tmp.time_stamp.secs;
	(*usecs) = tmp.time_stamp.usecs;
}


/*
 * Get the timestamp associated with the current
 * record
 */
void CannedDataReader::getCurTimeStamp(CannedDataTimeStamp *stamp)
{
        _warn_deprecated();
	(*stamp) = _index.current().time_stamp;
}


/*
 * Get the timestamp associated with the current
 * record
 */
Time CannedDataReader::getCurTimeStamp()
{
	return Time(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
}


/*
 * Get the timestamp associated with the first
 * record of the file 
 */
void CannedDataReader::getFirstTimeStamp(uint64_t *secs, uint64_t *usecs)
{
        _warn_deprecated();
	CannedDataIdxRecord rec;
	int pos = _index.getPos();
	_index.first();
	rec = _index.current();
	(*secs) = rec.time_stamp.secs;
	(*usecs) = rec.time_stamp.usecs;
	_index.setPos(pos);
}

/*
 * Get the timestamp associated with the first
 * record of the file 
 */
void CannedDataReader::getFirstTimeStamp(CannedDataTimeStamp *stamp)
{
        _warn_deprecated();
	int pos = _index.getPos();
	_index.first();
	(*stamp) = _index.current().time_stamp;
	_index.setPos(pos);
}

/*
 * Get the timestamp associated with the first
 * record of the file 
 */
Time CannedDataReader::getFirstTimeStamp()
{
        Time retval;
	int pos = _index.getPos();
	_index.first();
        retval.setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
	_index.setPos(pos);
        return retval;
}


/*
 * Get the timestamp associated with the last
 * record of the file 
 */
void CannedDataReader::getLastTimeStamp(uint64_t *secs, uint64_t *usecs)
{
	CannedDataIdxRecord rec;
	int pos = _index.getPos();
	_index.last();
	rec = _index.current();
	(*secs) = rec.time_stamp.secs;
	(*usecs) = rec.time_stamp.usecs;
	_index.setPos(pos);
}


/*
 * Get the timestamp associated with the last
 * record of the file 
 */
void CannedDataReader::getLastTimeStamp(CannedDataTimeStamp *stamp)
{
	int pos = _index.getPos();
	_index.last();
	(*stamp) = _index.current().time_stamp;
	_index.setPos(pos);
}

/*
 * Get the timestamp associated with the last
 * record of the file 
 */
Time CannedDataReader::getLastTimeStamp()
{
        Time retval;
	int pos = _index.getPos();
	_index.last();
        retval.setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
	_index.setPos(pos);
        return retval;
}


/*
 * Get the timestamp of the record before this
 * one.  Returns 0 on success, -1 if this is the
 * first record in the file.  If this is the
 * first record, secs and usecs are filled with
 * the values of the current record
 */
int CannedDataReader::getPrevTimeStamp(uint64_t *secs, uint64_t *usecs)
{
	CannedDataIdxRecord rec;
	if (_index.atStart()) {
		rec = _index.current();
		(*secs) = rec.time_stamp.secs;
		(*usecs) = rec.time_stamp.usecs;
		return -1;
	}
	_index.previous();
	rec = _index.current();
	(*secs) = rec.time_stamp.secs;
	(*usecs) = rec.time_stamp.usecs;
	_index.next();
	return 0;
}

/*
 * Get the timestamp of the record before this
 * one.  Returns 0 on success, -1 if this is the
 * first record in the file.  If this is the
 * first record, secs and usecs are filled with
 * the values of the current record
 */
int CannedDataReader::getPrevTimeStamp(CannedDataTimeStamp *stamp)
{
	if (_index.atStart()) {
		(*stamp) = _index.current().time_stamp;
		return -1;
	}
	_index.previous();
	(*stamp) = _index.current().time_stamp;
	_index.next();
	return 0;
}

/*
 * Get the timestamp of the record before this
 * one.  Returns 0 on success, -1 if this is the
 * first record in the file.  If this is the
 * first record, secs and usecs are filled with
 * the values of the current record
 */
int CannedDataReader::getPrevTimeStamp(Time *retval)
{
	if (_index.atStart()) {
		retval->setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
		return -1;
	}
	_index.previous();
        retval->setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
	_index.next();
	return 0;
}


/*
 * Get the timestamp of the record after this
 * one.  Returns 0 on success, -1 if this is the
 * last record in the file.  If this is the last
 * record, just return the current timestamp
 */
int CannedDataReader::getNextTimeStamp(uint64_t *secs, uint64_t *usecs)
{
	CannedDataIdxRecord rec;
	if (_index.atEnd()) {
		rec = _index.current();
		(*secs) = rec.time_stamp.secs;
		(*usecs) = rec.time_stamp.usecs;
		return -1;
	}
	_index.next();
	rec = _index.current();
	(*secs) = rec.time_stamp.secs;
	(*usecs) = rec.time_stamp.usecs;
	_index.previous();
	return 0;
}


/*
 * Get the timestamp of the record after this
 * one.  Returns 0 on success, -1 if this is the
 * last record in the file.  If this is the last
 * record, just return the current timestamp
 */
int CannedDataReader::getNextTimeStamp(CannedDataTimeStamp *stamp)
{
	if (_index.atEnd()) {
		(*stamp) = _index.current().time_stamp;
		return -1;
	}
	_index.next();
	(*stamp) = _index.current().time_stamp;
	_index.previous();
	return 0;
}

/*
 * Get the timestamp of the record after this
 * one.  Returns 0 on success, -1 if this is the
 * last record in the file.  If this is the last
 * record, just return the current timestamp
 */
int CannedDataReader::getNextTimeStamp(Time *retval)
{
	if (_index.atEnd()) {
                retval->setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
		return -1;
	}
	_index.next();
        retval->setValue(_index.current().time_stamp.secs, _index.current().time_stamp.usecs);
	_index.previous();
	return 0;
}


/*
 * Get the number of records in this file
 */
unsigned int CannedDataReader::getNumRecords()
{
	return _index.size();
}


/* 
 * Return a FILE pointer pointing at the start of
 * the current record. 
 */
FILE *CannedDataReader::getRecord()
{
	FILE *retval;
	CannedDataIdxRecord rec;
	rec = _index.current();
	retval = _data_files[rec.file_no];
	fseek(retval, rec.ofs, SEEK_SET);
	return retval;
}


/*
 * Return the size of the current record
 */
size_t CannedDataReader::getRecordSize()
{
	return _index.current().size;
}


/*
 * Get the description of this file 
 */
const String &CannedDataReader::getDescription()
{
	return _desc;
}

/*
 * Get the creation time, which is in seconds from the epoch
 */
uint64_t CannedDataReader::getCreationTime()
{
	return _creation_time;
}

__UTILS_END_NAMESPACE
