#include <math.h>

#include <utils/List.h>
#include <utils/StringDict.h>
#include <utils/String.h>
#include <utils/Output.h>
#include <utils/ConfigFile.h>
#include <utils/Generator.h>
#include <utils/Time.h>
#include <utils/SymbolTable.h>
#include <utils/formatting/FormConfig.h>

#include <utils/BinLogger.h>

__UTILS_BEGIN_NAMESPACE

static Generator<LogElem>* LogElemGenerator = NULL;

class LogElem {
public:
  LogElem();
  virtual ~LogElem() {};

  const char* getFormat() const { return _format.getString(); }
  void setFormat(const char* format) { _format = format; }
  const char* getName() const { return _name.getString(); }
  void setName(const char* name) { _name = name; }
  void* getAddress() const { return _address; }
  void setAddress(void* address) { _address = address; }
  virtual bool log(Output& output, void* address) = 0;
  virtual int size() const = 0;

  static Generator<LogElem>* generator();

protected:
  String _format;
  String _name;
  void* _address;
};

class FloatLogElem : public LogElem {
public:
  FloatLogElem(bool use_min, float min, bool use_max, float max,
               bool periodical)
    : LogElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period=0;
  }

  virtual bool log(Output& output, void* address) {
    float val = *(float*) address;
    if (_period) {
      while (val < _min) 
        val += _period;
      while (val > _max)
        val -= _period;
    } else {
      if (_use_min && val < _min)
        val = _min;
      if (_use_max && val > _max)
        val = _max;
    }
    
    output.write((unsigned char*) &val, sizeof(float));

    return true;
  }

  virtual int size() const { return sizeof(float); }

private:
  bool _use_min;
  float _min;
  bool _use_max;
  float _max;
  float _period;
};

class ShortLogElem : public LogElem {
public:
  ShortLogElem(bool use_min, short min, bool use_max, short max,
               bool periodical)
    : LogElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period=0;
  }

  virtual bool log(Output& output, void* address) {
    short val = *(short*) address;
    if (_period) {
      while (val < _min) 
        val += _period;
      while (val > _max)
        val -= _period;
    } else {
      if (_use_min && val < _min)
        val = _min;
      if (_use_max && val > _max)
        val = _max;
    }
    
    output.write((unsigned char*) &val, sizeof(short));

    return true;
  }

  virtual int size() const { return sizeof(short); }

private:
  bool _use_min;
  short _min;
  bool _use_max;
  short _max;
  short _period;
};

class ByteLogElem : public LogElem {
public:
  ByteLogElem(bool use_min, char min, bool use_max, char max,
               bool periodical)
    : LogElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      if (_min > _max)
        _period = 0;
      else
        _period = _max - _min;
    } else
      _period=0;
  }

  virtual bool log(Output& output, void* address) {
    char val = *(char*) address;
    if (_period) {
      while (val < _min) 
        val += _period;
      while (val > _max)
        val -= _period;
    } else {
      if (_use_min && val < _min)
        val = _min;
      if (_use_max && val > _max)
        val = _max;
    }
    
    output.write((unsigned char*) &val, sizeof(char));

    return true;
  }

  virtual int size() const { return sizeof(char); }

private:
  bool _use_min;
  char _min;
  bool _use_max;
  char _max;
  char _period;
};

class DoubleLogElem : public LogElem {
public:
  DoubleLogElem(bool use_min, double min, bool use_max, double max,
               bool periodical)
    : LogElem()
  {
    _use_min = use_min; _min = min;
    _use_max = use_max; _max = max;
    if (periodical) {
      _period = _max - _min;
      if (_period < 0)
        _period = 0;
    } else
      _period=0;
  }

  virtual bool log(Output& output, void* address) {
    double val = *(double*) address;
    if (_period) {
      while (val < _min) 
        val += _period;
      while (val > _max)
        val -= _period;
    } else {
      if (_use_min && val < _min)
        val = _min;
      if (_use_max && val > _max)
        val = _max;
    }
    
    output.write((unsigned char*) &val, sizeof(double));

    return true;
  }

  virtual int size() const { return sizeof(double); }

private:
  bool _use_min;
  double _min;
  bool _use_max;
  double _max;
  double _period;
};

class ArrayLogElem : public LogElem {
public:
  ArrayLogElem(int size, const char* elem_format) 
    : LogElem()
  {
    _elem = LogElemGenerator->interface(elem_format, NULL);
    _num_elems = size;
  }
  virtual ~ArrayLogElem() { if (_elem) delete _elem; }

  virtual int size() const { return _elem->size()*_num_elems; }

  virtual bool log(Output& output, void* address) {
    if (!_elem)
      return false;
    bool res = true;
    for (int i=0;i<_num_elems;i++) {
      if (!_elem->log(output, ((unsigned char*) address)+i*_elem->size()))
        res = false;
    }
    return res;
  }

private:
  LogElem* _elem;
  int _num_elems;
};

class VectorLogElem : public LogElem {
public:
  VectorLogElem(const char* elem_format) : LogElem()
  {
    _elem = LogElemGenerator->interface(elem_format, NULL);
  }
  virtual ~VectorLogElem() { if (_elem) delete _elem; }
  virtual int size() const { return sizeof(LogVectorSpec); }
  
  virtual bool log(Output& output, void* address) {
    if (!_elem)
      return false;
    LogVectorSpec* spec = (LogVectorSpec*) address;
    bool res = true;
    output.write((unsigned char*) &spec->size, sizeof(int));
    for (int i=0;i<spec->size;i++) {
      if (!_elem->log(output, ((unsigned char*) spec->data)+i*_elem->size()))
        res = false;
    }
    return res;
  }
  
private:
  LogElem* _elem;
};


class LoggingClass {
public:
  LoggingClass(const char* tag);
  ~LoggingClass();

  bool declare(const char* name, const char* format, void* address);
  bool startup(Output& output);
  bool log(Output& output, Time time);

private:
  List<LogElem>* _elems;
  StringDict<LogElem*>* _table;
  String _tag;
};

struct LogElemInfo {
  char* tag;
  LogElem* (*create_interface)(Generator<LogElem>*, ConfigFile*,
                               SymbolTable*);
};

static LogElem* create_float_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  float min=100, max=-100;
  if (params->getFloats("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getFloats("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new FloatLogElem(use_min, min, use_max, max, periodical);
}

static LogElem* create_short_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  int min=100, max=-100;
  if (params->getInts("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getInts("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new ShortLogElem(use_min, min, use_max, max, periodical);
}

static LogElem* create_byte_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  int min=100, max=-100;
  if (params->getInts("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getInts("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new ByteLogElem(use_min, min, use_max, max, periodical);
}

static LogElem* create_double_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  bool use_min=false, use_max=false;
  double min=100, max=-100;
  if (params->getDoubles("min", &min, 1, NULL, 0))
    use_min = true;
  if (params->getDoubles("max", &max, 1, NULL, 0))
    use_max = true;
  bool periodical = params->getBool("periodical", false);
  if (periodical && (!use_min || !use_max)) {
    printf("Periodical elements require range\n");
    return NULL;
  }

  return new DoubleLogElem(use_min, min, use_max, max, periodical);
}

static LogElem* create_angle_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  return new FloatLogElem(true, -M_PI, true, M_PI, true);
}

static LogElem* create_array_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  int size_array;
  if (!params->getInts("size", &size_array, 1, NULL, 0)) {
    printf("Array element requires a size\n");
    return NULL;
  }
  
  return new ArrayLogElem(size_array,
                          params->getString("format", "float"));
}

static LogElem* create_vector_log_elem(Generator<LogElem>*,
                                      ConfigFile* params, SymbolTable*)
{
  return new VectorLogElem(params->getString("format", "float"));
}

static struct LogElemInfo LogElem_interfaces[] = {
  {"Double", create_double_log_elem},
  {"Float", create_float_log_elem},
  {"Byte", create_byte_log_elem},
  {"Short", create_short_log_elem},
  {"Angle", create_angle_log_elem},
  {"Array", create_array_log_elem},
  {"Vector", create_vector_log_elem},
  {0, 0}
};


// get the generator in case someone wants to generate an instance of LogElem
// or add to the possible configurations
Generator<LogElem>* LogElem::generator()
{
  int i;
  Generator<LogElem>* gen = new Generator<LogElem>();
  for (i=0;LogElem_interfaces[i].tag; i++) {
    gen->registerInterface(LogElem_interfaces[i].tag,
                           LogElem_interfaces[i].create_interface,
                           0L);
  }

  return gen;
}    

LogElem::LogElem()
{
  _address = NULL;
}

LoggingClass::LoggingClass(const char* tag)
{
  _tag = tag;
  _elems = new List<LogElem>;
  _table = new StringDict<LogElem*>;
}

LoggingClass::~LoggingClass()
{
  ListIterator<LogElem> iter(*_elems);
  for (LogElem* current = iter.first(); current; current = iter.next()) 
    delete current;
  delete _elems;
  delete _table;
}

bool LoggingClass::declare(const char* name, const char* format, void* address)
{
  // create a LogElem from name, format, and address
  LogElem* elem = LogElemGenerator->interface(format, NULL);
  if (!elem)
    return false;
  elem->setFormat(format);
  elem->setName(name);
  elem->setAddress(address);

  _elems->append(elem);
  _table->enter(name, elem);

  return true;
}

bool LoggingClass::startup(Output& output)
{
  output.write("D ");
  output.write(_tag.getString());
  ListIterator<LogElem> iter(*_elems);
  for (LogElem* current = iter.first(); current; current = iter.next()) {
    output.write(' ');
    output.write(current->getName());
    output.write('=');
    output.write('"');
    output.write(current->getFormat());
    output.write('"');
  }
  output.write(';');
  
  return true;
}

bool LoggingClass::log(Output& output, Time time)
{
  output.write(_tag.getString());
  output.write('\0');
  long secs, usecs;
  time.getValue(secs, usecs);
  output.write((unsigned char*) &secs, sizeof(long));
  output.write((unsigned char*) &usecs, sizeof(long));
  ListIterator<LogElem> iter(*_elems);
  for (LogElem* current = iter.first(); current; current = iter.next()) {
    current->log(output, current->getAddress());
  }

  return true;
}

BinLogger::BinLogger()
{
  _tags = new StringDict<LoggingClass*>;
  if (!LogElemGenerator)
    LogElemGenerator = LogElem::generator();
  _close_output = false;
}

static void delete_logging_class(const char*, LoggingClass* tagged)
{
  delete tagged;
}

BinLogger::~BinLogger()
{
  _tags->applyToAll(delete_logging_class);
  delete _tags;

  if (_close_output)
    _output.closeFile();
  _close_output = false;
}

// close the logging file if we opened it
void BinLogger::close()
{
  _tags->applyToAll(delete_logging_class);
  delete _tags;
  _tags = new StringDict<LoggingClass*>;

  if (_close_output)
    _output.closeFile();
  _close_output = false;
}

bool BinLogger::declare(const char* tag, const char* name, const char* format,
                        void* address)
{
  LoggingClass* tagged;
  if (!_tags->find(tag, tagged)) {
    tagged = new LoggingClass(tag);
    _tags->enter(tag, tagged);
  }

  return tagged->declare(name, format, address);
}

static void write_class_header(const char*, LoggingClass* tagged, void* data)
{
  tagged->startup(*(Output*) data);
}

bool BinLogger::open(const char* file_name)
{
  if (!_output.openFile(file_name))
    return false;

  _output.write("BinLog 1.0 ");
  _output.write(BYTE_ORDER);
  _output.write(' ');
  _output.write(ALIGN);
  _output.write('\n');

  return true;
}

void BinLogger::commit()
{
  _tags->applyToAll(write_class_header, (void*) &_output);
}

bool BinLogger::log(const char* tag, Time time)
{
  LoggingClass* tagged;
  if (!_tags->find(tag, tagged))
    return false;
  return tagged->log(_output, time);
}

BinLogger* BinLogger::create(ConfigFile* params, SymbolTable* globals)
{
  const char* logger_name = params->getString("loggerName",
                                              "DefaultBinLogger");
  BinLogger* res = (BinLogger*) globals->get(logger_name);
  if (!res) {
    const char* file_name = params->getString("file", "test.log");
    res = new BinLogger();
    if (!res->open(file_name)) {
      delete res;
      res = NULL;
    }
  }
  return res;
}

__UTILS_END_NAMESPACE
