#ifndef IMG_DISPLAY_INTF_H
#define IMG_DISPLAY_INTF_H

#include <utils/Interface.h>
#include <utils/Time.h>
#include <utils/Vector.h>

UTILS_INTERFACE(ImgDisplay)
{
 public:
  enum ColorType {
    Grey8,		/*!< 8-bit greyscale */
    Grey16,		/*!< 16-bit greyscale */
    Grey32,		/*!< 32-bit greyscale */

    RGB1555,		/*!< 15-bit RGB (bit 15 unused) */
    RGB5155,		/*!< 15-bit RGB (bit 9 unused) */
    RGB5515,		/*!< 15-bit RGB (bit 6 unused) */
    RGB5551,		/*!< 15-bit RGB (bit 0 unused) */

    BGR1555,		/*!< 15-bit BGR (bit 15 unused) */
    BGR5155,		/*!< 15-bit BGR (bit 9 unused) */
    BGR5515,		/*!< 15-bit BGR (bit 6 unused) */
    BGR5551,		/*!< 15-bit BGR (bit 0 unused) */

    RGB565,		/*!< 16-bit RGB */
    BGR565,		/*!< 16-bit BGR */

    RGB24,		/*!< 24-bit RGB */
    BGR24,		/*!< 24-bit BGR */

    MSBPadded_RGB32,	/*!< 32-bit RGB (MSB unused) */
    LSBPadded_RGB32, /*!< 32-bit RGB (LSB unused) */

    MSBPadded_BGR32,	/*!< 32-bit BGR (MSB unused) */
    LSBPadded_BGR32, /*!< 32-bit BGR (MSB unused) */
  
    YUV422_Planar,	/*!< Planar YUV422 */
    YUV422_Packed,	/*!< Packed YUV422 */

    LastColorType,
    UnknownColorType
  };
  enum Color {
    Black=0, White, Red, Orange, Yellow, Green, Blue, Purple,
    Pink, LtBlue, LastColor, UnknownColor
  };
  static const unsigned int NoModifiers = 0x0;
  static const unsigned int Shift  =      0x1;
  static const unsigned int Alt  =        0x2;
  static const unsigned int Ctrl =        0x4;
  struct Selection {
    int x, y;
    unsigned int modifiers;  // bit-wise combination of Shift, Alt, Ctrl
  };
  static const bool Centered = true;

  class Window {
   public:
    Window(ImgDisplay* display, int width, int height);
    virtual ~Window() {}

    ImgDisplay* display() const { return _display; }
    
    int getWidth() const { return _width; }
    int getHeight() const { return _height; }
    virtual ColorType colorType() const = 0;
    virtual int bytesPerPixel() const = 0;
    virtual int bytesPerLine() const = 0;

    bool inBounds(int x, int y) const {
      return x >= 0 && y >= 0 && x < getWidth() && y < getHeight();
    }

    // color a pixel
    virtual void pixel(int x, int y, Color colorID) = 0;
    // color a pixel with default color
    void pixel(int x, int y);

    // draw a point of default width
    virtual void point(int x, int y, Color colorID);
    // draw a point with default color
    void point(int x, int y);

    // Fill assuming source is same width and height
    virtual void fill(unsigned char* src) = 0;

    // Fill whole image with color
    virtual void fill(Color color);

    // Clear an image
    virtual void clear();

    // draw a line with the default width. 
    virtual void line(int x1, int y1, int x2, int y2, Color color);
    // draw a line with the default color
    void line(int x1, int y1, int x2, int y2);

    // fill a box
    virtual void fillBox(int x1, int y1, int x2, int y2, Color color);
    // fill a box with the default color
    void fillBox(int x1, int y1, int x2, int y2);

    // draw a box outline with the default width
    virtual void box(int x1, int y1, int x2, int y2, Color color);
    // draw a box with the default color
    void box(int x1, int y1, int x2, int y2);

    // draw a circle with the default width
    virtual void circle(int x, int y, double radius, Color color);
    // draw a circle with the default color
    virtual void circle(int x, int y, double radius);

    // output a string of text at x, y.
    // Centered if centered = ImgDisplay::Centered
    virtual void text(int x, int y, const char* text, Color color,
                      bool centered = false);
    // output a string of text in the default color
    virtual void text(int x, int y, const char* text, bool centered = false);
    void setTextSize(int width, int height) {
      _text_width = width; _text_height = height;
    }
    void getTextSize(int& width, int& height) const {
      width = _text_width; height = _text_height;
    }
    
    // start drawing
    virtual bool begin() = 0;
    // end drawing
    virtual void end() = 0;

    virtual void setSelectable(bool selectable) { _selectable = selectable; }
    bool getSelectable() const { return _selectable; }
    // return the selections in v, and clear them atomically: probably 
    // the typical routine to be used
    int popSelections(utils::Vector<Selection>& v);
    // return the selections in v, leaving them alone
    int getSelections(utils::Vector<Selection>& v);
    // clear the selections:  not getSelections & clearSelections may
    // not be equivalent to popSelections for multithreaded interface 
    // implementations
    void clearSelections();
    // this may want to be protected, but I can imagine obscure cases
    // where it may be useful to be public
    void select(int x, int y, unsigned int modifiers);
    int numSelections() const;
    virtual bool waitForSelection(int& x, int& y, unsigned int& modifiers) {
      clearSelections(); return false;
    }

    // used to store time tag with an image
    virtual void setTimeTag(utils::Time) {}

    //set attributes for a window.
    virtual void setDefaultColor(Color color) { _default_color = color; }
    virtual void setLineWidth(unsigned int width) { _point_width = width; }
    unsigned int getLineWidth() const { return _point_width; }
    Color getDefaultColor() const { return _default_color; }

   protected:
    // just in case something funky is happening with threaded selections
    virtual void acquireSelectionLock() {};
    virtual void releaseSelectionLock() {};

   protected:
    unsigned int _point_width;
    Color _default_color;
    ImgDisplay* _display;
    bool _selectable;
    utils::Vector<Selection> _selections;
    int _text_width, _text_height;
    int _width, _height;
  };

  virtual Window* createWindow(const char* title, int width, int height,
                               ColorType src_type) = 0;
  Window* createWindow(const char* title, int width, int height,
                       const char* src_fmt) { 
    return createWindow(title, width, height, stringToColorType(src_fmt));
  }
  virtual void destroyWindow(Window*) = 0;
  virtual void destroyAllWindows() = 0;

  virtual ColorType defaultWindowColorType() const {
    return UnknownColorType;
  }

  static const char* colorTypeToString(ColorType);
  static ColorType stringToColorType(const char* fmt);

  UTILS_INTF_DECL(ImgDisplay);
};

#endif
