#include <utils/ConfigFile.h>
#include <utils/Player.h>

#include <TimeSource/TimeSource.h>

TimeSource::PlayerManager::PlayerManager()
{
  _player = NULL;
}

TimeSource::PlayerManager::~PlayerManager()
{
  delete _player;
}

bool TimeSource::PlayerManager::open(const char* default_name,
                                     utils::ConfigFile& params,
                                     utils::SymbolTable* globals)
{
  utils::Player* player = new utils::Player();
  if (!player->open(params.getString("name", default_name), false)) {
    delete player;
    return false;
  }
  _player = player;

  bool drive_time = params.getBool("drive_time", false);
  bool set_time = params.getBool("set_time", false);
  if (drive_time || set_time) {
    _time_source =
      TimeSource::interface(params.getString("time_source_spec", "driven"),
                            globals);
    TimeSource::setDefaultTimeSource(_time_source);
    if (_time_source) {
      _key = _time_source->getKey();
       _time_source->setExtent(player->getStart(), player->getEnd());
      float time_offset = params.getFloat("offset", 0.0);
      utils::Time t = player->getStart();
      if (time_offset) {
        t += utils::Time(time_offset);
        player->get(t);
      } 
      if (set_time) {
        _time_source->setTime(t, _key);
      }
      if (!drive_time)
        _time_source = NULL;
    }
  } else
    _time_source = NULL;

  if (!drive_time)
    _external_time = params.getBool("external_time", false);
  else
    _external_time = false;

  _real_time = params.getBool("real_time", false);

  return true;
}
  
bool TimeSource::PlayerManager::next(utils::Time& time)
{
  if (!_player)
    return false;

  if (_external_time) {
    time = TimeSource::now();
    if (!_player->get(time))
      return false;
  } else {
    if (_time_source) {
      utils::Time now;
      _time_source->getTime(now);
      if (!_last_time.isZero() && now != _last_time) {
	if (!_player->get(now))
	  return false;
      }
    }
      
    if (_player->eof() ||!_player->process(time))
      return false;
    _player->next();
  }

  if (_time_source) {
    _time_source->setTime(time, _key);
  }

  if (_real_time && !_last_time.isZero()) {
    utils::Time interval = time-_last_time;
    utils::Time::sleep(interval.getValue());
  }
  
  _last_time = time;

  return true;
}

bool TimeSource::PlayerManager::poll()
{
  if (_external_time) {
    utils::Time time = TimeSource::now();
    utils::Time play_time = time;
    if (!_player->get(time))
      return false;
    return play_time >= time && time != _last_time;
  } else {
    return true;
  }
}
